﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Api.h>
#include "resolver_Cancel.h"
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <algorithm>
#include <functional>
#include "resolver_ClientServerShared.h"

//#define TRACE_CANCEL_DUMP // debug logging for table
//#define LOG_LEVEL LOG_LEVEL_MAX

#define LOG_MODULE_NAME "autoContext" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver {

// Resolver Cancel Table

// ctor, dtor, etc.

ResolverCancelTable::ResolverCancelTable()
{
    for (unsigned idx = 0; idx<TABLE_MAX; ++idx)
    {
        m_HandleSlots[idx].m_ExpirationTimeInSeconds = 0;
        m_HandleSlots[idx].m_SlotState = SlotState_Pending;
        m_HandleSlots[idx].m_HandleID = 0;
        m_HandleSlots[idx].m_ProcessID = 0;
    };
};

ResolverCancelTable::~ResolverCancelTable()
{
    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        m_HandleSlots[idx].m_ExpirationTimeInSeconds = 0;
        m_HandleSlots[idx].m_SlotState = SlotState_Pending;
        m_HandleSlots[idx].m_ProcessID = 0;
        m_HandleSlots[idx].m_HandleID = 0;
    };
};

ResolverCancelTable::ResolverCancelTable(const ResolverCancelTable& o)
{
    NN_SDK_ASSERT(false);
};

bool ResolverCancelTable::operator==(const ResolverCancelTable& o)
{
    NN_SDK_ASSERT(false);
    if (this == &o)
        return true;
    return false;
};

ResolverCancelTable& ResolverCancelTable::GetCreateSharedInstance()
{
    static std::mutex s_InstanceLock;
    std::lock_guard<std::mutex> lock(s_InstanceLock);
    static ResolverCancelTable s_Instance;
    return s_Instance;
};

int ResolverCancelTable::AcquireSlot(int& handleID, const Bit64 processID,
                                     OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    ResolverCancelTable& instance = GetCreateSharedInstance();
    std::lock_guard<std::mutex> lock(instance.m_AccessLock);

    instance.DumpTableUnlocked("before AcquireSlot");
    instance.EnforceExpirationTimesUnlocked();

    int rc = -1;
    nn::socket::SetLastError(nn::socket::Errno::EInval);
    operationStateOut = OperationState_AcquireSlotGenericError;

    int oldestPending = -1;

    if (0 == handleID && 0 != CreateRandomId(handleID))
    {
        // rc & errno already set
        LogDebug("CreateRandomId failed, error: %d\n", __LINE__);
        operationStateOut = OperationState_AcquireSlotUnableToCreateHandleID;
        goto bail;
    };

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot & slot = instance.m_HandleSlots[idx];

        // already exists
        if (handleID == slot.m_HandleID && processID == slot.m_ProcessID)
        {
            if (SlotState_Pending != slot.m_SlotState)
            {
                operationStateOut = OperationState_AcquireSlotAlreadyExists;
                rc = -1;
                nn::socket::SetLastError(nn::socket::Errno::EAlready);
                goto bail;
            }
            oldestPending = idx;
            break;
        };

        if (SlotState_Pending == slot.m_SlotState)
        {
            if (-1 == oldestPending)
            {
                oldestPending = idx;
            }
            else if (slot.m_ExpirationTimeInSeconds <
                     instance.m_HandleSlots[oldestPending].m_ExpirationTimeInSeconds)
            {
                oldestPending = idx;
            };
        };
    };

    if (-1 == oldestPending)
    {
        //table full
        operationStateOut = OperationState_AcquireSlotTableFull;
        nn::socket::SetLastError(nn::socket::Errno::EAgain);
    }
    else
    {
        Slot& slot = instance.m_HandleSlots[oldestPending];
        operationStateOut = OperationState_AcquireSlotSuccess;
        slot.m_ExpirationTimeInSeconds = (ConvertToTimeSpan(nn::os::GetSystemTick()) +
                                          nn::TimeSpan::FromSeconds(MaxAllowedSeconds)).GetSeconds();
        slot.m_SlotState = SlotState_Acquired;
        slot.m_ProcessID = processID;
        slot.m_HandleID = handleID;
        rc = 0;
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);
    };

bail:
    instance.DumpTableUnlocked("after AcquireSlot");
    return rc;
};

int ResolverCancelTable::SetSlotRunning(const int handleID,
                                        const Bit64 processID,
                                        OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    ResolverCancelTable& instance = GetCreateSharedInstance();
    std::lock_guard<std::mutex> lock(instance.m_AccessLock);

    instance.DumpTableUnlocked("before SetSlotRunning");
    instance.EnforceExpirationTimesUnlocked();

    int rc = -1;
    nn::socket::SetLastError(nn::socket::Errno::EInval);
    operationStateOut = OperationState_SetSlotRunningGenericError;

    if (0 == handleID)
    {
        // rc & errno already set
        LogDebug("handleID == 0\n");
        operationStateOut = OperationState_SetSlotRunningZeroHandle;
        goto bail;
    };

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot& slot = instance.m_HandleSlots[idx];

        if (handleID == slot.m_HandleID && processID == slot.m_ProcessID)
        {
            switch (slot.m_SlotState)
            {
            // you cannot jump from pending to running
            // but Pending-> Acquired ->Running
            case SlotState_Pending:
                operationStateOut = OperationState_SetSlotRunningPending;
                nn::socket::SetLastError(nn::socket::Errno::EInval);
                goto bail;

            case SlotState_Acquired:
                operationStateOut = OperationState_SetSlotRunningSuccess;
                slot.m_SlotState = SlotState_Running;
                rc = 0;
                nn::socket::SetLastError(nn::socket::Errno::ESuccess);
                goto bail;

            // already set to running if state > Acquired
            default:
                operationStateOut = OperationState_SetSlotRunningDefault;
                nn::socket::SetLastError(nn::socket::Errno::EAlready);
                goto bail;
            };
        };
    };

    // not found
    operationStateOut = OperationState_SetSlotRunningNotFound;
    nn::socket::SetLastError(nn::socket::Errno::EInval);

bail:
    instance.DumpTableUnlocked("after SetSlotRunning");
    return rc;
};

bool ResolverCancelTable::IsSlotCancelledOrExpired(const int handleID,
                                                   const Bit64 processID,
                                                   OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    ResolverCancelTable& instance = GetCreateSharedInstance();
    std::lock_guard<std::mutex> lock(instance.m_AccessLock);

    instance.DumpTableUnlocked("before IsSlotCancelled");
    instance.EnforceExpirationTimesUnlocked();

    bool rc = false;
    nn::socket::SetLastError(nn::socket::Errno::EInval);
    operationStateOut = OperationState_IsSlotCancelledOrExpiredGenericError;

    if (0 == handleID)
    {
        // rc & errno already set
        LogDebug("handleID == 0\n");
        operationStateOut = OperationState_IsSlotCancelledOrExpiredZeroHandle;
        goto bail;
    };

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot& slot = instance.m_HandleSlots[idx];

        if (handleID == slot.m_HandleID && processID == slot.m_ProcessID)
        {
            switch (slot.m_SlotState)
            {
                // if pending and matches then set ealready
            case SlotState_Pending:
                operationStateOut = OperationState_IsSlotCancelledOrExpiredPending;
                rc = true;
                nn::socket::SetLastError(nn::socket::Errno::EAlready);
                goto bail;

                // no, not cancelled
            case SlotState_Acquired:
            case SlotState_Running:
                operationStateOut = (slot.m_SlotState == SlotState_Acquired ?
                                     OperationState_IsSlotCancelledOrExpiredAcquired :
                                     OperationState_IsSlotCancelledOrExpiredRunning);
                rc = false;
                nn::socket::SetLastError(nn::socket::Errno::ESuccess);
                goto bail;

                // yes, cancelled
            default:
                operationStateOut = (slot.m_SlotState == SlotState_Cancelled ?
                                     OperationState_IsSlotCancelledOrExpiredCancelled :
                                     OperationState_IsSlotCancelledOrExpiredExpired);
                rc = true;
                nn::socket::SetLastError(nn::socket::Errno::ESuccess);
                goto bail;
            };
        };
    };

    // not found
    operationStateOut = OperationState_IsSlotCancelledOrExpiredNotFound;

bail:
    instance.DumpTableUnlocked("after IsSlotCancelled");

    LogDebug("HandleID: %08x, ProcessID: %08x, Status: %s (%d)\n",
             handleID, processID,
             GetOperationStateString(operationStateOut, false),
             operationStateOut);

    return rc;
};

int ResolverCancelTable::SetSlotCancelledOrExpired(const int handleID,
                                                   const Bit64 processID,
                                                   SlotState stateIn,
                                                   OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    ResolverCancelTable& instance = GetCreateSharedInstance();
    std::lock_guard<std::mutex> lock(instance.m_AccessLock);

    instance.DumpTableUnlocked("before SetSlotCancelledOrExpired");
    instance.EnforceExpirationTimesUnlocked();

    int rc = -1;
    nn::socket::SetLastError(nn::socket::Errno::EInval);
    operationStateOut = OperationState_SetSlotCancelledOrExpiredGenericError;

    if (!(SlotState_Cancelled == stateIn || SlotState_Expired == stateIn))
    {
        // rc & errno already set
        LogDebug("Attempt to with state: %s (%d)\n",
                 instance.GetSlotStateString(stateIn, false),
                 stateIn);
        goto bail;
    }
    else if (0 == handleID)
    {
        // rc & errno already set
        LogDebug("handleID == 0\n");
        operationStateOut = OperationState_SetSlotCancelledOrExpiredZeroHandle;
        goto bail;
    };

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot& slot = instance.m_HandleSlots[idx];

        if (handleID == slot.m_HandleID && processID == slot.m_ProcessID)
        {
            switch (slot.m_SlotState)
            {
            case SlotState_Pending:
                // rc & errno already set
                operationStateOut = OperationState_SetSlotCancelledOrExpiredPending;
                nn::socket::SetLastError(nn::socket::Errno::EAlready);
                goto bail;

            case SlotState_Acquired:
            case SlotState_Running:
                rc = 0;
                nn::socket::SetLastError(nn::socket::Errno::ESuccess);
                operationStateOut = (slot.m_SlotState == SlotState_Acquired ?
                                     OperationState_SetSlotCancelledOrExpiredAcquired :
                                     OperationState_SetSlotCancelledOrExpiredRunning);
                slot.m_SlotState = stateIn;
                goto bail;
                // yes, cancelled

            default:
                operationStateOut = (slot.m_SlotState == SlotState_Cancelled ?
                                     OperationState_SetSlotCancelledOrExpiredCancelled :
                                     OperationState_SetSlotCancelledOrExpiredExpired);
                nn::socket::SetLastError(nn::socket::Errno::EAlready);
                goto bail;
            };
        };
    };

    // not found
    operationStateOut = OperationState_SetSlotCancelledOrExpiredNotFound;

bail:
    instance.DumpTableUnlocked("after SetSlotCancelledOrExpired");
    return rc;
};


int ResolverCancelTable::SetSlotCancelled(const int handleID,
                                          const Bit64 processID,
                                          OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    return SetSlotCancelledOrExpired(handleID,
                                     processID,
                                     SlotState_Cancelled,
                                     operationStateOut);
}

int ResolverCancelTable::SetSlotExpired(const int handleID, const
                                        Bit64 processID,
                                        OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    return SetSlotCancelledOrExpired(handleID,
                                     processID,
                                     SlotState_Expired,
                                     operationStateOut);
}

int ResolverCancelTable::SetSlotPending(const int handleID,
                                        const Bit64 processID,
                                        OperationState& operationStateOut)
{
    LogDebug("HandleID: %08x, ProcessID: %08x\n", handleID, processID);

    ResolverCancelTable& instance = GetCreateSharedInstance();
    std::lock_guard<std::mutex> lock(instance.m_AccessLock);

    instance.DumpTableUnlocked("before SetSlotPending");
    instance.EnforceExpirationTimesUnlocked();

    int rc = -1;
    nn::socket::SetLastError(nn::socket::Errno::EInval);
    operationStateOut = OperationState_SetSlotPendingGenericError;

    if (0 == handleID)
    {
        // rc & errno already set
        LogDebug("handleID == 0\n");
        operationStateOut = OperationState_SetSlotPendingZeroHandle;
        goto bail;
    };

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot& slot = instance.m_HandleSlots[idx];
        if (handleID == slot.m_HandleID && processID == slot.m_ProcessID)
        {
            if (slot.m_SlotState == SlotState_Pending)
            {
                rc = -1;
                operationStateOut = OperationState_SetSlotPendingPending;
                nn::socket::SetLastError(nn::socket::Errno::EAlready);
                goto bail;
            };

            if (slot.m_SlotState == SlotState_Acquired)
            {
                operationStateOut = OperationState_SetSlotPendingAcquired;
            }
            else if (slot.m_SlotState == SlotState_Running)
            {
                operationStateOut = OperationState_SetSlotPendingPending;
            }
            else if (slot.m_SlotState == SlotState_Cancelled)
            {
                operationStateOut = OperationState_SetSlotPendingCancelled;
            }
            else if (slot.m_SlotState == SlotState_Expired)
            {
                operationStateOut = OperationState_SetSlotPendingExpired;
            };

            rc = 0;
            slot.m_SlotState = SlotState_Pending;
            nn::socket::SetLastError(nn::socket::Errno::ESuccess);
            goto bail;
        };
    };

    // not found
    operationStateOut = OperationState_SetSlotPendingNotFound;

bail:
    instance.DumpTableUnlocked("after SetSlotPending");
    return rc;
};

void ResolverCancelTable::CancelAllSlotsForProcess(const Bit64 processID,
                                                   OperationState& operationStateOut)
{
    LogDebug("ProcessID: %08x\n", processID);

    ResolverCancelTable& instance = GetCreateSharedInstance();
    std::lock_guard<std::mutex> lock(instance.m_AccessLock);

    instance.DumpTableUnlocked("before CancelAllSlotsForProcess");
    instance.EnforceExpirationTimesUnlocked();

    operationStateOut = OperationState_CancelAllSlotsForProcessIDGenericError;
    unsigned numberOfSlotsCancelled = 0;

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot& slot = instance.m_HandleSlots[idx];

        if (processID == slot.m_ProcessID)
        {
            switch (slot.m_SlotState)
            {
            case SlotState_Pending:
            case SlotState_Cancelled:
            case SlotState_Expired:
                // nothing to do, next
                continue;
                // note: The autoContext ensures that a slot is only in
                // SlotState_Acquired or SlotState_Running when a resolver
                // thread is running; when the server thread returns from
                // the HIPC method in Resolver_Implementation.cpp it is Reset(),
                // returns to pending, and the slot is available for the next request
            case SlotState_Acquired:
            case SlotState_Running:
                ++numberOfSlotsCancelled;
                slot.m_SlotState = SlotState_Cancelled;
                continue;

                // default case handling required
            default:
                LogDebug("default\n");
                goto bail;
            };
        };
    };

    operationStateOut = (numberOfSlotsCancelled == 0 ?
                         OperationState_CancelAllSlotsForProcessIDNoSlotsCancelled :
                         OperationState_CancelAllSlotsForProcessIDSlotsCancelled);

bail:
    instance.DumpTableUnlocked("after CancelAllSlotsForProcess");
    return;
};

const char* ResolverCancelTable::GetSlotStateString(const SlotState& slotStateIn,
                                                    bool includeEnumName) const
{
    const char* returnValue = "ResolverCancelSlotState_Unknown";

    switch (slotStateIn)
    {
    case SlotState_Pending:
        returnValue = "ResolverCancelSlotState_Pending";
        break;
    case SlotState_Acquired:
        returnValue = "ResolverCancelSlotState_Acquired";
        break;
    case SlotState_Running:
        returnValue = "ResolverCancelSlotState_Running";
        break;
    case SlotState_Cancelled:
        returnValue = "ResolverCancelSlotState_Cancelled";
        break;
    case SlotState_Expired:
        returnValue = "ResolverCancelSlotState_Expired";
        break;
    default:
        break;
    };

    if ( false == includeEnumName )
    {
        returnValue = strrchr(returnValue, '_') + 1;
    };

    return returnValue;
}

void ResolverCancelTable::DumpTableUnlocked(const char* line) const
{
#if defined(TRACE_CANCEL_DUMP)
    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        const Slot& slot = m_HandleSlots[idx];
        NN_UNUSED(slot);

        LogDebug("slot[%u]: "
                 "{ m_SlotState: %s (%u), "
                 "m_ExpirationTime: %u, m_HandleID: %08x, "
                 "m_ProcessID: %08x }\n",
                 idx,
                 GetSlotStateString(slot.m_SlotState, false),
                 slot.m_SlotState,
                 slot.m_ExpirationTimeInSeconds, slot.m_HandleID,
                 slot.m_ProcessID);
    };
#endif //TRACE_CANCEL_DUMP
};

void ResolverCancelTable::EnforceExpirationTimesUnlocked()
{
    const int64_t now = ConvertToTimeSpan(nn::os::GetSystemTick()).GetSeconds();

    for (unsigned idx = 0; idx < TABLE_MAX; ++idx)
    {
        Slot& slot = m_HandleSlots[idx];

        if ( now >= slot.m_ExpirationTimeInSeconds )
        {
            switch (slot.m_SlotState)
            {
            case SlotState_Pending:
            case SlotState_Cancelled:
            case SlotState_Expired:
                continue;

            case SlotState_Acquired:
            case SlotState_Running:
                slot.m_SlotState = SlotState_Expired;
                LogDebug("now: %u, so expiring slot[%u]: "
                         "{ m_SlotState: %s (%u), m_ExpirationTime: %u, "
                         "m_HandleID: %08x, m_ProcessID: %08x }\n",
                         now,
                         idx,
                         GetSlotStateString(slot.m_SlotState, false),
                         slot.m_SlotState,
                         slot.m_ExpirationTimeInSeconds,
                         slot.m_HandleID,
                         slot.m_ProcessID);
                // default case handling required, but if hit then an error
                break;
            default:
                LogDebug("default\n");
                goto bail;
            };
        };
    };

bail:
    return;
};

void CancelAll(Bit64 pid) NN_NOEXCEPT
{
    OperationState operationState;

    ResolverCancelTable::CancelAllSlotsForProcess(pid, operationState);

    LogDebug("operation state for CancelAllSlotsForProcess(%u): %s (%d)\n",
             pid,
             GetOperationStateString(operationState, false),
             operationState);
};

}}};// namespace nn::socket::resolver

