﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"

/**
 * serialized version
 *
 * struct ResolverOption
 * {
 *     uint32_t ResolverOptionKey key;
 *     uint32_t ResolverOptionType type;
 *
 *     // if a pointer then the number of bytes
 *     // otherwise the number of bytes of the value
 *     uint32_t size;
 *
 *     union
 *     {
 *          bool        booleanValue;
 *          int         integerValue;
 *          uint32_t    unsigned32Value;
 *          uint64_t    unsigned64Value;  // largest
 *          double      doubleValue;
 *          const char* pointerValue;
 *     }
 *     data;
 * }
 */

//#define LOG_LEVEL LOG_LEVEL_HEX
#define LOG_MODULE_NAME "options serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

#define TO_BUFFER(rc, option, param,  buftype, outtype, pBuffer, size, pCursor, err) \
    {                                                                   \
        buftype tmp = static_cast<buftype>(option.param);               \
        if (-1 == ( rc = ToBuffer(pCursor,                              \
                                  size - (pCursor - pBuffer),           \
                                  tmp)))                                \
        {                                                               \
            LogMajor("ToBuffer failed for param: %s.\n", #param);       \
            goto err;                                                   \
        };                                                              \
        LogHex(pCursor, rc, "%s: ", #param);                            \
        pCursor += rc;                                                  \
    }

#define FROM_BUFFER(rc, option, param, buftype, outtype, pBuffer, size, pCursor, err) \
    {                                                                   \
        buftype tmp;                                                    \
        if (-1 == ( rc = FromBuffer(tmp,                                \
                                    pCursor,                            \
                                    size - (pCursor - pBuffer))))       \
        {                                                               \
            LogMajor("FromBuffer failed for param: %s.\n", #param);       \
            goto err;                                                   \
        };                                                              \
        LogHex(pCursor, rc, "%s: ", #param);                            \
        option.param = static_cast<outtype>(tmp);                       \
        pCursor += rc;                                                  \
    }

namespace nn { namespace socket { namespace resolver { namespace serializer {

template <>
size_t DNSSerializer::SizeOf(const ResolverOption & option)
{
    LogDebug("option: %p\n", &option);
    size_t rc = 0;
    rc += sizeof(uint32_t);       // key
    rc += sizeof(uint32_t);       // type
    rc += sizeof(uint32_t);       // size
    rc += sizeof(option.data);    // union
    if (ResolverOptionType::Pointer == option.type)
    {
        rc += option.size;
    };

    LogDebug("returning: %zu\n", rc);
    return rc;
};

template <>
size_t DNSSerializer::SizeOf(const ResolverOption* pOptions, size_t count)
{
    LogDebug("pOptions: %p, count: %zu\n", pOptions, count);
    size_t rc = 0;

    if (NULL == pOptions)
    {
        goto bail;
    };

    for (unsigned idx=0; idx<count; ++idx)
    {
        rc += SizeOf(pOptions[idx]);
    };

bail:
    LogDebug("returning: %zu\n", rc);
    return rc;
}

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const        pBufferOut,
                                size_t                 bufferSize,
                                const ResolverOption & option)
{
    LogDebug("pBufferOut: %p, bufferSize: %zu, option: %p\n",
             pBufferOut, bufferSize, &option);

    ssize_t rc = -1;
    uint8_t* pBufferCurrent = pBufferOut;
    size_t objectSize = SizeOf(option);

    if ( -1 == CheckToBufferArguments(pBufferCurrent,
                                      bufferSize,
                                      objectSize,
                                      __LINE__))
    {
        LogMajor("CheckToBufferArguments\n");
        rc = -1;
        goto bail;
    };

    TO_BUFFER(rc, option, key,  uint32_t, ResolverOptionKey,  pBufferOut, bufferSize, pBufferCurrent, bail);
    TO_BUFFER(rc, option, type, uint32_t, ResolverOptionType, pBufferOut, bufferSize, pBufferCurrent, bail);
    TO_BUFFER(rc, option, size, uint32_t, size_t,             pBufferOut, bufferSize, pBufferCurrent, bail);

    rc = sizeof(option.data);
    memcpy(pBufferCurrent, &option.data, rc);
    LogHex(pBufferCurrent, rc, "data: ");
    pBufferCurrent += rc;

    if (ResolverOptionType::Pointer == option.type)
    {
        memcpy(pBufferCurrent, option.data.pointerValue, option.size);
        LogHex(pBufferCurrent, option.size, "pointer data: ");
        pBufferCurrent += option.size;
    };

    rc = pBufferCurrent - pBufferOut;

bail:
    LogHex(pBufferOut, rc, "wrote %zd bytes to %p: ", rc, pBufferOut);
    return rc;
};

template <>
ssize_t DNSSerializer::FromBuffer(ResolverOption & optionOut,
                                  const uint8_t  * pBufferIn,
                                  size_t           bufferSize)
{
    LogDebug("optionOut: %p, pBufferIn: %p, bufferSize: %zu\n",
             &optionOut, pBufferIn, bufferSize);

    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;

    FROM_BUFFER(rc, optionOut, key,  uint32_t, ResolverOptionKey,  pBufferIn, bufferSize, pBufferCurrent, bail);
    FROM_BUFFER(rc, optionOut, type, uint32_t, ResolverOptionType, pBufferIn, bufferSize, pBufferCurrent, bail);
    FROM_BUFFER(rc, optionOut, size, uint32_t, size_t,             pBufferIn, bufferSize, pBufferCurrent, bail);

    memcpy(&optionOut.data, pBufferCurrent, sizeof(optionOut.data));
    LogHex(pBufferCurrent, sizeof(optionOut.data), "data: ");
    pBufferCurrent += sizeof(optionOut.data);

    if (ResolverOptionType::Pointer == optionOut.type)
    {
        optionOut.data.pointerValue = reinterpret_cast<const char*>(pBufferCurrent);
        LogHex(pBufferCurrent, optionOut.size, "pointer data: ");
        pBufferCurrent += optionOut.size;
    };

    rc = pBufferCurrent - pBufferIn;

bail:
    LogHex(pBufferIn, rc, "read %zd bytes from %p: ", rc, pBufferIn);
    return rc;
};

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const              pBufferOut,
                                size_t                       bufferSize,
                                const ResolverOption * const pOptionsIn,
                                size_t                       optionsCount)
{
    LogDebug("pBufferOut: %p, bufferSize: %zu, pOptionsIn: %p, optionsCount: %zu\n",
             pBufferOut, bufferSize, pOptionsIn, optionsCount);

    size_t rc = 0;
    auto pBufferCurrent = pBufferOut;

    if (NULL == pOptionsIn)
    {
        goto bail;
    };

    for (unsigned idx=0; idx<optionsCount; ++idx)
    {
        LogDebug("pOption: %p, cursor: %p\n", &pOptionsIn[idx], pBufferCurrent);

        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  bufferSize - (pBufferCurrent - pBufferOut),
                                  pOptionsIn[idx])))
        {
            LogMajor(" * * * * * * * * * * * option #%d\n", idx);
            goto bail;
        };

        LogDebug("pOption: %p, cursor: %p, option%u.size: %zu, written: %zu\n",
                 &pOptionsIn[idx], pBufferCurrent,
                 idx + 1,
                 SizeOf(pOptionsIn[idx]),
                 rc);

        pBufferCurrent += rc;
    };

    rc = pBufferCurrent - pBufferOut;

bail:
    LogDebug("returning: %zd\n", rc);
    return rc;
};

template <>
ssize_t DNSSerializer::FromBuffer(ResolverOption * const pOptionsOut,
                                  size_t                 optionsSize,
                                  const uint8_t * const  pBufferIn,
                                  size_t                 bufferSize,
                                  size_t                 optionsCount)
{
    LogDebug("pOptionsOut: %p, optionsSize: %zu, pBufferIn: %p, bufferSize: %zu, optionsCount: %zu\n",
             pOptionsOut, optionsSize, pBufferIn, bufferSize, optionsCount);

    auto pBufferCurrent = pBufferIn;
    ssize_t rc = 0;

    for (unsigned idx=0; idx<optionsCount; ++idx)
    {
        if ( -1 == (rc = FromBuffer(pOptionsOut[idx],
                                    pBufferCurrent,
                                    bufferSize - (pBufferCurrent - pBufferIn))))
        {
            LogMajor("\n");
            goto bail;
        };

        LogDebug("option#%u.size: %zu, cursor: %p, bufferLeft: %zu\n",
                 idx + 1,
                 rc,
                 pBufferCurrent,
                 bufferSize - (pBufferCurrent - pBufferIn));

        pBufferCurrent += rc;
    };

    rc = pBufferCurrent - pBufferIn;
bail:
    LogDebug("returning: %zd\n", rc);
    return rc;

}

}}}}// namespace nn::socket::resolver::serializer
