﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <map>

#define USE_WINSOCK_FD_SET
#include <nn/socket/private/socket_PlatformTypesTranslation.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <nn/nn_SdkLog.h>

#include "socket_Api.h"
#include "socket_Allocator.h"
#include "serializer.h"
#include "serializer_Specializations.h"

namespace nn     {
namespace socket {
namespace detail {

PosixWinSockConverter   posixWinSockConverter;

int PosixWinSockConverter::WinsockToPosixSocket(SOCKET winsockSocket)
NN_NOEXCEPT
{
    /* Winsock socket handles are always 4 byte aligned,
    * shift handle and use it as an initial index into handle array.
    * If there happens to be a lot of collisions, this may be a bit slow;
    * however, translations from Winsock to Posix should not happen that often.
    */
    int posixSocket;
    int initialIndex = (winsockSocket >> 2) % WinsockHandleTableSize;

    for (posixSocket = initialIndex;
        posixSocket < WinsockHandleTableSize;
        posixSocket++)
    {
        if (g_WinsockHandleTable[posixSocket] == winsockSocket)
        {
            return posixSocket;
        }
    }

    for (posixSocket = 0;
        posixSocket < initialIndex;
        posixSocket++)
    {
        if (g_WinsockHandleTable[posixSocket] == winsockSocket)
        {
            return posixSocket;
        }
    }

    return -1;
}

SOCKET PosixWinSockConverter::PosixToWinsockSocket(int posixSocket)
NN_NOEXCEPT
{
    SOCKET handle = INVALID_SOCKET;

    if (posixSocket >= 0
        && posixSocket < WinsockHandleTableSize
        && INVALID_SOCKET != (handle = g_WinsockHandleTable[posixSocket]))
    {
        return handle;
    };

    nn::socket::detail::SetLastError(Errno::EBadf);
    return handle;
}

void PosixWinSockConverter::ReleasePosixHandle(int posixSocket)
NN_NOEXCEPT
{
    if (posixSocket >= 0 && posixSocket < WinsockHandleTableSize)
    {
        g_WinsockHandleTable[posixSocket] = INVALID_SOCKET;
    }
    else
    {
        NN_SDK_ASSERT(false);
    };
}

int PosixWinSockConverter::AcquirePosixHandle(SOCKET winsockSocket)
NN_NOEXCEPT
{
    int posixSocket;
    int initialIndex = (winsockSocket >> 2) % WinsockHandleTableSize;

    std::lock_guard<std::mutex> lock(g_DescriptorTableLock);

    for (posixSocket = initialIndex;
        posixSocket < WinsockHandleTableSize;
        posixSocket++)
    {
        if (g_WinsockHandleTable[posixSocket] == INVALID_SOCKET)
        {
            g_WinsockHandleTable[posixSocket] = winsockSocket;
            return posixSocket;
        }
    }
    for (posixSocket = 0;
        posixSocket < initialIndex;
        posixSocket++)
    {
        if (g_WinsockHandleTable[posixSocket] == INVALID_SOCKET)
        {
            g_WinsockHandleTable[posixSocket] = winsockSocket;
            return posixSocket;
        }
    }

    /* out of handles */
    nn::socket::detail::SetLastError(Errno::EMFile);
    return -1;
}

void CopyToPlatform(sockaddr_in* pDest, const SockAddrIn* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->sin_family = MapFamilyValue(pSrc->sin_family);
        pDest->sin_port = pSrc->sin_port;
        std::memcpy(&pDest->sin_addr, &pSrc->sin_addr, sizeof(pDest->sin_addr));
        std::memcpy(pDest->sin_zero, pSrc->sin_zero, sizeof(pDest->sin_zero));
    }
}

void CopyFromPlatform(SockAddrIn* pDest, const sockaddr_in* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        pDest->sin_family = MapFamilyValue(static_cast<int8_t>(pSrc->sin_family));
        pDest->sin_port = pSrc->sin_port;
        std::memcpy(&pDest->sin_addr, &pSrc->sin_addr, sizeof(pDest->sin_addr));
        std::memcpy(pDest->sin_zero, pSrc->sin_zero, sizeof(pDest->sin_zero));
    }
}

void CopyToPlatform(in_addr* pDest, const InAddr* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->S_un.S_addr = pSrc->S_addr;
    }
}

void CopyFromPlatform(InAddr* pDest, const in_addr* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        pDest->S_addr = pSrc->S_un.S_addr;
    }
}

bool CopyFromPlatform(AddrInfo** pDest, const addrinfo* pSrc)
NN_NOEXCEPT
{
    bool bRet = true;
    AddrInfo* pNewAddrInfo = nullptr;

    if( nullptr != pDest )
    {
        pNewAddrInfo = reinterpret_cast<nn::socket::AddrInfo*>(nn::socket::detail::Calloc(1, sizeof(*pNewAddrInfo)));

        if( nullptr == pNewAddrInfo )
        {
            bRet = false;
            goto exit;
        }

        pNewAddrInfo->ai_flags = static_cast<AddrInfoFlag>(pSrc->ai_flags);
        pNewAddrInfo->ai_family = static_cast<Family>(pSrc->ai_family);
        pNewAddrInfo->ai_socktype = static_cast<Type>(pSrc->ai_socktype);
        pNewAddrInfo->ai_protocol = static_cast<Protocol>(pSrc->ai_protocol);
        pNewAddrInfo->ai_addrlen = static_cast<SockLenT>(sizeof(*(pNewAddrInfo->ai_addr)));

        if( nullptr != pSrc->ai_addr )
        {
            pNewAddrInfo->ai_addr = reinterpret_cast<nn::socket::SockAddr *>(nn::socket::detail::Calloc(1, sizeof(*pNewAddrInfo->ai_addr)));

            if( nullptr == pNewAddrInfo->ai_addr )
            {
                bRet = false;
                goto exit;
            }

            CopyFromPlatform(reinterpret_cast<SockAddrIn*>(pNewAddrInfo->ai_addr), reinterpret_cast<const sockaddr_in*>(pSrc->ai_addr));
        }

        if( nullptr != pSrc->ai_canonname )
        {
            size_t nameLen = strlen(pSrc->ai_canonname);
            pNewAddrInfo->ai_canonname = reinterpret_cast<char *>(nn::socket::detail::Calloc(1, nameLen + 1));

            if( nullptr == pNewAddrInfo->ai_canonname )
            {
                bRet = false;
                goto exit;
            }

            strncpy(pNewAddrInfo->ai_canonname, pSrc->ai_canonname, nameLen);
        }

        if( nullptr != pSrc->ai_next )
        {
            if( CopyFromPlatform(&pNewAddrInfo->ai_next, pSrc->ai_next) == false )
            {
                bRet = false;
                goto exit;
            }
        }

        *pDest = pNewAddrInfo;
    }

exit:
    if( false == bRet && nullptr != pNewAddrInfo )
    {
        nn::socket::detail::Free(pNewAddrInfo->ai_canonname);
        nn::socket::detail::Free(pNewAddrInfo->ai_addr);
        nn::socket::detail::Free(pNewAddrInfo);
        nn::socket::detail::SetLastError(Errno::ENoMem);
    }

    return bRet;
}

void FreeCopiedAddrInfo(AddrInfo *pAddrInfo)
NN_NOEXCEPT
{
    if( nullptr == pAddrInfo )
    {
        return;
    }

    nn::socket::detail::Free(pAddrInfo->ai_canonname);
    nn::socket::detail::Free(pAddrInfo->ai_addr);

    FreeCopiedAddrInfo(pAddrInfo->ai_next);

    nn::socket::detail::Free(pAddrInfo);
}

void CopyToPlatform(hostent* pDest, const HostEnt* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->h_name = pSrc->h_name;
        pDest->h_aliases = pSrc->h_aliases;
        pDest->h_addrtype = static_cast<short>(pSrc->h_addrtype);
        pDest->h_length = static_cast<short>(pSrc->h_length);
        pDest->h_addr_list = pSrc->h_addr_list;
    }
}

void CopyFromPlatform(HostEnt* pDest, const hostent* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        resolver::serializer::DNSSerializer serializer;
        size_t hostentBufferSize = 512;
        uint8_t* pOutBuffer = nullptr;

        pOutBuffer = static_cast<uint8_t*>(socket::detail::Calloc(1, hostentBufferSize));
        if ( nullptr == pOutBuffer )
        {
            *nn::socket::GetHError() = nn::socket::HErrno::Netdb_Internal;
            socket::SetLastError(nn::socket::Errno::ENoMem);
            return;
        }

        ssize_t actualSize = serializer.ToBuffer(pOutBuffer, hostentBufferSize, *pSrc);
        serializer.FromBuffer(*pDest, pOutBuffer, actualSize);
        socket::detail::Free(pOutBuffer);
    }
}

void CopyToPlatform(timeval* pDest, const TimeVal* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->tv_sec = pSrc->tv_sec;
        pDest->tv_usec = pSrc->tv_usec;
    }
}

void CopyFromPlatform(TimeVal* pDest, const timeval* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        pDest->tv_sec = pSrc->tv_sec;
        pDest->tv_usec = pSrc->tv_usec;
    }
}

void CopyToPlatform(pollfd* pDest, const PollFd* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->fd = pSrc->fd;
        pDest->events = MapPollEventValue(pSrc->events);
        pDest->revents = MapPollEventValue(pSrc->revents);
    }
}

void CopyFromPlatform(PollFd* pDest, const pollfd* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        pDest->fd = static_cast<int>(pSrc->fd);
        pDest->events = MapPollEventValue(pSrc->events);
        pDest->revents = MapPollEventValue(pSrc->revents);
    }
}

void CopyToPlatform(fd_set* pDest, const FdSet* pSrc)
NN_NOEXCEPT
{
    if( nullptr == pSrc )
    {
        return;
    }

    FD_ZERO(pDest);

    for( int posixSocket = 0; posixSocket < FD_SETSIZE; posixSocket++ )
    {
        if( nn::socket::FdSetIsSet(posixSocket, pSrc) )
        {
            FD_SET(posixWinSockConverter.PosixToWinsockSocket(posixSocket), pDest);
        }
    }
}

void CopyFromPlatform(FdSet* pDest, const fd_set* pSrc)
NN_NOEXCEPT
{
    if( nullptr == pDest )
    {
        return;
    }

    nn::socket::FdSetZero(pDest);

    for( int posixSocket = 0; posixSocket < FD_SETSIZE; posixSocket++ )
    {
        SOCKET s = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
        if( FD_ISSET(s, pSrc) )
        {
            nn::socket::FdSetSet(posixSocket, pDest);
        }
    }
}

void CopyToPlatform(linger* pDest, const Linger* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pSrc )
    {
        pDest->l_onoff = static_cast<u_short>(pSrc->l_onoff);
        pDest->l_linger = static_cast<u_short>(pSrc->l_linger);
    }
}

void CopyFromPlatform(Linger* pDest, const linger* pSrc)
NN_NOEXCEPT
{
    if( nullptr != pDest )
    {
        pDest->l_onoff = pSrc->l_onoff;
        pDest->l_linger = pSrc->l_linger;
    }
}

int32_t MapProtocolValue(Protocol protocol)
{
    int32_t mappedProtocol = -1;

    switch( protocol )
    {
    case Protocol::IpProto_Ip:      mappedProtocol = IPPROTO_IP;    break;
    case Protocol::IpProto_Icmp:    mappedProtocol = IPPROTO_ICMP;  break;
    case Protocol::IpProto_Tcp:     mappedProtocol = IPPROTO_TCP;   break;
    case Protocol::IpProto_Udp:     mappedProtocol = IPPROTO_UDP;   break;
    case Protocol::IpProto_None:    mappedProtocol = IPPROTO_NONE;  break;
    case Protocol::IpProto_UdpLite: mappedProtocol = IPPROTO_UDP;   break;
    case Protocol::IpProto_Raw:     mappedProtocol = IPPROTO_RAW;   break;
    case Protocol::IpProto_Max:     mappedProtocol = IPPROTO_MAX;   break;
    default: NN_SDK_LOG("WARNING: Invalid nn::socket::Protocol %d.\n", protocol);   break;
    }

    if( mappedProtocol == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::Protocol %d is not supported by Win32/Win64.\n", protocol);
    }

    return mappedProtocol;
}

Protocol MapProtocolValue(int32_t protocol)
{
    Protocol mappedProtocol = Protocol::IpProto_None;

    switch( protocol )
    {
    case IPPROTO_IP:    mappedProtocol = Protocol::IpProto_Ip;      break;
    case IPPROTO_ICMP:  mappedProtocol = Protocol::IpProto_Icmp;    break;
    case IPPROTO_TCP:   mappedProtocol = Protocol::IpProto_Tcp;     break;
    case IPPROTO_UDP:   mappedProtocol = Protocol::IpProto_Udp;     break;
    case IPPROTO_NONE:  mappedProtocol = Protocol::IpProto_None;    break;
    case IPPROTO_RAW:   mappedProtocol = Protocol::IpProto_Raw;     break;
    case IPPROTO_MAX:   mappedProtocol = Protocol::IpProto_Max;     break;
    default: NN_SDK_LOG("WARNING: Invalid socket protocol %d.\n", protocol);   break;
    }

    return mappedProtocol;
}

int32_t MapTypeValue(Type type)
{
    int32_t mappedType = -1;

    switch( type )
    {
    case Type::Sock_Default:    mappedType = 0;                 break;
    case Type::Sock_Stream:     mappedType = SOCK_STREAM;       break;
    case Type::Sock_Dgram:      mappedType = SOCK_DGRAM;        break;
    case Type::Sock_Raw:        mappedType = SOCK_RAW;          break;
    case Type::Sock_SeqPacket:  mappedType = SOCK_SEQPACKET;    break;
    case Type::Sock_NonBlock:   mappedType = -1;                break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::Type %d.\n", type);    break;
    }

    if( mappedType == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::Type %d is not supported by Win32/Win64.\n", type);
    }

    return mappedType;
}

Type MapTypeValue(int32_t type)
{
    Type mappedType = Type::Sock_Default;

    switch( type )
    {
    case 0:                 mappedType = Type::Sock_Default;      break;
    case SOCK_STREAM:       mappedType = Type::Sock_Stream;       break;
    case SOCK_DGRAM:        mappedType = Type::Sock_Dgram;        break;
    case SOCK_RAW:          mappedType = Type::Sock_Raw;          break;
    case SOCK_SEQPACKET:    mappedType = Type::Sock_SeqPacket;    break;
    default:    NN_SDK_LOG("WARNING: Invalid socket type %d.\n", type); break;
    }

    return mappedType;
}

int8_t MapFamilyValue(Family family)
{
    int8_t mappedFamily = -1;

    switch( family )
    {
    case Family::Af_Unspec: mappedFamily = AF_UNSPEC;   break;
    case Family::Af_Inet:   mappedFamily = AF_INET;     break;
    case Family::Af_Route:  mappedFamily = -1;          break;
    case Family::Af_Link:   mappedFamily = AF_LINK;     break;
    case Family::Af_Inet6:  mappedFamily = AF_INET6;    break;
    case Family::Af_Max:    mappedFamily = AF_MAX;      break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::Family %d.\n", family);    break;
    }

    if( mappedFamily == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::Family %d is not supported by Win32/Win64.\n", family);
    }

    return mappedFamily;
}

Family MapFamilyValue(int8_t family)
{
    Family mappedFamily = Family::Af_Unspec;

    switch( family )
    {
    case AF_UNSPEC: mappedFamily = Family::Af_Unspec;   break;
    case AF_INET:   mappedFamily = Family::Af_Inet;     break;
    case AF_LINK:   mappedFamily = Family::Af_Link;     break;
    case AF_INET6:  mappedFamily = Family::Af_Inet6;    break;
    case AF_MAX:    mappedFamily = Family::Af_Max;      break;
    default:    NN_SDK_LOG("WARNING: Invalid socket family %d.\n", family); break;
    }

    return mappedFamily;
}

int32_t MapMsgFlagValue(MsgFlag flag)
{
    int32_t mappedMsgFlag = 0;

    if( (flag & MsgFlag::Msg_Oob) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_OOB;
    }
    if( (flag & MsgFlag::Msg_Peek) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_PEEK;
    }
    if( (flag & MsgFlag::Msg_DontRoute) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_DONTROUTE;
    }
    if( (flag & MsgFlag::Msg_Trunc) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_TRUNC;
    }
    if( (flag & MsgFlag::Msg_CTrunc) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_CTRUNC;
    }
    if( (flag & MsgFlag::Msg_WaitAll) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_WAITALL;
    }
    if( (flag & MsgFlag::Msg_DontWait) != MsgFlag::Msg_None )
    {
        mappedMsgFlag |= MSG_DONTWAIT;
    }

    return mappedMsgFlag;
}

MsgFlag MapMsgFlagValue(int32_t flag)
{
    MsgFlag mappedMsgFlag = MsgFlag::Msg_None;

    if( flag & MSG_OOB )
    {
        mappedMsgFlag |= MsgFlag::Msg_Oob;
    }
    if( flag & MSG_PEEK )
    {
        mappedMsgFlag |= MsgFlag::Msg_Peek;
    }
    if( flag & MSG_DONTROUTE )
    {
        mappedMsgFlag |= MsgFlag::Msg_DontRoute;
    }
    if( flag & MSG_TRUNC )
    {
        mappedMsgFlag |= MsgFlag::Msg_Trunc;
    }
    if( flag & MSG_CTRUNC )
    {
        mappedMsgFlag |= MsgFlag::Msg_CTrunc;
    }
    if( flag & MSG_WAITALL )
    {
        mappedMsgFlag |= MsgFlag::Msg_WaitAll;
    }
    if( flag & MSG_DONTWAIT )
    {
        mappedMsgFlag |= MsgFlag::Msg_DontWait;
    }

    return mappedMsgFlag;
}

int16_t MapPollEventValue(PollEvent poll)
{
    int16_t mappedPollEvent = 0;

    if( (poll & PollEvent::PollIn) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLIN;
    }
    if( (poll & PollEvent::PollPri) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLPRI;
    }
    if( (poll & PollEvent::PollOut) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLOUT;
    }
    if( (poll & PollEvent::PollErr) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLERR;
    }
    if( (poll & PollEvent::PollHup) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLHUP;
    }
    if( (poll & PollEvent::PollNVal) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLNVAL;
    }
    if( (poll & PollEvent::PollRdNorm) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLRDNORM;
    }
    if( (poll & PollEvent::PollRdBand) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLRDBAND;
    }
    if( (poll & PollEvent::PollWrNorm) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLWRNORM;
    }
    if( (poll & PollEvent::PollWrBand) != PollEvent::PollNone )
    {
        mappedPollEvent |= POLLWRBAND;
    }

    return mappedPollEvent;
}

PollEvent MapPollEventValue(int16_t poll)
{
    PollEvent mappedPollEvent = PollEvent::PollNone;

    if( (poll & POLLIN) == POLLIN )
    {
        mappedPollEvent |= PollEvent::PollIn;
    }
    if( (poll & POLLPRI) == POLLPRI )
    {
        mappedPollEvent |= PollEvent::PollPri;
    }
    if( (poll & POLLOUT) == POLLOUT )
    {
        mappedPollEvent |= PollEvent::PollOut;
    }
    if( (poll & POLLERR) == POLLERR )
    {
        mappedPollEvent |= PollEvent::PollErr;
    }
    if( (poll & POLLHUP) == POLLHUP )
    {
        mappedPollEvent |= PollEvent::PollHup;
    }
    if( (poll & POLLNVAL) == POLLNVAL )
    {
        mappedPollEvent |= PollEvent::PollNVal;
    }
    if( (poll & POLLRDNORM) == POLLRDNORM )
    {
        mappedPollEvent |= PollEvent::PollRdNorm;
    }
    if( (poll & POLLRDBAND) == POLLRDBAND )
    {
        mappedPollEvent |= PollEvent::PollRdBand;
    }
    if( (poll & POLLWRNORM) == POLLWRNORM )
    {
        mappedPollEvent |= PollEvent::PollWrNorm;
    }
    if( (poll & POLLWRBAND) == POLLWRBAND )
    {
        mappedPollEvent |= PollEvent::PollWrBand;
    }

    return mappedPollEvent;
}

uint32_t MapAddrInfoFlagValue(AddrInfoFlag aiFlag)
{
    uint32_t mappedAddrInfoFlag = 0;

    if( (aiFlag & AddrInfoFlag::Ai_Passive) != AddrInfoFlag::Ai_None )
    {
        mappedAddrInfoFlag |= AI_PASSIVE;
    }
    if( (aiFlag & AddrInfoFlag::Ai_CanonName) != AddrInfoFlag::Ai_None )
    {
        mappedAddrInfoFlag |= AI_CANONNAME;
    }
    if( (aiFlag & AddrInfoFlag::Ai_NumericHost) != AddrInfoFlag::Ai_None )
    {
        mappedAddrInfoFlag |= AI_NUMERICHOST;
    }
    if( (aiFlag & AddrInfoFlag::Ai_NumericServ) != AddrInfoFlag::Ai_None )
    {
        mappedAddrInfoFlag |= AI_NUMERICSERV;
    }
    if( (aiFlag & AddrInfoFlag::Ai_AddrConfig) != AddrInfoFlag::Ai_None )
    {
        mappedAddrInfoFlag |= AI_ADDRCONFIG;
    }

    return mappedAddrInfoFlag;
}

AddrInfoFlag MapAddrInfoFlagValue(uint32_t aiFlag)
{
    AddrInfoFlag mappedAddrInfoFlag = AddrInfoFlag::Ai_None;

    if( aiFlag & AI_PASSIVE )
    {
        mappedAddrInfoFlag |= AddrInfoFlag::Ai_Passive;
    }
    if( aiFlag & AI_CANONNAME )
    {
        mappedAddrInfoFlag |= AddrInfoFlag::Ai_CanonName;
    }
    if( aiFlag & AI_NUMERICHOST )
    {
        mappedAddrInfoFlag |= AddrInfoFlag::Ai_NumericHost;
    }
    if( aiFlag & AI_NUMERICSERV )
    {
        mappedAddrInfoFlag |= AddrInfoFlag::Ai_NumericServ;
    }
    if( aiFlag & AI_ADDRCONFIG )
    {
        mappedAddrInfoFlag |= AddrInfoFlag::Ai_AddrConfig;
    }

    return mappedAddrInfoFlag;
}

uint32_t MapNameInfoFlagValue(NameInfoFlag niFlag)
{
    uint32_t mappedNameInfoFlag = 0;

    if( (niFlag & NameInfoFlag::Ni_NoFqdn) != NameInfoFlag::Ni_None )
    {
        mappedNameInfoFlag |= NI_NOFQDN;
    }
    if( (niFlag & NameInfoFlag::Ni_NumericHost) != NameInfoFlag::Ni_None )
    {
        mappedNameInfoFlag |= NI_NUMERICHOST;
    }
    if( (niFlag & NameInfoFlag::Ni_NameReqd) != NameInfoFlag::Ni_None )
    {
        mappedNameInfoFlag |= NI_NAMEREQD;
    }
    if( (niFlag & NameInfoFlag::Ni_NumericServ) != NameInfoFlag::Ni_None )
    {
        mappedNameInfoFlag |= NI_NUMERICSERV;
    }
    if( (niFlag & NameInfoFlag::Ni_Dgram) != NameInfoFlag::Ni_None )
    {
        mappedNameInfoFlag |= NI_DGRAM;
    }

    return mappedNameInfoFlag;
}

NameInfoFlag MapNameInfoFlagValue(uint32_t niFlag)
{
    NameInfoFlag mappedNameInfoFlag = NameInfoFlag::Ni_None;

    if( niFlag & NI_NOFQDN )
    {
        mappedNameInfoFlag |= NameInfoFlag::Ni_NoFqdn;
    }
    if( niFlag & NI_NUMERICHOST )
    {
        mappedNameInfoFlag |= NameInfoFlag::Ni_NumericHost;
    }
    if( niFlag & NI_NAMEREQD )
    {
        mappedNameInfoFlag |= NameInfoFlag::Ni_NameReqd;
    }
    if( niFlag & NI_NUMERICSERV )
    {
        mappedNameInfoFlag |= NameInfoFlag::Ni_NumericServ;
    }
    if( niFlag & NI_DGRAM )
    {
        mappedNameInfoFlag |= NameInfoFlag::Ni_Dgram;
    }

    return mappedNameInfoFlag;
}

uint32_t MapShutdownMethodValue(ShutdownMethod how)
{
    // we want Shutdown() to fail if a valid method is not passed in
    int32_t mappedShutdownMethod = -1;

    switch( how )
    {
    case ShutdownMethod::Shut_Rd:   mappedShutdownMethod = SHUT_RD;     break;
    case ShutdownMethod::Shut_Wr:   mappedShutdownMethod = SHUT_WR;     break;
    case ShutdownMethod::Shut_RdWr: mappedShutdownMethod = SHUT_RDWR;   break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::ShutdownMethod %d.\n", how);   break;
    }

    return mappedShutdownMethod;
}

ShutdownMethod MapShutdownMethodValue(uint32_t how)
{
    // we want Shutdown() to fail if a valid method is not passed in
    ShutdownMethod mappedShutdownMethod = static_cast<ShutdownMethod>(-1);

    switch( how )
    {
    case SHUT_RD:   mappedShutdownMethod = ShutdownMethod::Shut_Rd;     break;
    case SHUT_WR:   mappedShutdownMethod = ShutdownMethod::Shut_Wr;     break;
    case SHUT_RDWR: mappedShutdownMethod = ShutdownMethod::Shut_RdWr;   break;
    default:    NN_SDK_LOG("WARNING: Invalid socket shutdown %d.\n", how);   break;
    }

    return mappedShutdownMethod;
}

uint32_t MapIoctlCommandValue(IoctlCommand command)
{
    int32_t mappedIoctlCommand = 0;

    switch( command )
    {
    case IoctlCommand::FionRead:    mappedIoctlCommand = FIONREAD;   break;
    case IoctlCommand::FionWrite:   mappedIoctlCommand = -1;        break;
    case IoctlCommand::FionSpace:   mappedIoctlCommand = -1;        break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::IoctlCommand %d.\n", command);   break;
    }

    if( mappedIoctlCommand == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::IoctlCommand %d is not supported by Win32/Win64.\n", command);
    }

    return mappedIoctlCommand;
}

IoctlCommand MapIoctlCommandValue(uint32_t command)
{
    IoctlCommand mappedIoctlCommand = IoctlCommand::FionRead;

    switch( command )
    {
    case SHUT_RD:    mappedIoctlCommand = IoctlCommand::FionRead;   break;
    default:    NN_SDK_LOG("WARNING: Invalid socket ioctl command %d.\n", command);   break;
    }

    return mappedIoctlCommand;
}

uint32_t MapFcntlCommandValue(FcntlCommand command)
{
    int32_t mappedFcntlCommand = 0;

    switch( command )
    {
    case FcntlCommand::F_GetFl: mappedFcntlCommand = F_GETFL;   break;
    case FcntlCommand::F_SetFl: mappedFcntlCommand = F_SETFL;   break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::FcntlCommand %d.\n", command);   break;
    }

    if( mappedFcntlCommand == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::FcntlCommand %d is not supported by Win32/Win64.\n", command);
    }

    return mappedFcntlCommand;
}

FcntlCommand MapFcntlCommandValue(uint32_t command)
{
    FcntlCommand mappedFcntlCommand = FcntlCommand::F_GetFl;

    switch( command )
    {
    case F_GETFL: mappedFcntlCommand = FcntlCommand::F_GetFl;   break;
    case F_SETFL: mappedFcntlCommand = FcntlCommand::F_SetFl;   break;
    default:    NN_SDK_LOG("WARNING: Invalid socket fcntl command %d.\n", command);   break;
    }

    return mappedFcntlCommand;
}

uint32_t MapFcntlFlagValue(FcntlFlag flag)
{
    uint32_t mappedFcntlFlag = 0;

    switch( flag )
    {
    case FcntlFlag::O_NonBlock: mappedFcntlFlag = O_NONBLOCK;   break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::FcntlFlag %d.\n", flag);   break;
    }

    return mappedFcntlFlag;
}

FcntlFlag MapFcntlFlagValue(uint32_t flag)
{
    FcntlFlag mappedFcntlFlag = FcntlFlag::None;

    switch( flag )
    {
    case O_NONBLOCK:    mappedFcntlFlag = FcntlFlag::O_NonBlock;    break;
    default:    NN_SDK_LOG("WARNING: Invalid socket fcntl flag %d.\n", flag);   break;
    }

    return mappedFcntlFlag;
}

int32_t MapLevelValue(Level level)
{
    int32_t mappedLevel = -1;

    switch( level )
    {
    case Level::Sol_Socket:     mappedLevel = SOL_SOCKET;   break;
    case Level::Sol_Ip:         mappedLevel = IPPROTO_IP;   break;
    case Level::Sol_Icmp:       mappedLevel = IPPROTO_ICMP; break;
    case Level::Sol_Tcp:        mappedLevel = IPPROTO_TCP;  break;
    case Level::Sol_Udp:        mappedLevel = IPPROTO_UDP;  break;
    case Level::Sol_UdpLite:    mappedLevel = IPPROTO_UDP;  break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::Level %d.\n", level);  break;
    }

    return mappedLevel;
}

Level MapLevelValue(int32_t level)
{
    Level mappedLevel = static_cast<Level>(0);

    switch( level )
    {
    case SOL_SOCKET:    mappedLevel = Level::Sol_Socket;    break;
    case IPPROTO_IP:    mappedLevel = Level::Sol_Ip;        break;
    case IPPROTO_ICMP:  mappedLevel = Level::Sol_Icmp;      break;
    case IPPROTO_TCP:   mappedLevel = Level::Sol_Tcp;       break;
    case IPPROTO_UDP:   mappedLevel = Level::Sol_Udp;       break;
    default:    NN_SDK_LOG("WARNING: Invalid socket level %d.\n", level);   break;
    }

    return mappedLevel;
}

int32_t MapOptionValue(Level level, Option option)
{
    int32_t mappedOption = -1;

    switch( level )
    {
    case Level::Sol_Socket:
    {
        switch( option )
        {
        case Option::So_Debug:                  mappedOption = SO_DEBUG;                    break;
        case Option::So_AcceptConn:             mappedOption = SO_ACCEPTCONN;               break;
        case Option::So_ReuseAddr:              mappedOption = SO_REUSEADDR;                break;
        case Option::So_KeepAlive:              mappedOption = SO_KEEPALIVE;                break;
        case Option::So_DontRoute:              mappedOption = SO_DONTROUTE;                break;
        case Option::So_Broadcast:              mappedOption = SO_BROADCAST;                break;
        case Option::So_UseLoopback:            mappedOption = SO_USELOOPBACK;              break;
        case Option::So_Linger:                 mappedOption = SO_LINGER;                   break;
        case Option::So_OobInline:              mappedOption = SO_OOBINLINE;                break;
        case Option::So_ReusePort:              mappedOption = -1;                          break;
        case Option::So_SndBuf:                 mappedOption = SO_SNDBUF;                   break;
        case Option::So_RcvBuf:                 mappedOption = SO_RCVBUF;                   break;
        case Option::So_SndLoWat:               mappedOption = SO_SNDLOWAT;                 break;
        case Option::So_RcvLoWat:               mappedOption = SO_RCVLOWAT;                 break;
        case Option::So_SndTimeo:               mappedOption = SO_SNDTIMEO;                 break;
        case Option::So_RcvTimeo:               mappedOption = SO_RCVTIMEO;                 break;
        case Option::So_Error:                  mappedOption = SO_ERROR;                    break;
        case Option::So_Type:                   mappedOption = SO_TYPE;                     break;
        case Option::So_Label:                  mappedOption = -1;                          break;
        case Option::So_PeerLabel:              mappedOption = -1;                          break;
        case Option::So_ListenQLimit:           mappedOption = -1;                          break;
        case Option::So_ListenQLen:             mappedOption = -1;                          break;
        case Option::So_ListenIncQLen:          mappedOption = -1;                          break;
        case Option::So_SetFib:                 mappedOption = -1;                          break;
        case Option::So_User_Cookie:            mappedOption = -1;                          break;
        case Option::So_Protocol:               mappedOption = -1;                          break;
        case Option::So_Vendor:                 mappedOption = -1;                          break;
        case Option::So_Nn_Linger:              mappedOption = -1;                          break;
        case Option::So_Nn_Shutdown_Exempt:     mappedOption = -1;                          break;
        default:    NN_SDK_LOG("WARNING: Invalid nn::socket::Option %d for Level::Sol_Socket.\n", option);        break;
        }
    }
    break;

    case Level::Sol_Ip:
    {
        switch( option )
        {
        case Option::Ip_Options:                mappedOption = IP_OPTIONS;                  break;
        case Option::Ip_HdrIncl:                mappedOption = IP_HDRINCL;                  break;
        case Option::Ip_Tos:                    mappedOption = IP_TOS;                      break;
        case Option::Ip_Ttl:                    mappedOption = IP_TTL;                      break;
        case Option::Ip_RecvOpts:               mappedOption = -1;                          break;
        case Option::Ip_Multicast_If:           mappedOption = IP_MULTICAST_IF;             break;
        case Option::Ip_Multicast_Ttl:          mappedOption = IP_MULTICAST_TTL;            break;
        case Option::Ip_Multicast_Loop:         mappedOption = IP_MULTICAST_LOOP;           break;
        case Option::Ip_Add_Membership:         mappedOption = IP_ADD_MEMBERSHIP;           break;
        case Option::Ip_Drop_Membership:        mappedOption = IP_DROP_MEMBERSHIP;          break;
        case Option::Ip_Multicast_Vif:          mappedOption = -1;                          break;
        case Option::Ip_Rsvp_On:                mappedOption = -1;                          break;
        case Option::Ip_Rsvp_Off:               mappedOption = -1;                          break;
        case Option::Ip_Rsvp_Vif_On:            mappedOption = -1;                          break;
        case Option::Ip_Rsvp_Vif_Off:           mappedOption = -1;                          break;
        case Option::Ip_PortRange:              mappedOption = -1;                          break;
        case Option::Ip_Faith:                  mappedOption = -1;                          break;
        case Option::Ip_OnesBcast:              mappedOption = -1;                          break;
        case Option::Ip_BindAny:                mappedOption = -1;                          break;
        case Option::Ip_RecvTtl:                mappedOption = -1;                          break;
        case Option::Ip_MinTtl:                 mappedOption = -1;                          break;
        case Option::Ip_DontFrag:               mappedOption = -1;                          break;
        case Option::Ip_RecvTos:                mappedOption = -1;                          break;
        case Option::Ip_Add_Source_Membership:  mappedOption = IP_ADD_SOURCE_MEMBERSHIP;    break;
        case Option::Ip_Drop_Source_Membership: mappedOption = IP_DROP_SOURCE_MEMBERSHIP;   break;
        case Option::Ip_Block_Source:           mappedOption = IP_BLOCK_SOURCE;             break;
        case Option::Ip_Unblock_Source:         mappedOption = IP_UNBLOCK_SOURCE;           break;
        default:    NN_SDK_LOG("WARNING: Invalid nn::socket::Option %d for Level::Sol_Ip.\n", option);        break;
        }
    }
    break;

    case Level::Sol_Tcp:
    {
        switch( option )
        {
        case Option::Tcp_NoDelay:               mappedOption = TCP_NODELAY;                 break;
        case Option::Tcp_MaxSeg:                mappedOption = TCP_MAXSEG;                  break;
        case Option::Tcp_NoPush:                mappedOption = -1;                          break;
        case Option::Tcp_NoOpt:                 mappedOption = -1;                          break;
        case Option::Tcp_Md5Sig:                mappedOption = -1;                          break;
        case Option::Tcp_Info:                  mappedOption = -1;                          break;
        case Option::Tcp_Congestion:            mappedOption = -1;                          break;
        case Option::Tcp_KeepInit:              mappedOption = -1;                          break;
        case Option::Tcp_KeepIdle:              mappedOption = -1;                          break;
        case Option::Tcp_KeepIntvl:             mappedOption = -1;                          break;
        case Option::Tcp_KeepCnt:               mappedOption = -1;                          break;
        case Option::Tcp_Vendor:                mappedOption = -1;                          break;
        default:    NN_SDK_LOG("WARNING: Invalid nn::socket::Option %d for Level::Sol_Tcp.\n", option);        break;
        }
    }
    break;

    default:
        NN_SDK_LOG("WARNING: Invalid option level %d.\n", level);
        break;
    }

    if( mappedOption == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::Option %d is not supported by Win32/Win64.\n", option);
    }

    return mappedOption;
} // NOLINT(impl/function_size)

Option MapOptionValue(int32_t level, int32_t option)
{
    Option mappedOption = static_cast<Option>(0);

    switch( level )
    {
    case SOL_SOCKET:
    {
        switch( option )
        {
        case SO_DEBUG:          mappedOption = Option::So_Debug;        break;
        case SO_ACCEPTCONN:     mappedOption = Option::So_AcceptConn;   break;
        case SO_REUSEADDR:      mappedOption = Option::So_ReuseAddr;    break;
        case SO_KEEPALIVE:      mappedOption = Option::So_KeepAlive;    break;
        case SO_DONTROUTE:      mappedOption = Option::So_DontRoute;    break;
        case SO_BROADCAST:      mappedOption = Option::So_Broadcast;    break;
        case SO_USELOOPBACK:    mappedOption = Option::So_UseLoopback;  break;
        case SO_LINGER:         mappedOption = Option::So_Linger;       break;
        case SO_OOBINLINE:      mappedOption = Option::So_OobInline;    break;
        case SO_SNDBUF:         mappedOption = Option::So_SndBuf;       break;
        case SO_RCVBUF:         mappedOption = Option::So_RcvBuf;       break;
        case SO_SNDLOWAT:       mappedOption = Option::So_SndLoWat;     break;
        case SO_RCVLOWAT:       mappedOption = Option::So_RcvLoWat;     break;
        case SO_SNDTIMEO:       mappedOption = Option::So_SndTimeo;     break;
        case SO_RCVTIMEO:       mappedOption = Option::So_RcvTimeo;     break;
        case SO_ERROR:          mappedOption = Option::So_Error;        break;
        case SO_TYPE:           mappedOption = Option::So_Type;         break;
        default:    NN_SDK_LOG("WARNING: Invalid socket option %d fo SOL_SOCKET.\n", option);        break;
        }
    }
    break;

    case IPPROTO_IP:
    {
        switch( option )
        {
        case IP_OPTIONS:                mappedOption = Option::Ip_Options;                  break;
        case IP_HDRINCL:                mappedOption = Option::Ip_HdrIncl;                  break;
        case IP_TOS:                    mappedOption = Option::Ip_Tos;                      break;
        case IP_TTL:                    mappedOption = Option::Ip_Ttl;                      break;
        case IP_MULTICAST_IF:           mappedOption = Option::Ip_Multicast_If;             break;
        case IP_MULTICAST_TTL:          mappedOption = Option::Ip_Multicast_Ttl;            break;
        case IP_MULTICAST_LOOP:         mappedOption = Option::Ip_Multicast_Loop;           break;
        case IP_ADD_MEMBERSHIP:         mappedOption = Option::Ip_Add_Membership;           break;
        case IP_DROP_MEMBERSHIP:        mappedOption = Option::Ip_Drop_Membership;          break;
        case IP_ADD_SOURCE_MEMBERSHIP:  mappedOption = Option::Ip_Add_Source_Membership;    break;
        case IP_DROP_SOURCE_MEMBERSHIP: mappedOption = Option::Ip_Drop_Source_Membership;   break;
        case IP_BLOCK_SOURCE:           mappedOption = Option::Ip_Block_Source;             break;
        case IP_UNBLOCK_SOURCE:         mappedOption = Option::Ip_Unblock_Source;           break;
        default:    NN_SDK_LOG("WARNING: Invalid socket option %d fo SOL_IP.\n", option);        break;
        }
    }
    break;

    case IPPROTO_TCP:
    {
        switch( option )
        {
        case TCP_NODELAY:   mappedOption = Option::Tcp_NoDelay; break;
        case TCP_MAXSEG:    mappedOption = Option::Tcp_MaxSeg;  break;
        default:    NN_SDK_LOG("WARNING: Invalid socket option %d fo SOL_TCP.\n", option);        break;
        }
    }
    break;

    default:
        NN_SDK_LOG("WARNING: Invalid option level %d.\n", level);
        break;
    }

    return mappedOption;
}

int32_t MapErrnoValue(Errno error)
{
    int32_t mappedErrno = -1;

    switch (error)
    {
    case Errno::EIntr:            mappedErrno = WSAEINTR;           break;
    case Errno::EBadf:            mappedErrno = WSAEBADF;           break;
    case Errno::EAcces:           mappedErrno = WSAEACCES;          break;
    case Errno::EFault:           mappedErrno = WSAEFAULT;          break;
    case Errno::EInval:           mappedErrno = WSAEINVAL;          break;
    case Errno::EMFile:           mappedErrno = WSAEMFILE;          break;
    case Errno::EWouldBlock:      mappedErrno = WSAEWOULDBLOCK;     break;
    case Errno::EInProgress:      mappedErrno = WSAEINPROGRESS;     break;
    case Errno::EAlready:         mappedErrno = WSAEALREADY;        break;
    case Errno::ENotSock:         mappedErrno = WSAENOTSOCK;        break;
    case Errno::EDestAddrReq:     mappedErrno = WSAEDESTADDRREQ;    break;
    case Errno::EMsgSize:         mappedErrno = WSAEMSGSIZE;        break;
    case Errno::EPrototype:       mappedErrno = WSAEPROTOTYPE;      break;
    case Errno::ENoProtoOpt:      mappedErrno = WSAENOPROTOOPT;     break;
    case Errno::EProtoNoSupport:  mappedErrno = WSAEPROTONOSUPPORT; break;
    case Errno::ESocktNoSupport:  mappedErrno = WSAESOCKTNOSUPPORT; break;
    case Errno::EOpNotSupp:       mappedErrno = WSAEOPNOTSUPP;      break;
    case Errno::EPfNoSupport:     mappedErrno = WSAEPFNOSUPPORT;    break;
    case Errno::EAfNoSupport:     mappedErrno = WSAEAFNOSUPPORT;    break;
    case Errno::EAddrInUse:       mappedErrno = WSAEADDRINUSE;      break;
    case Errno::EAddrNotAvail:    mappedErrno = WSAEADDRNOTAVAIL;   break;
    case Errno::ENetDown:         mappedErrno = WSAENETDOWN;        break;
    case Errno::ENetUnreach:      mappedErrno = WSAENETUNREACH;     break;
    case Errno::ENetReset:        mappedErrno = WSAENETRESET;       break;
    case Errno::EConnAborted:     mappedErrno = WSAECONNABORTED;    break;
    case Errno::EConnReset:       mappedErrno = WSAECONNRESET;      break;
    case Errno::ENoBufs:          mappedErrno = WSAENOBUFS;         break;
    case Errno::EIsConn:          mappedErrno = WSAEISCONN;         break;
    case Errno::ENotConn:         mappedErrno = WSAENOTCONN;        break;
    case Errno::EShutDown:        mappedErrno = WSAESHUTDOWN;       break;
    case Errno::ETooManyRefs:     mappedErrno = WSAETOOMANYREFS;    break;
    case Errno::ETimedOut:        mappedErrno = WSAETIMEDOUT;       break;
    case Errno::EConnRefused:     mappedErrno = WSAECONNREFUSED;    break;
    case Errno::ELoop:            mappedErrno = WSAELOOP;           break;
    case Errno::ENameTooLong:     mappedErrno = WSAENAMETOOLONG;    break;
    case Errno::EHostDown:        mappedErrno = WSAEHOSTDOWN;       break;
    case Errno::EHostUnreach:     mappedErrno = WSAEHOSTUNREACH;    break;
    case Errno::ENotEmpty:        mappedErrno = WSAENOTEMPTY;       break;
    case Errno::EProcLim:         mappedErrno = WSAEPROCLIM;        break;
    case Errno::EUsers:           mappedErrno = WSAEUSERS;          break;
    case Errno::EDQuot:           mappedErrno = WSAEDQUOT;          break;
    case Errno::EStale:           mappedErrno = WSAESTALE;          break;
    case Errno::ERemote:          mappedErrno = WSAEREMOTE;         break;
    case Errno::ECanceled:        mappedErrno = WSAECANCELLED;      break;
    default:                      mappedErrno = static_cast<int32_t>(error);   break;
    }

    return mappedErrno;
}

Errno MapErrnoValue(int32_t error)
{
    Errno mappedErrno = Errno::ESuccess;

    switch( error )
    {
    case WSAEINTR:           mappedErrno = Errno::EIntr;            break;
    case WSAEBADF:           mappedErrno = Errno::EBadf;            break;
    case WSAEACCES:          mappedErrno = Errno::EAcces;           break;
    case WSAEFAULT:          mappedErrno = Errno::EInval;           break; // Remapped
    case WSAEINVAL:          mappedErrno = Errno::EInval;           break;
    case WSAEMFILE:          mappedErrno = Errno::EMFile;           break;
    case WSAEWOULDBLOCK:     mappedErrno = Errno::EWouldBlock;      break;
    case WSAEINPROGRESS:     mappedErrno = Errno::EInProgress;      break;
    case WSAEALREADY:        mappedErrno = Errno::EAlready;         break;
    case WSAENOTSOCK:        mappedErrno = Errno::EBadf;            break; // Remapped
    case WSAEDESTADDRREQ:    mappedErrno = Errno::EDestAddrReq;     break;
    case WSAEMSGSIZE:        mappedErrno = Errno::EMsgSize;         break;
    case WSAEPROTOTYPE:      mappedErrno = Errno::EPrototype;       break;
    case WSAENOPROTOOPT:     mappedErrno = Errno::ENoProtoOpt;      break;
    case WSAEPROTONOSUPPORT: mappedErrno = Errno::EProtoNoSupport;  break;
    case WSAESOCKTNOSUPPORT: mappedErrno = Errno::ESocktNoSupport;  break;
    case WSAEOPNOTSUPP:      mappedErrno = Errno::EOpNotSupp;       break;
    case WSAEPFNOSUPPORT:    mappedErrno = Errno::EPfNoSupport;     break;
    case WSAEAFNOSUPPORT:    mappedErrno = Errno::EAfNoSupport;     break;
    case WSAEADDRINUSE:      mappedErrno = Errno::EAddrInUse;       break;
    case WSAEADDRNOTAVAIL:   mappedErrno = Errno::EAddrNotAvail;    break;
    case WSAENETDOWN:        mappedErrno = Errno::ENetDown;         break;
    case WSAENETUNREACH:     mappedErrno = Errno::ENetUnreach;      break;
    case WSAENETRESET:       mappedErrno = Errno::ENetReset;        break;
    case WSAECONNABORTED:    mappedErrno = Errno::EConnAborted;     break;
    case WSAECONNRESET:      mappedErrno = Errno::EConnReset;       break;
    case WSAENOBUFS:         mappedErrno = Errno::ENoBufs;          break;
    case WSAEISCONN:         mappedErrno = Errno::EIsConn;          break;
    case WSAENOTCONN:        mappedErrno = Errno::ENotConn;         break;
    case WSAESHUTDOWN:       mappedErrno = Errno::EShutDown;        break;
    case WSAETOOMANYREFS:    mappedErrno = Errno::ETooManyRefs;     break;
    case WSAETIMEDOUT:       mappedErrno = Errno::EAgain;           break;  // Remapped
    case WSAECONNREFUSED:    mappedErrno = Errno::EConnRefused;     break;
    case WSAELOOP:           mappedErrno = Errno::ELoop;            break;
    case WSAENAMETOOLONG:    mappedErrno = Errno::ENameTooLong;     break;
    case WSAEHOSTDOWN:       mappedErrno = Errno::EHostDown;        break;
    case WSAEHOSTUNREACH:    mappedErrno = Errno::EHostUnreach;     break;
    case WSAENOTEMPTY:       mappedErrno = Errno::ENotEmpty;        break;
    case WSAEPROCLIM:        mappedErrno = Errno::EProcLim;         break;
    case WSAEUSERS:          mappedErrno = Errno::EUsers;           break;
    case WSAEDQUOT:          mappedErrno = Errno::EDQuot;           break;
    case WSAESTALE:          mappedErrno = Errno::EStale;           break;
    case WSAEREMOTE:         mappedErrno = Errno::ERemote;          break;
    case WSAECANCELLED:      mappedErrno = Errno::ECanceled;        break;
    default:                 mappedErrno = static_cast<Errno>(error); break;
    }

    return mappedErrno;
}

int32_t MapHErrnoValue(HErrno error)
{
    int32_t mappedHErrno = -1;

    switch( error )
    {
    case HErrno::Netdb_Internal:  mappedHErrno = -1;                break;
    case HErrno::Netdb_Success:   mappedHErrno = 0;                 break;
    case HErrno::Host_Not_Found:  mappedHErrno = WSAHOST_NOT_FOUND; break;
    case HErrno::Try_Again:       mappedHErrno = WSATRY_AGAIN;      break;
    case HErrno::No_Recovery:     mappedHErrno = WSANO_RECOVERY;    break;
    case HErrno::No_Data:         mappedHErrno = WSANO_DATA;        break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::HErrno %d.\n", error);  break;
    }

    if( mappedHErrno == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::HErrno %d is not supported by Win32/Win64.\n", error);
    }

    return mappedHErrno;
}

HErrno MapHErrnoValue(int32_t error)
{
    HErrno mappedHErrno = static_cast<HErrno>(-1);

    switch( error )
    {
    case -1:                mappedHErrno = HErrno::Netdb_Internal;  break;
    case 0:                 mappedHErrno = HErrno::Netdb_Success;   break;
    case WSAHOST_NOT_FOUND: mappedHErrno = HErrno::Host_Not_Found;  break;
    case WSATRY_AGAIN:      mappedHErrno = HErrno::Try_Again;       break;
    case WSANO_RECOVERY:    mappedHErrno = HErrno::No_Recovery;     break;
    case WSANO_DATA:        mappedHErrno = HErrno::No_Data;         break;
    default:    NN_SDK_LOG("WARNING: Invalid socket h_errno %d.\n", error);  break;
    }

    return mappedHErrno;
}

int32_t MapAiErrnoValue(AiErrno error)
{
    int32_t mappedAiErrno = -1;

    switch( error )
    {
    case AiErrno::EAi_Success:      mappedAiErrno = 0;                      break;
    case AiErrno::EAi_AddrFamily:   mappedAiErrno = -1;                     break;
    case AiErrno::EAi_Again:        mappedAiErrno = WSATRY_AGAIN;           break;
    case AiErrno::EAi_BadFlags:     mappedAiErrno = WSAEINVAL;              break;
    case AiErrno::EAi_Fail:         mappedAiErrno = WSANO_RECOVERY;         break;
    case AiErrno::EAi_Family:       mappedAiErrno = WSAEAFNOSUPPORT;        break;
    case AiErrno::EAi_Memory:       mappedAiErrno = WSA_NOT_ENOUGH_MEMORY;  break;
    case AiErrno::EAi_NoData:       mappedAiErrno = WSANO_DATA;             break;
    case AiErrno::EAi_NoName:       mappedAiErrno = WSAHOST_NOT_FOUND;      break;
    case AiErrno::EAi_Service:      mappedAiErrno = WSATYPE_NOT_FOUND;      break;
    case AiErrno::EAi_SockType:     mappedAiErrno = WSAESOCKTNOSUPPORT;     break;
    case AiErrno::EAi_System:       mappedAiErrno = -1;                     break;
    case AiErrno::EAi_BadHints:     mappedAiErrno = -1;                     break;
    case AiErrno::EAi_Protocol:     mappedAiErrno = -1;                     break;
    case AiErrno::EAi_Overflow:     mappedAiErrno = -1;                     break;
    case AiErrno::EAi_Max:          mappedAiErrno = -1;                     break;
    default:    NN_SDK_LOG("WARNING: Invalid nn::socket::AiErrno %d.\n", error);  break;
    }

    if( mappedAiErrno == -1 )
    {
        NN_SDK_LOG("WARNING: nn::socket::AiErrno %d is not supported by Win32/Win64.\n", error);
    }

    return mappedAiErrno;
}

AiErrno MapAiErrnoValue(int32_t error)
{
    AiErrno mappedAiErrno = static_cast<AiErrno>(-1);

    switch( error )
    {
    case 0:                     mappedAiErrno = AiErrno::EAi_Success;   break;
    case WSATRY_AGAIN:          mappedAiErrno = AiErrno::EAi_Again;     break;
    case WSAEINVAL:             mappedAiErrno = AiErrno::EAi_BadFlags;  break;
    case WSANO_RECOVERY:        mappedAiErrno = AiErrno::EAi_Fail;      break;
    case WSAEAFNOSUPPORT:       mappedAiErrno = AiErrno::EAi_Family;    break;
    case WSA_NOT_ENOUGH_MEMORY: mappedAiErrno = AiErrno::EAi_Memory;    break;
    case WSANO_DATA:            mappedAiErrno = AiErrno::EAi_NoData;    break;
    case WSAHOST_NOT_FOUND:     mappedAiErrno = AiErrno::EAi_NoName;    break;
    case WSATYPE_NOT_FOUND:     mappedAiErrno = AiErrno::EAi_Service;   break;
    case WSAESOCKTNOSUPPORT:    mappedAiErrno = AiErrno::EAi_SockType;  break;
    default:    NN_SDK_LOG("WARNING: Invalid socket ai_errno %d.\n", error);  break;
    }

    return mappedAiErrno;
}

}}} // namespace nn::socket::detail
