﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/socket/resolver/resolver_Client.h>
#include <nn/socket/private/socket_PlatformTypesTranslation.h>
#include "socket_Allocator.h"
#include "socket_Api.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
// VC warns about gethostbyname / gethostbyaddr being deprecated
#pragma warning(push)
#pragma warning(disable : 4996)
#include <nn/socket/socket_TypesPrivate.h>

#else // END NN_BUILD_CONFIG_OS_WIN32

#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_Errno.h>
#define htonl InetHtonl

#include "../resolver/resolver_ThreadLocalStorage.h"
#include "../resolver/serializer/serializer.h"
#include "../resolver/serializer/serializer_Specializations.h"

#endif // ! Windows

#include <cstring>

namespace nn     {
    namespace socket {
        namespace detail {

            int g_SocketWsaHErrno = 0;
            HErrno g_SocketHErrno = HErrno::Netdb_Success;

            // GetHostByName
            HostEnt* GetHostEntByName(const char* pName,
                                      const ResolverOption* pOptions,
                                      size_t optionsCount)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                static HostEnt hostEnt = {};
                hostent *he = ::gethostbyname(pName);
                g_SocketWsaHErrno = WSAGetLastError();

                if( nullptr != he )
                {
                    CopyFromPlatform(&hostEnt, he);
                }

                return (nullptr != he)? &hostEnt : nullptr;    // TODO: NOT thread-safe! Could use C++11 thread_local to fix.
#else
                return nn::socket::resolver::GetHostEntByName(pName, pOptions, optionsCount);
#endif
            }

            // GetHostByName
            hostent* GetHostByName(const char* pName,
                                   const ResolverOption* options, size_t optionsCount)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                NN_UNUSED(options);
                NN_UNUSED(optionsCount);
                struct hostent *result = NULL;
                result = ::gethostbyname(pName);
                g_SocketWsaHErrno = WSAGetLastError();
                return result;
#else
                return nn::socket::resolver::GetHostByName(pName, options, optionsCount);
#endif
            }

            // GetHostByAddr
            HostEnt* GetHostEntByAddr(const void* pAddress, SockLenT length, Family addressFamily,
                                      const ResolverOption* pOptions, size_t optionsCount)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                static HostEnt hostEnt = {};
                hostent *he = ::gethostbyaddr((const char *) pAddress, length, MapFamilyValue(addressFamily));
                g_SocketWsaHErrno = WSAGetLastError();

                if( nullptr != he )
                {
                    CopyFromPlatform(&hostEnt, he);
                }

                return (nullptr != he)? &hostEnt : nullptr;    // TODO: NOT thread-safe! Could use C++11 thread_local to fix.
#else
                return nn::socket::resolver::GetHostEntByAddr(pAddress, length, addressFamily, pOptions, optionsCount);
#endif
            }

            // GetHostByAddr
            hostent* GetHostByAddr(const void* pAddress, socklen_t length, int addressFamily,
                                   const ResolverOption* options, size_t optionsCount)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                NN_UNUSED(options);
                NN_UNUSED(optionsCount);
                hostent *result = NULL;
                result = ::gethostbyaddr((const char *) pAddress,length,addressFamily);
                g_SocketWsaHErrno = WSAGetLastError();
                return result;
#else
                return nn::socket::resolver::GetHostByAddr(pAddress, length, addressFamily, options, optionsCount);
#endif
            }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
            char* PopulateErrorString(int wsaError)
            {
                static char s_errorBuffer[nn::socket::Ni_MaxHost] = { '\0' };

                memset(s_errorBuffer, '\0', sizeof(s_errorBuffer));

                FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                               NULL,
                               wsaError,
                               MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                               reinterpret_cast<LPSTR>(&s_errorBuffer),
                               sizeof(s_errorBuffer),
                               NULL);

                return s_errorBuffer;
            };
#endif // WINDOWS

            const char* HStrError(nn::socket::HErrno errorNumber)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                return PopulateErrorString(static_cast<int>(errorNumber));
#else
                return nn::socket::resolver::GetHostStringError(static_cast<int>(errorNumber));
#endif
            }

            const char* HStrError(int errorNumber)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                return PopulateErrorString(errorNumber);
#else
                return nn::socket::resolver::GetHostStringError(errorNumber);
#endif
            }

            const char* GAIStrError(AiErrno errorCode)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                return ::gai_strerrorA(static_cast<int>(errorCode));
#else
                return nn::socket::resolver::GetGaiStringError(static_cast<int>(errorCode));
#endif
            }

            const char* GAIStrError(int errorCode)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                return ::gai_strerrorA(errorCode);
#else
                return nn::socket::resolver::GetGaiStringError(errorCode);
#endif
            }

        int ResolverGetOption(ResolverOption* pOptionOut, uint32_t key) NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            return -1;
#else
            return nn::socket::resolver::ResolverGetOption(pOptionOut, static_cast<ResolverOptionKey>(key));
#endif
        }

        int ResolverSetOption(const ResolverOption& option) NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            return -1;
#else
            return nn::socket::resolver::ResolverSetOption(option);
#endif
        }

            // GetAddrInfo
            AiErrno GetAddrInfo(const char* pNodeName,
                                const char* pServiceName,
                                const AddrInfo* pHints,
                                AddrInfo** pResult,
                                const ResolverOption* pOptions, size_t optionsCount)
                                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                AiErrno     rc          = AiErrno::EAi_System;
                addrinfo*   pAiResult   = nullptr;
                addrinfo    aiHints     = {};

                if (nullptr == pResult)
                {
                    nn::socket::SetLastError(Errno::EInval);
                    goto bail;
                }
                else if (nullptr != pHints)
                {
                    aiHints.ai_family = MapFamilyValue(pHints->ai_family);
                    aiHints.ai_socktype = MapTypeValue(pHints->ai_socktype);
                    aiHints.ai_protocol = MapProtocolValue(pHints->ai_protocol);
                    aiHints.ai_flags = MapAddrInfoFlagValue(pHints->ai_flags);

                    if (AF_UNSPEC == aiHints.ai_family)
                    {
                        aiHints.ai_family = AF_INET;
                    }
                    else if (AF_INET6 == aiHints.ai_family)
                    {
                        nn::socket::SetLastError(Errno::ENotSup);
                        goto bail;
                    };
                };

                rc = MapAiErrnoValue(::getaddrinfo(pNodeName,
                                                   pServiceName,
                                                   pHints == NULL ? NULL : &aiHints,
                                                   &pAiResult));

                if( nullptr != pAiResult )
                {
                    nn::socket::detail::CopyFromPlatform(pResult, pAiResult);

                    // CopyFromPlatform() allocated all new AddrInfo structure and
                    // copied over data allocated inside of getaddrinfo(), so
                    // that needs to be freed.
                    ::freeaddrinfo(pAiResult);
                }

            bail:
                return rc;
#else
                return nn::socket::resolver::GetAddrInfo(pNodeName, pServiceName,
                                                         pHints, pResult,
                                                         pOptions, optionsCount);
#endif
            }

            // GetAddrInfo
        int GetAddrInfo(const char* pNodeName, const char* pServiceName,
                        const addrinfo* pHints, addrinfo** pResult,
                        const ResolverOption* options, size_t optionsCount)
            NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            int rc = EAI_SYSTEM;
            NN_UNUSED(options);
            NN_UNUSED(optionsCount);
            struct addrinfo hints;
            memset(&hints, '\0', sizeof(struct addrinfo));

            if (NULL == pResult)
            {
                nn::socket::SetLastError(Errno::EInval);
                goto bail;
            }
            else if (NULL != pHints)
            {
                hints = *pHints;

                if (AF_UNSPEC == hints.ai_family)
                {
                    hints.ai_family = AF_INET;
                }
                else if (AF_INET6 == hints.ai_family)
                {
                    nn::socket::SetLastError(Errno::ENotSup);
                    goto bail;
                };
            };

            rc = ::getaddrinfo(pNodeName,
                               pServiceName,
                               pHints == NULL ? NULL : &hints,
                               pResult);

        bail:
            return rc;
#else
            return nn::socket::resolver::GetAddrInfo(pNodeName, pServiceName, pHints, pResult,
                                                     options, optionsCount);
#endif
        }

        // GetNameInfo
        AiErrno GetNameInfo(const SockAddr* pSocketAddress, SockLenT socketAddressLength,
                            char* pHost, size_t hostLength,
                            char* pService, size_t serviceLength,
                            NameInfoFlag flags,
                            const ResolverOption* pOptions, size_t optionsCount)
            NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            if( nullptr == pSocketAddress )
            {
                nn::socket::SetLastError(Errno::EInval);
                return AiErrno::EAi_System;
            }

            if( socketAddressLength == sizeof(SockAddrIn) )
            {
                socketAddressLength = sizeof(sockaddr_in);
            }

            sockaddr sa = {};
            nn::socket::detail::CopyToPlatform(reinterpret_cast<sockaddr_in*>(&sa), reinterpret_cast<const SockAddrIn*>(pSocketAddress));

            return static_cast<AiErrno>(::getnameinfo(&sa, socketAddressLength,
                                                      pHost, static_cast<DWORD>(hostLength),
                                                      pService, static_cast <DWORD>(serviceLength),
                                                      static_cast<int>(flags)));
#else
            return nn::socket::resolver::GetNameInfo(pSocketAddress, socketAddressLength,
                                                     pHost, hostLength,
                                                     pService, serviceLength,
                                                     flags,
                                                     pOptions, optionsCount);
#endif
        }

            // GetNameInfo
            int GetNameInfo(const sockaddr* pSocketAddress, socklen_t socketAddressLength,
                            char* pHost, socklen_t hostLength,
                            char* pService, socklen_t serviceLength,
                            int flags,
                            const ResolverOption* options, size_t optionsCount)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                NN_UNUSED(options);
                NN_UNUSED(optionsCount);
                return ::getnameinfo(pSocketAddress, socketAddressLength,
                                     pHost, static_cast<DWORD>(hostLength),
                                     pService, static_cast <DWORD>(serviceLength),
                                     flags);
#else
                return nn::socket::resolver::GetNameInfo(pSocketAddress, socketAddressLength,
                                                         pHost, hostLength,
                                                         pService, serviceLength,
                                                         flags,
                                                         options, optionsCount);
#endif
            }

            //
            void FreeAddrInfo(AddrInfo* pAddrInfoStorage)
                NN_NOEXCEPT
            {
                nn::socket::detail::FreeCopiedAddrInfo(pAddrInfoStorage);
            }

            //
            void FreeAddrInfo(addrinfo* pAddrInfoStorage)
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                return ::freeaddrinfo(pAddrInfoStorage);
#else
                if ( NULL != pAddrInfoStorage )
                {
                    nn::socket::resolver::serializer::FreeAddrinfoContents(*pAddrInfoStorage);
                    nn::socket::detail::Free(pAddrInfoStorage);
                };
#endif
            }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#define SET_LAST_ERROR(x) SetLastError(x)
#else
#define SET_LAST_ERROR(x) { *(GetHErrno()) = x; };
#endif

            HErrno* GetHError()
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                g_SocketHErrno = MapHErrnoValue(g_SocketWsaHErrno);
                return &g_SocketHErrno;
#else
                return reinterpret_cast<HErrno*>(&nn::socket::resolver::tls::Client::InternalHostErrno());
#endif
            }

            int* GetHErrno()
                NN_NOEXCEPT
            {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                return &g_SocketWsaHErrno;
#else
                return &nn::socket::resolver::tls::Client::InternalHostErrno();
#endif
            }
        }
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
// end windows warnings about gethostbyname / gethostbyaddr
#pragma warning(pop)
#endif
