﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>

#include "sf_HipcEmulatedInProcessServiceResolver.h"
#include "sf_HipcEmulatedNamedPipeServiceResolver.h"
#include "sf_HipcEmulatedPort.h"
#include "sf_HipcEmulatedSession.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <cstring>
#include <algorithm>
#include <nn/os/os_Mutex.h>
#include <type_traits>
#include <new>

namespace nn { namespace sf { namespace hipc {

namespace {

void CheckServiceNameLength(const char* name) NN_NOEXCEPT
{
    auto last = name + HipcServiceNameLengthMax + 1;
    auto p = std::find(name, last, 0);
    NN_ABORT_UNLESS(p != last);
}

template <typename T>
struct Singleton
{
    nn::os::MutexType _mutex;
    typename std::aligned_storage<sizeof(T), NN_ALIGNOF(T)>::type _storage;
    T* _p;

    T& Get()
    {
        nn::os::LockMutex(&_mutex);
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::UnlockMutex(&_mutex);
        };
        if (!_p)
        {
            this->_p = new (&_storage) T;
        }
        return *_p;
    }
};

Singleton<HipcEmulatedInProcessServiceResolver> g_InProcessServiceResolver = { NN_OS_MUTEX_INITIALIZER(false) };
Singleton<HipcEmulatedNamedPipeServiceResolver> g_NamedPipeServiceResolver = { NN_OS_MUTEX_INITIALIZER(false) };

}

void InitializeHipcServiceResolution() NN_NOEXCEPT
{
    // nop
}

void FinalizeHipcServiceResolution() NN_NOEXCEPT
{
    // nop
}

Result RegisterHipcService(HipcServerPortHandle* pOut, int32_t maxSessions, const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    return g_NamedPipeServiceResolver.Get().RegisterService(pOut, name, maxSessions);
}

Result RegisterInProcessHipcService(HipcServerPortHandle* pOut, int32_t maxSessions, const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    return g_InProcessServiceResolver.Get().RegisterService(pOut, name, maxSessions);
}

void UnregisterHipcService(const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    g_NamedPipeServiceResolver.Get().UnregisterService(name);
}

void UnregisterInProcessHipcService(const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    g_InProcessServiceResolver.Get().UnregisterService(name);
}

bool IsHipcServiceRegistered(const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    return g_InProcessServiceResolver.Get().IsServiceRegistered(name) || g_NamedPipeServiceResolver.Get().IsServiceRegistered(name);
}

Result ConnectToHipcService(HipcClientSessionHandle* pOut, const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    if (g_InProcessServiceResolver.Get().IsServiceRegistered(name))
    {
        return g_InProcessServiceResolver.Get().ConnectToService(pOut, name);
    }
    else
    {
        return g_NamedPipeServiceResolver.Get().ConnectToService(pOut, name);
    }
}

Result TryConnectToHipcService(bool* pValid, HipcClientSessionHandle* pOut, const char* name) NN_NOEXCEPT
{
    CheckServiceNameLength(name);
    Result result = g_InProcessServiceResolver.Get().TryConnectToService(pValid, pOut, name);
    if (*pValid)
    {
        return result;
    }
    else
    {
        return g_NamedPipeServiceResolver.Get().TryConnectToService(pValid, pOut, name);
    }
}

}}}
