﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sf_HipcEmulatedInProcessPort.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/sf/impl/sf_AllocationPolicies.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/os/os_Semaphore.h>

#include <memory>
#include <mutex>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_Exchange.h>

#include "sf_HipcEmulatedObjectBase.h"
#include "sf_HipcEmulatedInProcessSession.h"


namespace nn { namespace sf { namespace hipc {

class HipcEmulatedInProcessPort
    : public HipcEmulatedObjectBase
{
private:

    class Request
    {
    private:

        SharedPointer<ISharedObject> m_Parent;
        nn::os::Mutex* m_pMutex;
        nn::os::ConditionVariable m_Condition;
        bool m_Accepted;
        HipcEmulatedClientSession* m_Session;

    public:

        explicit Request(ISharedObject* parent, nn::os::Mutex* pMutex) NN_NOEXCEPT
            : m_Parent(parent, true)
            , m_pMutex(pMutex)
            , m_Accepted(false)
        {
        }

        void Accept(HipcEmulatedClientSession* clientSession) NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pMutex)> lk(*m_pMutex);
            this->m_Session = clientSession;
            this->m_Accepted = true;
            m_Condition.Signal();
        }

        HipcEmulatedClientSession* Wait() NN_NOEXCEPT
        {
            std::lock_guard<decltype(*m_pMutex)> lk(*m_pMutex);
            while (!m_Accepted)
            {
                m_Condition.Wait(*m_pMutex);
            }
            return m_Session;
        }

    };

    nn::os::Mutex m_Mutex;
    uintptr_t m_ConnectQueueBuffer;
    nn::os::MessageQueue m_ConnectQueue;
    nn::os::Semaphore m_SessionCountSemaphore;

protected:

    explicit HipcEmulatedInProcessPort(int maxSessions) NN_NOEXCEPT
        : m_Mutex(false)
        , m_ConnectQueue(&m_ConnectQueueBuffer, 1)
        , m_SessionCountSemaphore(maxSessions, maxSessions)
    {
    }

public:

    static HipcEmulatedInProcessPort* Create(int maxSessions) NN_NOEXCEPT
    {
        return Factory<HipcEmulatedInProcessPort>::Create(maxSessions);
    }

    // for client
    void AttachConnectEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::InitializeMultiWaitHolder(pHolder, m_SessionCountSemaphore.GetBase());
    }

    // for client
    HipcEmulatedClientSession* Connect(bool blocking) NN_NOEXCEPT
    {
        if (blocking)
        {
            m_SessionCountSemaphore.Acquire();
        }
        else
        {
            if (!m_SessionCountSemaphore.TryAcquire())
            {
                return nullptr;
            }
        }
        Request request(this, &m_Mutex);
        m_ConnectQueue.Send(reinterpret_cast<uintptr_t>(&request));
        return request.Wait();
    }

    // for server
    void AttachAcceptEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::InitializeMultiWaitHolder(pHolder, m_ConnectQueue.GetBase(), nn::os::MessageQueueWaitType_WaitForNotEmpty);
    }

    // for server
    HipcEmulatedServerSession* Accept() NN_NOEXCEPT
    {
        uintptr_t p;
        m_ConnectQueue.Receive(&p);
        auto& request = *reinterpret_cast<Request*>(p);
        auto sessionPair = CreateHipcEmulatedInProcessSessionPair(this);
        request.Accept(sessionPair.second);
        return sessionPair.first;
    }

    // utility
    static std::pair<
        HipcEmulatedInProcessServerPort*,
        HipcEmulatedInProcessClientPort*
    > CreateHipcEmulatedInProcessPortPair(int maxSessions) NN_NOEXCEPT;

    void OnChildSessionClosed() NN_NOEXCEPT
    {
        m_SessionCountSemaphore.Release();
    }

};

HipcEmulatedInProcessClientPort::HipcEmulatedInProcessClientPort(HipcEmulatedInProcessPort* port) NN_NOEXCEPT
    : m_Port(port, true)
{
}

HipcEmulatedInProcessClientPort::~HipcEmulatedInProcessClientPort() NN_NOEXCEPT
{
}

void HipcEmulatedInProcessClientPort::AttachConnectEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    return m_Port->AttachConnectEvent(pHolder);
}

HipcEmulatedClientSession* HipcEmulatedInProcessClientPort::Connect(bool blocking) NN_NOEXCEPT
{
    return m_Port->Connect(blocking);
}

HipcEmulatedInProcessServerPort::HipcEmulatedInProcessServerPort(HipcEmulatedInProcessPort* port) NN_NOEXCEPT
    : m_Port(port, true)
{
}

HipcEmulatedInProcessServerPort::~HipcEmulatedInProcessServerPort() NN_NOEXCEPT
{
}

void HipcEmulatedInProcessServerPort::AttachAcceptEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    m_Port->AttachAcceptEvent(pHolder);
}

void HipcEmulatedInProcessServerPort::WaitAccept() NN_NOEXCEPT
{
    nn::os::MultiWaitHolderType holder;
    AttachAcceptEvent(&holder);
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::LinkMultiWaitHolder(&multiWait, &holder);
    nn::os::WaitAny(&multiWait);
    nn::os::UnlinkMultiWaitHolder(&holder);
    nn::os::FinalizeMultiWait(&multiWait);
    nn::os::FinalizeMultiWaitHolder(&holder);
}

HipcEmulatedServerSession* HipcEmulatedInProcessServerPort::Accept() NN_NOEXCEPT
{
    return m_Port->Accept();
}

namespace {

    template <typename T>
    SharedPointer<T> MakeSharedAttached(T* p) NN_NOEXCEPT
    {
        return SharedPointer<T>(p, false);
    }

}

inline std::pair<
    HipcEmulatedInProcessServerPort*,
    HipcEmulatedInProcessClientPort*
> HipcEmulatedInProcessPort::CreateHipcEmulatedInProcessPortPair(int maxSessions) NN_NOEXCEPT
{
    auto port = MakeSharedAttached(HipcEmulatedInProcessPort::Create(maxSessions));
    return std::make_pair(
        Factory<HipcEmulatedInProcessServerPort>::Create(port.Get()),
        Factory<HipcEmulatedInProcessClientPort>::Create(port.Get())
    );
}

std::pair<
    HipcEmulatedInProcessServerPort*,
    HipcEmulatedInProcessClientPort*
> CreateHipcEmulatedInProcessPortPair(int maxSessions) NN_NOEXCEPT
{
    return HipcEmulatedInProcessPort::CreateHipcEmulatedInProcessPortPair(maxSessions);
}

namespace detail {

void OnChildSessionClosed(ISharedObject* port) NN_NOEXCEPT
{
    static_cast<HipcEmulatedInProcessPort*>(port)->OnChildSessionClosed();
}

}

}}}
