﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/sf_HipcDirectApi.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <cstring>

namespace nn { namespace sf { namespace hipc {

namespace {

    inline void CloseHandleImpl(svc::Handle handle) NN_NOEXCEPT
    {
        auto result = svc::CloseHandle(handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    inline void* GetMessageBufferOnTlsImpl() NN_NOEXCEPT
    {
        return svc::GetThreadLocalRegion()->messageBuffer;
    }

}

void CloseClientPortHandle(HipcClientPortHandle handle) NN_NOEXCEPT
{
    CloseHandleImpl(handle);
}

Result ConnectToPort(HipcClientSessionHandle* pOut, HipcClientPortHandle port) NN_NOEXCEPT
{
    return svc::ConnectToPort(pOut, port);
}

void CloseClientSessionHandle(HipcClientSessionHandle handle) NN_NOEXCEPT
{
    CloseHandleImpl(handle);
}

void* GetMessageBufferOnTls() NN_NOEXCEPT
{
    return GetMessageBufferOnTlsImpl();
}

namespace {

    Result SendSyncRequestImpl(HipcClientSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
    {
        if (GetMessageBufferOnTlsImpl() == messageBuffer)
        {
            return svc::SendSyncRequest(session);
        }
        else
        {
            return svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(messageBuffer), messageBufferSize, session);
        }
    }

    inline Result ConvertSendRequestResult(Result result) NN_NOEXCEPT
    {
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(svc::ResultSessionClosed)
            {
                NN_RESULT_THROW(hipc::ResultSessionClosed());
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

}

Result SendSyncRequest(HipcClientSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    auto result = SendSyncRequestImpl(session, messageBuffer, messageBufferSize);
    return ConvertSendRequestResult(result);
}

Result SendAsyncRequest(HipcClientRequestHandle* pOut, HipcClientSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    auto result = svc::SendAsyncRequestWithUserBuffer(pOut, reinterpret_cast<uintptr_t>(messageBuffer), messageBufferSize, session);
    return ConvertSendRequestResult(result);
}

Result WaitClientRequestHandle(HipcClientRequestHandle request, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(messageBufferSize);
    int32_t dummy;
    auto result = svc::WaitSynchronization(&dummy, &request, 1, -1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    {
        // メッセージバッファの先頭 8B がゼロ埋めであるとき、サーバ側からのクローズであることが判定できる
        Bit64 buffer[1];
        std::memcpy(buffer, messageBuffer, 8);
        NN_RESULT_THROW_UNLESS(buffer[0] != 0, hipc::ResultSessionClosed());
    }

    NN_RESULT_SUCCESS;
}

void CloseClientRequestHandle(HipcClientRequestHandle request) NN_NOEXCEPT
{
    CloseHandleImpl(request);
}

}}}
