﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include "sf_HipcEmulatedObjectBase.h"

#include "sf_HipcEmulatedInProcessPort.h"
#include "sf_HipcEmulatedInProcessSession.h"
#include "sf_HipcEmulatedNamedPipePort.h"
#include "sf_HipcEmulatedNamedPipeSession.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <type_traits>

namespace nn { namespace sf { namespace hipc {

namespace {

mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT
{
    static std::aligned_storage<1024 * 1024>::type g_Buffer;
    NN_FUNCTION_LOCAL_STATIC(mem::StandardAllocator, g_Allocator, (&g_Buffer, sizeof(g_Buffer)));
    return g_Allocator;
}

}

void* HipcEmulatedObjectBase::StaticAllocator::Allocate(size_t size) NN_NOEXCEPT
{
    return GetStandardAllocator().Allocate(size);
}

void HipcEmulatedObjectBase::StaticAllocator::Deallocate(void* p, size_t) NN_NOEXCEPT
{
    GetStandardAllocator().Free(p);
}

Result CreatePort(HipcServerPortHandle* pOutServer, HipcClientPortHandle* pOutClient, int maxSessions) NN_NOEXCEPT
{
    auto pair = CreateHipcEmulatedNamedPipePortPair(maxSessions);
    *pOutServer = pair.first;
    *pOutClient = pair.second;
    NN_RESULT_SUCCESS;
}

Result CreateInProcessPort(HipcServerPortHandle* pOutServer, HipcClientPortHandle* pOutClient, int maxSessions) NN_NOEXCEPT
{
    auto pair = CreateHipcEmulatedInProcessPortPair(maxSessions);
    *pOutServer = pair.first;
    *pOutClient = pair.second;
    NN_RESULT_SUCCESS;
}

void CloseServerPortHandle(HipcServerPortHandle handle) NN_NOEXCEPT
{
    handle->Release();
}

void CloseClientPortHandle(HipcClientPortHandle handle) NN_NOEXCEPT
{
    handle->Release();
}

void AttachWaitHolderForAccept(nn::os::MultiWaitHolderType* pHolder, HipcServerPortHandle port) NN_NOEXCEPT
{
    port->AttachAcceptEvent(pHolder);
}

Result AcceptSession(HipcServerSessionHandle* pOut, HipcServerPortHandle port) NN_NOEXCEPT
{
    *pOut = port->Accept();
    NN_RESULT_SUCCESS;
}

void AttachWaitHolderForConnect(nn::os::MultiWaitHolderType* pHolder, HipcClientPortHandle port) NN_NOEXCEPT
{
    port->AttachConnectEvent(pHolder);
}

Result ConnectToPort(HipcClientSessionHandle* pOut, HipcClientPortHandle port, bool blocking) NN_NOEXCEPT
{
    *pOut = port->Connect(blocking);
    NN_RESULT_SUCCESS;
}

Result CreateSession(HipcServerSessionHandle* pOutServer, HipcClientSessionHandle* pOutClient) NN_NOEXCEPT
{
    auto pair = CreateHipcEmulatedNamedPipeSessionPair();
    *pOutServer = pair.first;
    *pOutClient = pair.second;
    NN_RESULT_SUCCESS;
}

Result CreateInProcessSession(HipcServerSessionHandle* pOutServer, HipcClientSessionHandle* pOutClient) NN_NOEXCEPT
{
    auto pair = CreateHipcEmulatedInProcessSessionPair();
    *pOutServer = pair.first;
    *pOutClient = pair.second;
    NN_RESULT_SUCCESS;
}

void CloseServerSessionHandle(HipcServerSessionHandle handle) NN_NOEXCEPT
{
    handle->Release();
}

void CloseClientSessionHandle(HipcClientSessionHandle handle) NN_NOEXCEPT
{
    handle->Release();
}

void AttachWaitHolderForReceive(nn::os::MultiWaitHolderType* pHolder, HipcServerSessionHandle session) NN_NOEXCEPT
{
    session->AttachReceiveEvent(pHolder);
}

namespace {
    __declspec(thread) char tls_MessageBuffer[MessageBufferSizeOnTls + 16];
    __declspec(thread) char* tls_MessageBufferHead;
}

void* GetMessageBufferOnTls() NN_NOEXCEPT
{
    auto ret = tls_MessageBufferHead;
    if (!ret)
    {
        ret = reinterpret_cast<char*>(nn::util::align_up(reinterpret_cast<uintptr_t>(tls_MessageBuffer), 16));
        tls_MessageBufferHead = ret;
    }
    return ret;
}

// ビルド互換性維持用
Result Receive(bool* pClosed, HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    return session->ReceiveRequest(pClosed, messageBuffer, messageBufferSize);
}

Result Receive(ReceiveResult* pReceiveResult, HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    bool closed;
    NN_RESULT_DO(session->ReceiveRequest(&closed, messageBuffer, messageBufferSize));
    *pReceiveResult = closed ? ReceiveResult::Close : ReceiveResult::Process;
    NN_RESULT_SUCCESS;
}

Result Reply(HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    session->Reply(messageBuffer, messageBufferSize);
    NN_RESULT_SUCCESS;
}

Result SendSyncRequest(HipcClientSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    HipcClientRequestHandle request;
    NN_RESULT_DO(SendAsyncRequest(&request, session, messageBuffer, messageBufferSize));
    NN_UTIL_SCOPE_EXIT
    {
        CloseClientRequestHandle(request);
    };
    return WaitClientRequestHandle(request, messageBuffer, messageBufferSize);
}

Result SendAsyncRequest(HipcClientRequestHandle* pOut, HipcClientSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    auto ret = session->CreateRequest(messageBuffer, messageBufferSize);
    NN_RESULT_THROW_UNLESS(ret, hipc::ResultSessionClosed());
    *pOut = ret;
    NN_RESULT_SUCCESS;
}

void AttachWaitHolderForReply(nn::os::MultiWaitHolderType* pHolder, HipcClientRequestHandle request) NN_NOEXCEPT
{
    HipcEmulatedClientSession::AttachReplyEvent(request, pHolder);
}

Result WaitClientRequestHandle(HipcClientRequestHandle request, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(messageBuffer);
    NN_UNUSED(messageBufferSize);
    auto success = HipcEmulatedClientSession::WaitRequest(request);
    NN_RESULT_THROW_UNLESS(success, hipc::ResultSessionClosed());
    NN_RESULT_SUCCESS;
}

void CloseClientRequestHandle(HipcClientRequestHandle request) NN_NOEXCEPT
{
    HipcEmulatedClientSession::CloseRequest(request);
}

}}}
