﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/server/sf_HipcServerSessionManager.h>
#include <nn/sf/sf_HipcServerSessionManagerHandler.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <utility>
#include <atomic>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/sf/hipc/detail/sf_HipcMessageFormat.h>
#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include <nn/sf/hipc/detail/sf_HipcHandleRegistration.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor2.h>
#include <nn/util/util_BitUtil.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <nn/sf/detail/sf_InternalResult.h>
#include <nn/os/os_ThreadLocalStorageApi.h>
#include <nn/os/os_SdkThreadLocalStorageApi.h>

#include "sf_HipcServerMessage.h"

namespace nn { namespace sf {

namespace {

bool g_InitializedForHipcServerSessionManagerHandler = false;
os::TlsSlot g_HipcServerSessionManagerClientId;

}

void InitializeForHipcServerSessionManagerHandler() NN_NOEXCEPT
{
    if (g_InitializedForHipcServerSessionManagerHandler)
    {
        return;
    }
    os::SdkAllocateTlsSlot(&g_HipcServerSessionManagerClientId, nullptr);
    g_InitializedForHipcServerSessionManagerHandler = true;
}

namespace {

void SetCurrentClientInfo(const hipc::server::ClientInfo& clientInfo) NN_NOEXCEPT
{
    if (!g_InitializedForHipcServerSessionManagerHandler)
    {
        return;
    }
    os::SetTlsValue(g_HipcServerSessionManagerClientId, clientInfo.clientId);
}

hipc::server::ClientInfo GetCurrentClientInfo() NN_NOEXCEPT
{
    if (!g_InitializedForHipcServerSessionManagerHandler)
    {
        return {};
    }
    return {os::GetTlsValue(g_HipcServerSessionManagerClientId)};
}

}

Bit64 GetCurrentHipcClientIdOnServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InitializedForHipcServerSessionManagerHandler);
    auto info = GetCurrentClientInfo();
    NN_SDK_REQUIRES(info.IsValid());
    return info.clientId;
}

namespace hipc { namespace server { inline namespace v1 {

ClientInfo ClientInfo::NewClientInfo() NN_NOEXCEPT
{
    static std::atomic<uintptr_t> g_CurrentClientId{1};
    return {g_CurrentClientId.fetch_add(1)};
}

Result HipcServerSessionBase::Reply(void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    return hipc::Reply(m_ServerHandle, messageBuffer, messageBufferSize);
}

Result HipcServerSessionManagerBase::RegisterBase(HipcServerSessionBase* p, HipcServerSessionHandle serverSessionHandle) NN_NOEXCEPT
{
    hipc::AttachWaitHolderForReceive(p, serverSessionHandle);
    p->m_Closed = false;
    p->m_ServerHandle = serverSessionHandle;
    RegisterServerSessionToWaitBase(p);
    NN_RESULT_SUCCESS;
}

Result HipcServerSessionManagerBase::AcceptBase(HipcServerSessionBase* p, HipcServerPortHandle serverPortHandle) NN_NOEXCEPT
{
    HipcServerSessionHandle serverSessionHandle;
    NN_RESULT_DO(hipc::AcceptSession(&serverSessionHandle, serverPortHandle));
    auto success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            hipc::CloseServerSessionHandle(serverSessionHandle);
        }
    };
    NN_RESULT_DO(RegisterBase(p, serverSessionHandle));
    success = true;
    NN_RESULT_SUCCESS;
}

Result HipcServerSessionManagerBase::ReceiveRequestBase(HipcServerSessionBase* p, void* messageBuffer, size_t messageBufferSize, void* pointerBuffer, size_t pointerBufferSize) NN_NOEXCEPT
{
retry:
    if (pointerBuffer)
    {
        detail::HipcMessageHeaderInfo headerInfo = {};
        headerInfo.receiveBufferMode = detail::HipcMessageReceiveBufferMode_Single;
        headerInfo.receiveListCount = 1;
        detail::HipcMessageBufferAccessor message;
        message.SetupHeader(messageBuffer, messageBufferSize, headerInfo);
        message.SetReceiveList(0, hipc::detail::RegisterAddress(reinterpret_cast<uintptr_t>(pointerBuffer)), pointerBufferSize);
    }
    else
    {
        detail::HipcMessageHeaderInfo headerInfo = {};
        headerInfo.receiveBufferMode = detail::HipcMessageReceiveBufferMode_MessageBuffer;
        headerInfo.receiveListCount = 0;
        detail::HipcMessageBufferAccessor message;
        message.SetupHeader(messageBuffer, messageBufferSize, headerInfo);
    }
    hipc::ReceiveResult receiveResult;
    NN_RESULT_DO(hipc::Receive(&receiveResult, p->m_ServerHandle, messageBuffer, messageBufferSize));
    switch (receiveResult)
    {
        case nn::sf::hipc::ReceiveResult::Process:
        {
            p->m_Closed = false;
            NN_RESULT_SUCCESS;
        }
        case nn::sf::hipc::ReceiveResult::Close:
        {
            p->m_Closed = true;
            NN_RESULT_SUCCESS;
        }
        case nn::sf::hipc::ReceiveResult::Retry:
        {
            goto retry;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

Result HipcServerSessionManagerBase::ProcessRequest(HipcServerSessionBase* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    if (p->m_Closed)
    {
        CloseSessionBase(p);
        NN_RESULT_SUCCESS;
    }
    auto tag = detail::HipcMessageReader::ReadTag(messageBuffer);
    switch (tag)
    {
        case detail::MessageType_Release:
        {
            CloseSessionBase(p);
            NN_RESULT_SUCCESS;
        }
        default:
        {
            NN_RESULT_TRY(ProcessMessage2(p, messageBuffer, messageBufferSize, messageBuffer, messageBufferSize))
                NN_RESULT_CATCH(nn::sf::detail::ResultContextControl)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    CloseSessionBase(p);
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY
            RegisterServerSessionToWaitBase(p);
            NN_RESULT_SUCCESS;
        }
    };
}

void HipcServerSessionManagerBase::CloseSessionBase(HipcServerSessionBase* p) NN_NOEXCEPT
{
    auto serverHandle = p->m_ServerHandle;
    os::FinalizeMultiWaitHolder(p);
    DestroyServerSession(p);
    hipc::CloseServerSessionHandle(serverHandle);
}

namespace {

class HipcObjectDomain final
    : public IHipcObjectDomain
{
private:

    HipcServerSessionManager* m_pManager;

public:

    explicit HipcObjectDomain(HipcServerSessionManager* pManager) NN_NOEXCEPT
        : m_pManager(pManager)
    {
    }

    virtual Result Register(HipcServerSessionHandle handle, cmif::server::CmifServerObjectInfo&& x) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pManager->Register(handle, std::move(x), GetCurrentClientInfo());
    }
};

}

void HipcServerSessionManager::SetManagerHandler(IHipcServerSessionManagerHandler* pManagerHandler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_InitializedForHipcServerSessionManagerHandler, "nn::sf::InitializeForHipcServerSessionManagerHandler() is not called.");
    this->m_pManagerHandler = pManagerHandler;
}

inline IHipcServerSessionManagerHandler* HipcServerSessionManager::GetHipcServerSessionManagerHandler() const NN_NOEXCEPT
{
    return m_pManagerHandler;
}

Result HipcServerSessionManager::Register(HipcServerSessionHandle serverSessionHandle, cmif::server::CmifServerObjectInfo&& x, const ClientInfo& clientInfo) NN_NOEXCEPT
{
    HipcServerSessionBase* pSession;
    return Base::Register(&pSession, serverSessionHandle, std::move(x), clientInfo);
}

Result HipcServerSessionManager::Accept(HipcServerPortHandle serverPortHandle, cmif::server::CmifServerObjectInfo&& x) NN_NOEXCEPT
{
    HipcServerSessionBase* pSession;
    return Base::Accept(&pSession, serverPortHandle, std::move(x), ClientInfo::NewClientInfo());
}

Result HipcServerSessionManager::Process2InvokeMethodImpl(cmif::server::CmifServerObjectInfo&& targetObject, HipcServerSession* p, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT
{
    detail::HipcMessageReader reader;
    reader.Initialize(inMessageBuffer);
    NN_RESULT_THROW_UNLESS(reader.GetMessageByteSize() <= inMessageBufferSize, sf::hipc::ResultInvalidHipcRequestMessageSize());
    HipcObjectDomain domain(this);
    Hipc2ServerMessage serverMessage(&domain, reader, p->m_PointerBuffer, p->m_PointerBufferSize, outMessageBuffer, outMessageBufferSize, HipcServerApiModelHolder::GetHipcServerApiModel());
    {
        auto pHandler = GetHipcServerSessionManagerHandler();
        if (pHandler)
        {
            SetCurrentClientInfo(p->m_ClientInfo);
            NN_RESULT_DO(pHandler->BeforeInvoke());
        }
        NN_UTIL_SCOPE_EXIT
        {
            if (pHandler)
            {
                pHandler->AfterInvoke();
                SetCurrentClientInfo({});
            }
        };
        cmif::PointerAndSize inRawData;
        inRawData.pointer = nn::util::align_up(reader.GetRawPointer(), 16);
        inRawData.size = reader.GetRawSize() - 16;
        NN_RESULT_DO(targetObject.ProcessServerMessage(&serverMessage, inRawData));
    }
    {
        NN_UTIL_SCOPE_EXIT
        {
            serverMessage.AfterReply();
        };
        NN_RESULT_DO(p->Reply(outMessageBuffer, outMessageBufferSize));
    }
    NN_RESULT_SUCCESS;
}

Result HipcServerSessionManager::Process2ManagerInvoke(HipcServerSession*, void*, size_t, void*, size_t) NN_NOEXCEPT
{
    NN_RESULT_THROW(ResultNotSupported());
}

Result HipcServerSessionManager::ProcessMessage2(HipcServerSessionBase* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT
{
    auto p = static_cast<HipcServerSession*>(pSession);
    auto tag = detail::HipcMessageReader::ReadTag(inMessageBuffer);
    auto getInlineContext = [&]() -> cmif::InlineContext
    {
        switch (tag)
        {
            case detail::MessageType_Invoke2Method:
            case detail::MessageType_Invoke2ManagerMethod:
            {
                auto contextPointer = static_cast<const char*>(detail::HipcMessageReader::GetRawDataPointer(inMessageBuffer)) + 12;
                cmif::InlineContext ret = {};
                std::memcpy(&ret, contextPointer, sizeof(ret));
                return ret;
            }
            default: return {};
        }
    };
    cmif::ScopedInlineContextChanger sicc{getInlineContext()};
    switch (tag)
    {
        case detail::MessageType_Invoke2MethodOld:
        case detail::MessageType_Invoke2Method:
        {
            return Process2InvokeMethodImpl(p->m_X.Clone(), p, inMessageBuffer, inMessageBufferSize, outMessageBuffer, outMessageBufferSize);
        }
        case detail::MessageType_Invoke2ManagerMethodOld:
        case detail::MessageType_Invoke2ManagerMethod:
        {
            return Process2ManagerInvoke(p, inMessageBuffer, inMessageBufferSize, outMessageBuffer, outMessageBufferSize);
        }
        default:
        {
            NN_RESULT_THROW(sf::hipc::ResultUnknownHipcRequestKind());
        }
    }
}

HipcServerSession* HipcServerSessionManager::CreateServerSession(cmif::server::CmifServerObjectInfo&& objectInfo, const ClientInfo& clientInfo) NN_NOEXCEPT
{
    auto ret = AllocateServerSession();
    if (!ret)
    {
        return nullptr;
    }
    NN_SDK_ASSERT_NOT_NULL(objectInfo);
    ret->m_X = std::move(objectInfo);
    ret->m_PointerBuffer = AllocatePointerTransferBuffer(&ret->m_PointerBufferSize, ret);
    ret->m_ClientInfo = clientInfo;
    if (auto pHandler = GetHipcServerSessionManagerHandler())
    {
        NN_SDK_ASSERT(clientInfo.IsValid());
        pHandler->OnAddClient(clientInfo.clientId);
    }
    return ret;
}


void HipcServerSessionManager::DestroyServerSession(HipcServerSessionBase* pSession) NN_NOEXCEPT
{
    auto p = static_cast<HipcServerSession*>(pSession);
    p->m_X.Reset();
    if (auto pHandler = GetHipcServerSessionManagerHandler())
    {
        NN_SDK_ASSERT(p->m_ClientInfo.IsValid());
        pHandler->OnRemoveClient(p->m_ClientInfo.clientId);
    }
    p->m_ClientInfo = {};
    DeallocatePointerTransferBuffer(p, p->m_PointerBuffer, p->m_PointerBufferSize);
    DeallocateServerSession(p);
}

}}}}}
