﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <mutex>
#include <nn/sf/hipc/detail/sf_HipcHandleRegistration.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_Windows.h>

#include "sf_HipcHandleRegistrationInternal.h"
#include "../sf_HipcEmulatedSession.h"

namespace nn { namespace sf { namespace hipc { namespace detail {

namespace
{

class KeyUtil
{
    typedef nn::util::BitPack32::Field<0,           24, int> Index;
    typedef nn::util::BitPack32::Field<Index::Next, 8,  int> Tag;

public:
    static const int MaxIndexValue = (1 << 24) - 1;
    static const int MaxTagValue = (1 << 8) - 1;

    template <int TagValue>
    static InternalHandleValue Generate(int index) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(TagValue != 0);
        nn::util::BitPack32 bits = {};
        bits.Set<Index>(index);
        bits.Set<Tag>(TagValue);

        InternalHandleValue v = {bits.storage};
        return v;
    }
    static uint32_t ExtractTag(InternalHandleValue key) NN_NOEXCEPT
    {
        nn::util::BitPack32 bits = {key.value};
        return bits.Get<Tag>();
    }
    static uint32_t ExtractIndex(InternalHandleValue key) NN_NOEXCEPT
    {
        nn::util::BitPack32 bits = {key.value};
        return bits.Get<Index>();
    }
};


template <typename T, int Capacity, int Tag>
class HandleStorage
{
    static_assert(Capacity <= KeyUtil::MaxIndexValue, "Capacity range");
    static_assert(Tag <= KeyUtil::MaxTagValue, "Tag range");

private:
    struct HandleStore
    {
        bool isUsed;
        T handle;

        HandleStore() NN_NOEXCEPT
            : isUsed(false)
        {
        }
        void Attach(T target) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(!isUsed, "[SF-Internal]");
            isUsed = true;
            this->handle = target;
        }
        T Detach() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(isUsed, "[SF-Internal]");
            isUsed = false;
            return this->handle;
        }
        NN_IMPLICIT operator bool() const NN_NOEXCEPT
        {
            return isUsed;
        }
    };

    HandleStore m_Storage[Capacity];
    mutable os::Mutex m_Mutex;

    InternalHandleValue StoreImpl(T handle) NN_NOEXCEPT
    {
        for (auto i = 0; i < Capacity; i++)
        {
            auto& storage = m_Storage[i];
            if (!storage)
            {
                storage.Attach(handle);
                return KeyUtil::Generate<Tag>(i);
            }
        }
        NN_ABORT("[SF-Internal] No space to store NativeHandle");
    }

public:
    HandleStorage() NN_NOEXCEPT
        : m_Mutex(false)
    {
    }

    InternalHandleValue Store(T handle) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);

        return StoreImpl(handle);
    }

    T Load(InternalHandleValue key) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(KeyUtil::ExtractTag(key) == Tag);
        std::lock_guard<os::Mutex> lock(m_Mutex);

        auto index = KeyUtil::ExtractIndex(key);
        NN_SDK_ASSERT(index < Capacity && m_Storage[index], "[SF-Internal]");
        return m_Storage[index].Detach();
    }

    bool IsAcceptableKey(InternalHandleValue key) NN_NOEXCEPT
    {
        return KeyUtil::ExtractTag(key) == Tag;
    }
};

// ../sf_HipcServiceResolutionApi-os.win32.cpp からのコピー
template <typename T>
struct Singleton
{
    nn::os::MutexType _mutex;
    typename std::aligned_storage<sizeof(T), NN_ALIGNOF(T)>::type _storage;
    T* _p;

    T& Get()
    {
        nn::os::LockMutex(&_mutex);
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::UnlockMutex(&_mutex);
        };
        if (!_p)
        {
            this->_p = new (&_storage) T;
        }
        return *_p;
    }
};

enum HandleTag : uint8_t
{
    HandleTag_Os = 1,
    HandleTag_Session = 2,
    HandleTag_Address = 3,
};

Singleton<HandleStorage<os::NativeHandle, 1024, HandleTag_Os>> g_OsHandleStorage = { NN_OS_MUTEX_INITIALIZER(false) };
Singleton<HandleStorage<HipcClientSessionHandle, 1024, HandleTag_Session>> g_SessionHandleStorage = { NN_OS_MUTEX_INITIALIZER(false) };
Singleton<HandleStorage<uintptr_t, 1024, HandleTag_Address>> g_AddressStorage = { NN_OS_MUTEX_INITIALIZER(false) };

} // ~ anonymous

InternalHandleValue RegisterOsHandle(os::NativeHandle handle) NN_NOEXCEPT
{
    // TODO: 手動でセッションハンドルが通るシーケンスがあるため、もうしばらくの間、ハンドルの型チェックを行う
    // 本来は return g_OsHandleStorage.Get().Store(handle); のみ
    DWORD dummy;
    if (::GetHandleInformation(handle, &dummy))
    {
        return g_OsHandleStorage.Get().Store(handle);
    }
    else
    {
        return g_SessionHandleStorage.Get().Store(reinterpret_cast<const HipcClientSessionHandle&>(handle));
    }
}
os::NativeHandle UnregisterOsHandle(InternalHandleValue key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(key != InvalidInternalHandleValue, "");

    // TODO: 手動でセッションハンドルが通るシーケンスがあるため、もうしばらくの間、ハンドルの型チェックを行う
    // 本来は return g_OsHandleStorage.Get().Load(key); のみ
    switch (KeyUtil::ExtractTag(key))
    {
        case HandleTag_Os:
            return g_OsHandleStorage.Get().Load(key);
        case HandleTag_Session:
            return g_SessionHandleStorage.Get().Load(key);
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}
InternalHandleValue RegisterSessionHandle(HipcClientSessionHandle handle) NN_NOEXCEPT
{
    return g_SessionHandleStorage.Get().Store(handle);
}
HipcClientSessionHandle UnregisterSessionHandle(InternalHandleValue key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(key != InvalidInternalHandleValue, "");
    return g_SessionHandleStorage.Get().Load(key);
}
InternalHandleType GetHandleType(InternalHandleValue key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(key != InvalidInternalHandleValue, "");
    switch (KeyUtil::ExtractTag(key))
    {
    case HandleTag_Os:
        return InternalHandleType_OsHandle;
    case HandleTag_Session:
        return InternalHandleType_SessionHandle;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
InternalHandleValue DuplicateHandle(InternalHandleValue key, bool move) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(key != InvalidInternalHandleValue, "");
    switch (KeyUtil::ExtractTag(key))
    {
    case HandleTag_Os:
        {
            auto& storage = g_OsHandleStorage.Get();
            auto handle = storage.Load(key);

            HANDLE dupHandle = os::InvalidNativeHandle;
            if (handle != os::InvalidNativeHandle)
            {
                // Windows が認識できるハンドル
                ::DuplicateHandle(
                    ::GetCurrentProcess(),
                    handle,
                    ::GetCurrentProcess(),
                    &dupHandle,
                    0,
                    FALSE,
                    DUPLICATE_SAME_ACCESS);

                if (move)
                {
                    ::CloseHandle(handle);
                }
            }
            return storage.Store(dupHandle);
        }

    case HandleTag_Session:
        {
            auto& storage = g_SessionHandleStorage.Get();
            auto handle = storage.Load(key);

            if (handle)
            {
                if (!move)
                {
                    handle->AddReference();
                }
            }
            return storage.Store(handle);
        }

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void DisposeHandle(InternalHandleValue key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(key != InvalidInternalHandleValue, "");
    switch (KeyUtil::ExtractTag(key))
    {
    case HandleTag_Os:
        {
            auto& storage = g_OsHandleStorage.Get();
            auto handle = storage.Load(key);
            if (handle != os::InvalidNativeHandle)
            {
                ::CloseHandle(handle);
            }
            return;
        }

    case HandleTag_Session:
        {
            auto& storage = g_SessionHandleStorage.Get();
            auto handle = storage.Load(key);
            if (handle)
            {
                handle->Release();
            }
            return;
        }

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

InternalAddressValue RegisterAddress(uintptr_t address) NN_NOEXCEPT
{
    if (address == 0)
    {
        return 0;
    }
    auto& storage = g_AddressStorage.Get();
    auto internalValue = storage.Store(address);
    return static_cast<InternalAddressValue>(internalValue.value);
}

uintptr_t UnregisterAddress(InternalAddressValue value) NN_NOEXCEPT
{
    if (value == 0)
    {
        return 0;
    }
    InternalHandleValue v = { static_cast<nn::Bit32>(value) };
    auto& storage = g_AddressStorage.Get();
    auto ret = storage.Load(v);
    return ret;
}

}}}}
