﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Common.h>
#if (defined(NN_DETAIL_SDMMC_USE_PCV_FOR_CLOCK_RESET_CONTROLLER))
    #include "detail/sdmmc_ClockResetController.tegra.h"
#endif
#include "detail/sdmmc_IHostController.h"
#include "detail/sdmmc_IDeviceAccessor.h"
#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    #include "detail/sdmmc_PortMmc0.h"
#endif
#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    #include "detail/sdmmc_PortSdCard0.h"
#endif
#if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
    #include "detail/sdmmc_PortGcAsic0.h"
#endif
#include <nn/nn_Abort.h>
#include <nn/dd.h>

namespace nn { namespace sdmmc {

namespace
{
    detail::IHostController* GetHostController(Port port) NN_NOEXCEPT
    {
        detail::IHostController* pHostController = nullptr;
        switch (port)
        {
        #if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
            case Port_Mmc0:
                pHostController = detail::GetHostControllerOfPortMmc0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
            case Port_SdCard0:
                pHostController = detail::GetHostControllerOfPortSdCard0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
            case Port_GcAsic0:
                pHostController = detail::GetHostControllerOfPortGcAsic0();
                break;
        #endif
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS_NOT_NULL(pHostController);
        return pHostController;
    }

    detail::IDeviceAccessor* GetDeviceAccessor(Port port) NN_NOEXCEPT
    {
        detail::IDeviceAccessor* pDeviceAccessor = nullptr;
        switch (port)
        {
        #if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
            case Port_Mmc0:
                pDeviceAccessor = detail::GetDeviceAccessorOfPortMmc0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
            case Port_SdCard0:
                pDeviceAccessor = detail::GetDeviceAccessorOfPortSdCard0();
                break;
        #endif
        #if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
            case Port_GcAsic0:
                pDeviceAccessor = detail::GetDeviceAccessorOfPortGcAsic0();
                break;
        #endif
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT_UNLESS_NOT_NULL(pDeviceAccessor);
        return pDeviceAccessor;
    }
}

void SwitchToPcvClockResetControl() NN_NOEXCEPT
{
#if (defined(NN_DETAIL_SDMMC_USE_PCV_FOR_CLOCK_RESET_CONTROLLER))
    detail::ClockResetController::SwitchToPcvControl();
#else
    NN_ABORT("NN_DETAIL_SDMMC_USE_PCV_FOR_CLOCK_RESET_CONTROLLER isn't defined.\n");
#endif
}

void Initialize(Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    pDeviceAccessor->Initialize();
}

void Finalize(Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    pDeviceAccessor->Finalize();
}

void RegisterDeviceVirtualAddress(Port port, uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
        pDeviceAccessor->RegisterDeviceVirtualAddress(bufferAddress, bufferSize, bufferDeviceVirtualAddress);
    #else
        NN_UNUSED(port);
        NN_UNUSED(bufferAddress);
        NN_UNUSED(bufferSize);
        NN_UNUSED(bufferDeviceVirtualAddress);
        NN_ABORT("NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED isn't defined.\n");
    #endif
}

void UnregisterDeviceVirtualAddress(Port port, uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
        pDeviceAccessor->UnregisterDeviceVirtualAddress(bufferAddress, bufferSize, bufferDeviceVirtualAddress);
    #else
        NN_UNUSED(port);
        NN_UNUSED(bufferAddress);
        NN_UNUSED(bufferSize);
        NN_UNUSED(bufferDeviceVirtualAddress);
        NN_ABORT("NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED isn't defined.\n");
    #endif
}

#if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
    void SetWorkBufferForHostController(Port port, void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        detail::IHostController* pHostController = GetHostController(port);
        pHostController->SetWorkBuffer(pWorkBuffer, workBufferSize);
    }
#endif

void ChangeCheckTransferInterval(Port port, uint32_t milliSeconds) NN_NOEXCEPT
{
    detail::IHostController* pHostController = GetHostController(port);
    pHostController->ChangeCheckTransferInterval(milliSeconds);
}

void SetDefaultCheckTransferInterval(Port port) NN_NOEXCEPT
{
    detail::IHostController* pHostController = GetHostController(port);
    pHostController->SetDefaultCheckTransferInterval();
}

Result Activate(Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->Activate();
}

void Deactivate(Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    pDeviceAccessor->Deactivate();
}

Result Read(void* pOutDataBuffer, size_t dataBufferSize, Port port, uint32_t sectorIndex, uint32_t numSectors) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->ReadWrite(sectorIndex, numSectors, pOutDataBuffer, dataBufferSize, true);
}

Result Write(Port port, uint32_t sectorIndex, uint32_t numSectors, const void* pDataBuffer, size_t dataBufferSize) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->ReadWrite(sectorIndex, numSectors, const_cast<void*>(pDataBuffer), dataBufferSize, false);
}

Result CheckConnection(SpeedMode* pOutHostSpeedMode, BusWidth* pOutHostBusWidth, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->CheckConnection(pOutHostSpeedMode, pOutHostBusWidth);
}

void GetHostBusStatus(HostBusStatus* pOutHostBusStatus, Port port) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHostBusStatus);
    detail::IHostController* pHostController = GetHostController(port);
    pOutHostBusStatus->busPower = pHostController->GetBusPower();
    pOutHostBusStatus->busWidth = pHostController->GetBusWidth();
    pOutHostBusStatus->deviceClockFrequencyKHz = pHostController->GetDeviceClockFrequencyKHz();;
}

Result GetDeviceSpeedMode(SpeedMode* pOutSpeedMode, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetSpeedMode(pOutSpeedMode);
}

Result GetDeviceMemoryCapacity(uint32_t* pOutNumSectors, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetMemoryCapacity(pOutNumSectors);
}

Result GetDeviceStatus(uint32_t* pOutDeviceStatus, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetDeviceStatus(pOutDeviceStatus);
}

Result GetDeviceOcr(uint32_t* pOutOcr, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetOcr(pOutOcr);
}

Result GetDeviceRca(uint16_t* pOutRca, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetRca(pOutRca);
}

Result GetDeviceCid(void* pOutCid, size_t cidSize, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetCid(pOutCid, cidSize);
}

Result GetDeviceCsd(void* pOutCsd, size_t csdSize, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetCsd(pOutCsd, csdSize);
}

void SetAbortCommandForDebug(Port port, uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        detail::IHostController* pHostController = GetHostController(port);
        return pHostController->SetAbortCommandForDebug(commandIndex, commandArgument, abortTimingNanoSeconds);
    #else
        NN_UNUSED(port);
        NN_UNUSED(commandIndex);
        NN_UNUSED(commandArgument);
        NN_UNUSED(abortTimingNanoSeconds);
        NN_ABORT("NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG isn't defined.\n");
    #endif
}

void ClearAbortCommandForDebug(Port port) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        detail::IHostController* pHostController = GetHostController(port);
        return pHostController->ClearAbortCommandForDebug();
    #else
        NN_UNUSED(port);
        NN_ABORT("NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG isn't defined.\n");
    #endif
}

Result IssueCommandForDebug(uint32_t* pOutResponse, Port port, uint32_t commandIndex, uint32_t commandArgument,
    DataTransfer* pDataTransfer, bool isBusy) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG))
        detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
        return pDeviceAccessor->IssueCommandForDebug(pOutResponse, commandIndex, commandArgument, pDataTransfer, isBusy);
    #else
        NN_UNUSED(pOutResponse);
        NN_UNUSED(port);
        NN_UNUSED(commandIndex);
        NN_UNUSED(commandArgument);
        NN_UNUSED(pDataTransfer);
        NN_UNUSED(isBusy);
        NN_ABORT("NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG isn't defined.\n");
        return ResultNotSupported();
    #endif
}

void GetAndClearErrorInfo(ErrorInfo* pOutErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize, Port port) NN_NOEXCEPT
{
    detail::IDeviceAccessor* pDeviceAccessor = GetDeviceAccessor(port);
    return pDeviceAccessor->GetAndClearErrorInfo(pOutErrorInfo, pOutLogSize, pOutLogBuffer, logBufferSize);
}

}} // namespace nn { namespace sdmmc {
