﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_SdmmcController.tegra-k1.h"
#include <nn/result/result_HandlingUtility.h>
#include "sdmmc_Log.h"

namespace nn { namespace sdmmc {
namespace detail {

namespace
{
    // レジスタログ
    #define NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(basePointer, memberVariable) NN_DETAIL_SDMMC_DEBUG_LOG(#memberVariable "(0x%03x) 0x%08X\n",\
        static_cast<uint32_t>(reinterpret_cast<uintptr_t>(&(basePointer->memberVariable)) - reinterpret_cast<uintptr_t>(basePointer)), (basePointer->memberVariable))

    void GetDividerSetting(uint32_t* pOutTargetSourceClockFrequencyKHz, uint16_t* pOutX, SpeedMode speedMode) NN_NOEXCEPT
    {
        // Divided by "X = 1 / (2 * N)", 1 分周より大きいものは 2 の倍数
        switch (speedMode)
        {
        case SpeedMode_MmcIdentification:
            // 400kHz 以下、26MHz 以下を 66 分周（設定削減のため、Legacy Speed と同じクロックソースを選択）
            *pOutTargetSourceClockFrequencyKHz = 26000;
            *pOutX = 66;
            return;
        case SpeedMode_SdCardIdentification:
            // 400kHz 以下、25MHz 以下を 64 分周（設定削減のため、SDR12, Default Speed と同じクロックソースを選択）
            *pOutTargetSourceClockFrequencyKHz = 25000;
            *pOutX = 64;
            return;
        case SpeedMode_SdCardDefaultSpeed:
            // 25MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 25000;
            *pOutX = 1;
            return;
        case SpeedMode_MmcLegacySpeed:
            // 26MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 26000;
            *pOutX = 1;
            return;
        case SpeedMode_SdCardHighSpeed:
            // 50MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 50000;
            *pOutX = 1;
            return;
        case SpeedMode_MmcHighSpeed:
            // 52MHz 以下を 1 分周
            *pOutTargetSourceClockFrequencyKHz = 52000;
            *pOutX = 1;
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void SdmmcController::ReleaseReset(SpeedMode speedMode) NN_NOEXCEPT
{
    detail::ClockResetController::Module module = GetClockResetModule();
    if (detail::ClockResetController::IsAvailable(module))
    {
        // クロックソース変更によるグリッジを防ぐためにクロック供給を止める
        SdHostStandardController::DisableDeviceClock();
        SdHostStandardController::EnsureControl();
    }

    uint32_t targetSourceClockFrequencyKHz;
    uint16_t x;
    GetDividerSetting(&targetSourceClockFrequencyKHz, &x, speedMode);
    detail::ClockResetController::ReleaseReset(module, targetSourceClockFrequencyKHz);
}

void SdmmcController::AssertReset() NN_NOEXCEPT
{
    detail::ClockResetController::AssertReset(GetClockResetModule());
}

void SdmmcController::LogRegisters() NN_NOEXCEPT
{
    SdHostStandardController::LogRegisters();

    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorClockCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorSysSwCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorCapOverrides0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorBootCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorBootActTimeout0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, bootDatTimeout0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorDebounceCount0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorMiscCntrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, maxCurrentOverride0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, maxCurrentOverrideHi0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorClkGateHysteresisCount0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorPresetVal00);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorPresetVal10);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, vendorPresetVal20);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, sdmemcomppadctrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, autoCalConfig0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, autoCalInterval0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, autoCalStatus0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, mmcifFifoctrl0);
    NN_DETAIL_SDMMC_DEBUG_LOG_32BIT_REGISTER(m_pSdmmcRegisters, timeoutWcalSdmmc0);
}

Result SdmmcController::Startup(BusPower busPower, BusWidth busWidth, SpeedMode speedMode, bool isPowerSavingEnable) NN_NOEXCEPT
{
    ReleaseReset(speedMode);

    // ここまで実行したら Shutdown() が可能
    m_IsShutdowned = false;

    NN_RESULT_DO(SdHostStandardController::EnableInternalClock());

    SdHostStandardController::SetBusWidth(busWidth);
    SdHostStandardController::SetBusPower(busPower);
    NN_RESULT_DO(SetSpeedMode(speedMode));
    SdHostStandardController::SetPowerSaving(isPowerSavingEnable);
    SdHostStandardController::EnableDeviceClock();
    SdHostStandardController::EnsureControl();

    NN_RESULT_SUCCESS;
}

void SdmmcController::Shutdown() NN_NOEXCEPT
{
    if (m_IsShutdowned)
    {
        return;
    }
    SdHostStandardController::DisableDeviceClock();
    SdHostStandardController::SetBusPower(BusPower_Off);
    SdHostStandardController::EnsureControl();
    AssertReset();
    m_IsShutdowned = true;
}

Result SdmmcController::SetSpeedMode(SpeedMode speedMode) NN_NOEXCEPT
{
    bool isClockEnabled = false;

    if ((m_pSdmmcRegisters->sdHostStandardRegisters.clockControl & REG_CLOCK_CONTROL__SD_CLOCK_ENABLE) != 0)
    {
        isClockEnabled = true;

        // クロックソースを変更するため、一時的にクロックを停止する
        m_pSdmmcRegisters->sdHostStandardRegisters.clockControl &= (~REG_CLOCK_CONTROL__SD_CLOCK_ENABLE);
    }

    switch (speedMode)
    {
    case SpeedMode_MmcIdentification:
        NN_FALL_THROUGH;
    case SpeedMode_SdCardIdentification:
        m_pSdmmcRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        break;
    case SpeedMode_SdCardDefaultSpeed:
        m_pSdmmcRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        break;
    case SpeedMode_MmcLegacySpeed:
        m_pSdmmcRegisters->sdHostStandardRegisters.hostControl1 &= (~REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE);
        break;
    case SpeedMode_SdCardHighSpeed:
        m_pSdmmcRegisters->sdHostStandardRegisters.hostControl1 |= REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE;
        break;
    case SpeedMode_MmcHighSpeed:
        m_pSdmmcRegisters->sdHostStandardRegisters.hostControl1 |= REG_HOST_CONTROL_1__HIGH_SPEED_ENABLE;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    SdHostStandardController::EnsureControl();

    // 分周設定の取得
    uint32_t targetSourceClockFrequencyKHz;
    uint16_t x; // Divided by "X = 1 / (2 * N)", 1 分周より大きいものは 2 の倍数
    GetDividerSetting(&targetSourceClockFrequencyKHz, &x, speedMode);

    // ソースクロックの設定
    uint32_t actualSourceClockFrequencyKHz;
    detail::ClockResetController::SetClockFrequencyKHz(&actualSourceClockFrequencyKHz, GetClockResetModule(), targetSourceClockFrequencyKHz);

    // デバイスクロックの算出（Wait 時間算出に使用するため、繰り上げ計算）
    uint32_t actualDeviceClockFrequencyKHz = (actualSourceClockFrequencyKHz + (x - 1)) / x;
    SdHostStandardController::SetDeviceClockFrequencyKHz(actualDeviceClockFrequencyKHz);
    NN_DETAIL_SDMMC_DEBUG_LOG("Device Clock: %u (kHz)\n", actualDeviceClockFrequencyKHz);

    // SDMMC での分周設定
    NN_ABORT_UNLESS((x == 1) || ((x & 0x1) == 0));
    uint16_t n = x / 2;     // N = 1 / (2 * X), 0: Base Clock
    uint16_t upper_n = 0;
    if (n > 0xFF)
    {
        upper_n = n >> 8;
    }
    uint16_t mask = REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__MASK | REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__MASK;
    m_pSdmmcRegisters->sdHostStandardRegisters.clockControl = (m_pSdmmcRegisters->sdHostStandardRegisters.clockControl & (~mask)) | (n << REG_CLOCK_CONTROL__SDCLK_FREQUENCY_SELECT__SHIFT) | (upper_n << REG_CLOCK_CONTROL__UPPER_BITS_OF_SDCLK_FREQUENCY_SELECT__SHIFT);

    if (isClockEnabled)
    {
        m_pSdmmcRegisters->sdHostStandardRegisters.clockControl |= REG_CLOCK_CONTROL__SD_CLOCK_ENABLE;
    }

    NN_RESULT_SUCCESS;
}

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc3Controller::m_InterruptEvent;
    #endif
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::InterruptEventType Sdmmc4Controller::m_InterruptEvent;
    #endif
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc {
