﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_ClockResetController.tegra.pcv.h"
#include <nn/nn_Abort.h>
#include "sdmmc_Timer.h"
#include "sdmmc_Log.h"
#include <cstring>
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif
#include <nn/pcv/pcv.h>

namespace nn { namespace sdmmc {
namespace detail {

namespace ClockResetController {
namespace pcv {

namespace
{
    #if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
        nn::os::Mutex g_Mutex(false);
        #define NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(g_Mutex)
    #else
        #define NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD
    #endif

    bool g_IsInitialized = false;
    bool g_IsUsed[Module_Num];

    struct ModuleInfo
    {
        uint32_t targetClockFrequencyKHz;
        uint32_t actualClockFrequencyKHz;
    };

    ModuleInfo g_ModuleInfos[Module_Num];

    nn::pcv::Module GetPcvModule(Module module)
    {
        switch (module)
        {
        case Module_Sdmmc1:
            return nn::pcv::Module_Sdmmc1;
        case Module_Sdmmc2:
            return nn::pcv::Module_Sdmmc2;
        case Module_Sdmmc3:
            return nn::pcv::Module_Sdmmc3;
        case Module_Sdmmc4:
            return nn::pcv::Module_Sdmmc4;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetClockSourceSdmmcAndEnable(uint32_t* pOutActualClockFrequencyKHz, Module module, nn::pcv::Module pcvModule, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutActualClockFrequencyKHz);

        nn::pcv::ClockHz clockRateHz = targetClockFrequencyKHz * 1000;
        Result result = nn::pcv::SetClockRate(pcvModule, clockRateHz);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetClockRate(%d, %u) is failure(Module:%d, Description:%d)\n",
            pcvModule, clockRateHz, result.GetModule(), result.GetDescription());

        result = nn::pcv::GetClockRate(&clockRateHz, pcvModule);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::GetClockRate(%d) is failure(Module:%d, Description:%d)\n",
            pcvModule, result.GetModule(), result.GetDescription());
        NN_ABORT_UNLESS_NOT_NULL(pOutActualClockFrequencyKHz);
        *pOutActualClockFrequencyKHz = nn::util::DivideUp(clockRateHz, 1000);
        g_ModuleInfos[module].targetClockFrequencyKHz = targetClockFrequencyKHz;
        g_ModuleInfos[module].actualClockFrequencyKHz = *pOutActualClockFrequencyKHz;
        NN_DETAIL_SDMMC_DEBUG_LOG("SDMMC Clock Source: %u (kHz)\n", *pOutActualClockFrequencyKHz);
    }
}

void Initialize(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(module < Module_Num);

    NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD;

    if (g_IsInitialized)
    {
        g_IsUsed[module] = true;
        return; // 初期化済み
    }

    for (int i = 0; i < Module_Num; i++)
    {
        g_IsUsed[i] = false;
    }
    g_IsUsed[module] = true;

    std::memset(g_ModuleInfos, 0, sizeof(g_ModuleInfos));

    // PLLC4 998.4MH, TMCLK = PLLP 408MHz / 34 = 12MHz 設定済みを想定

    nn::pcv::Initialize();
    g_IsInitialized = true;
}

void Finalize(Module module) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(module < Module_Num);

    NN_DETAIL_SDMMC_INITIALIZE_LOCK_GUARD;

    if (!g_IsInitialized)
    {
        return; // 終了済み
    }

    g_IsUsed[module] = false;
    for (int i = 0; i < Module_Num; i++)
    {
        if (g_IsUsed[i])
        {
            return; // 使用している module がある
        }
    }
    // どの module も使用していない

    g_IsInitialized = false;
    nn::pcv::Finalize();
}

bool IsAvailable(Module module) NN_NOEXCEPT
{
    nn::pcv::Module pcvModule = GetPcvModule(module);

    nn::pcv::ModuleState moduleState;
    nn::Result result = nn::pcv::GetState(&moduleState, pcvModule);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::GetState(%d) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());

    return ((!moduleState.resetAsserted) && (moduleState.clockEnabled));
}

void SetClockFrequencyKHz(uint32_t* pOutActualClockFrequencyKHz, Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
{
    if (targetClockFrequencyKHz == g_ModuleInfos[module].targetClockFrequencyKHz)
    {
        // 設定済み
        *pOutActualClockFrequencyKHz = g_ModuleInfos[module].actualClockFrequencyKHz;
        return;
    }

    nn::pcv::Module pcvModule = GetPcvModule(module);

    nn::pcv::ModuleState moduleState;
    nn::Result result = nn::pcv::GetState(&moduleState, pcvModule);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::GetState(%d) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());
    bool isClockEnabled = false;
    if (moduleState.clockEnabled)
    {
        result = nn::pcv::SetClockEnabled(pcvModule, false);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetClockEnabled(%d, false) is failure(Module:%d, Description:%d)\n",
            pcvModule, result.GetModule(), result.GetDescription());
        isClockEnabled = true;
    }

    SetClockSourceSdmmcAndEnable(pOutActualClockFrequencyKHz, module, pcvModule, targetClockFrequencyKHz);

    if (!isClockEnabled)
    {
        // nn::pcv::SetClockRate() でクロック供給されるため、停止
        result = nn::pcv::SetClockEnabled(pcvModule, false);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetClockEnabled(%d, false) is failure(Module:%d, Description:%d)\n",
            pcvModule, result.GetModule(), result.GetDescription());
    }
}

void ReleaseReset(Module module, uint32_t targetClockFrequencyKHz) NN_NOEXCEPT
{
    nn::pcv::Module pcvModule = GetPcvModule(module);

    nn::pcv::ModuleState moduleState;
    nn::Result result = nn::pcv::GetState(&moduleState, pcvModule);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::GetState(%d) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());
    if (moduleState.clockEnabled)
    {
        result = nn::pcv::SetClockEnabled(pcvModule, false);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetClockEnabled(%d, false) is failure(Module:%d, Description:%d)\n",
            pcvModule, result.GetModule(), result.GetDescription());
    }

    result = nn::pcv::SetReset(pcvModule, true);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetReset(%d, true) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());

    uint32_t actualSourceClockFrequencyKHz;
    SetClockSourceSdmmcAndEnable(&actualSourceClockFrequencyKHz, module, pcvModule, targetClockFrequencyKHz);

    // nn::pcv::SetClockRate() でクロック供給されるため、nn::pcv::SetClockEnabled(true) 不要

    WaitClocks(100, actualSourceClockFrequencyKHz);

    result = nn::pcv::SetReset(pcvModule, false);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetReset(%d, false) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());
}

void AssertReset(Module module) NN_NOEXCEPT
{
    nn::pcv::Module pcvModule = GetPcvModule(module);

    nn::Result result = nn::pcv::SetReset(pcvModule, true);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetReset(%d, true) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());
    result = nn::pcv::SetClockEnabled(pcvModule, false);
    NN_ABORT_UNLESS(result.IsSuccess(), "nn::pcv::SetClockEnabled(%d, false) is failure(Module:%d, Description:%d)\n",
        pcvModule, result.GetModule(), result.GetDescription());
}

} // namespace pcv {
} // namespace ClockResetController {

} // namespace detail {
}} // namespace nn { namespace sdmmc {
