﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/ro/detail/ro_RoModule.h>
#include <nn/ro/detail/ro_RoExceptionInfo.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>

#include <cstring>
#include <shared_mutex>

#include "ro_ReaderWriterLock.h"

#define NN_RO_ABORT_UNLESS(x) NN_ABORT_UNLESS(x)
#define NN_RO_PUTS(...) NN_SDK_LOG(__VA_ARGS__)

namespace nn { namespace ro {
void BindEntry() NN_NOEXCEPT;

namespace detail {

RoModuleList* g_pAutoLoadList;
RoModuleList* g_pManualLoadList;
StaticReaderWriterLock g_RoLock;
bool *g_pRoDebugFlag;
LookupGlobalAutoFunctionPointer g_pLookupGlobalAutoFunctionPointer;
LookupGlobalManualFunctionPointer* g_pLookupGlobalManualFunctionPointer;

uintptr_t Bind(RoModule* pModule, uint32_t index) NN_NOEXCEPT
{
    return pModule->BindLazy(index);
}

uintptr_t LookupGlobalManual(const char* pName) NN_NOEXCEPT
{
    std::shared_lock<detail::StaticReaderWriterLock> readerLock(detail::g_RoLock);

    uintptr_t symAddr = 0;
    for (RoModuleList::const_iterator it = g_pManualLoadList->cbegin(); it != g_pManualLoadList->end(); it++)
    {
        Elf::Sym* pSym = it->Lookup(pName);
        if (pSym && pSym->GetBind() != Elf::STB_LOCAL)
        {
            symAddr = it->GetBase() + pSym->GetValue();
            break;
        }
    }

    return symAddr;
}

Elf::Sym* RoModule::Lookup(const char* pName) const NN_NOEXCEPT
{
    for (uint32_t index = m_pBuckets[Elf::NameToHash(pName) % m_HashSize]; index; index = m_pChains[index])
    {
        Elf::Sym* pSym = &m_pDynSym[index];
        if (pSym->GetType() != Elf::STT_FILE &&
                pSym->GetShndx() != Elf::SHN_UNDEF &&
                pSym->GetShndx() != Elf::SHN_COMMON &&
                std::strcmp(pName, m_pStrTab + pSym->GetName()) == 0)
        {
            return pSym;
        }
    }
    return nullptr;
}

// FixRelative は nro では Relocation に統合

void RoModule::Initialize(uintptr_t base, size_t moduleSize, const Elf::Dyn* pDynamicSection) NN_NOEXCEPT
{
    m_Base = base;
    m_ModuleSize = moduleSize;
    m_pDynamicSection = pDynamicSection;

    uintptr_t pltRel = 0;

    m_IsPltRela = false;
    m_pPlt.rel = nullptr;
    m_PltRelSz = 0;
    m_pInit = nullptr;
    m_pFini = nullptr;
    m_pBuckets = nullptr;
    m_pChains = nullptr;
    m_pStrTab = nullptr;
    m_pDynSym = nullptr;
    m_StrSz = 0;
    m_pGot = nullptr;
    m_pDyn.rel = nullptr;
    m_DynRelSz = 0;
    m_DynRelaSz = 0;
    m_RelCount = 0;
    m_RelaCount = 0;
    m_Symbols = 0;
    m_HashSize = 0;
    m_DefaultPltGot = 0;
    m_SonameOffset = 0;

    for (const Elf::Dyn* pDyn = pDynamicSection; pDyn->GetTag() != Elf::DT_NULL; pDyn++)
    {
        switch (pDyn->GetTag())
        {
        case Elf::DT_INIT:
            {
                m_pInit = reinterpret_cast<void(*)()>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_FINI:
            {
                m_pFini = reinterpret_cast<void(*)()>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_HASH:
            {
                const uint32_t* pHash = reinterpret_cast<const uint32_t*>(base + pDyn->GetPtr());
                m_HashSize = pHash[0];
                m_Symbols = pHash[1];
                m_pBuckets = &pHash[2];
                m_pChains  = &m_pBuckets[m_HashSize];
            }
            break;
        case Elf::DT_STRTAB:
            {
                m_pStrTab = reinterpret_cast<const char*>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_SYMTAB:
            {
                m_pDynSym = reinterpret_cast<Elf::Sym*>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_STRSZ:
            {
                m_StrSz = pDyn->GetVal();
            }
            break;
        case Elf::DT_PLTGOT:
            {
                m_pGot = reinterpret_cast<Elf::Addr*>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_PLTRELSZ:
            {
                m_PltRelSz = pDyn->GetVal();
            }
            break;
        case Elf::DT_PLTREL:
            {
                m_IsPltRela = (pDyn->GetVal() == Elf::DT_RELA);
                NN_RO_ABORT_UNLESS((pDyn->GetVal() == Elf::DT_RELA) || (pDyn->GetVal() == Elf::DT_REL));
            }
            break;
        case Elf::DT_JMPREL:
            {
                pltRel = base + pDyn->GetPtr();
            }
            break;
        case Elf::DT_REL:
            {
                m_pDyn.rel = reinterpret_cast<Elf::Rel*>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_RELSZ:
            {
                m_DynRelSz = pDyn->GetVal();
            }
            break;
        case Elf::DT_RELCOUNT:
            {
                m_RelCount = pDyn->GetVal();
            }
            break;
        case Elf::DT_RELA:
            {
                m_pDyn.rela = reinterpret_cast<Elf::Rela*>(base + pDyn->GetPtr());
            }
            break;
        case Elf::DT_RELASZ:
            {
                m_DynRelaSz = pDyn->GetVal();
            }
            break;
        case Elf::DT_RELACOUNT:
            {
                m_RelaCount = pDyn->GetVal();
            }
            break;
        case Elf::DT_RELENT:
            {
                NN_RO_ABORT_UNLESS(pDyn->GetVal() == sizeof(Elf::Rel));
            }
            break;
        case Elf::DT_RELAENT:
            {
                NN_RO_ABORT_UNLESS(pDyn->GetVal() == sizeof(Elf::Rela));
            }
            break;
        case Elf::DT_SYMENT:
            {
                NN_RO_ABORT_UNLESS(pDyn->GetVal() == sizeof(Elf::Sym));
            }
            break;
        case Elf::DT_SONAME:
            {
                m_SonameOffset = pDyn->GetVal();
            }
            break;
        default:
            break;
        }
    }
    if (m_IsPltRela)
    {
        m_pPlt.rela = reinterpret_cast<Elf::Rela*>(pltRel);
    }
    else
    {
        m_pPlt.rel = reinterpret_cast<Elf::Rel*>(pltRel);
    }
} // NOLINT(readability/fn_size)

void RoModule::CallInit() NN_NOEXCEPT
{
    if (m_pInit)
    {
        m_pInit();
    }
}

void RoModule::CallFini() NN_NOEXCEPT
{
    if (m_pFini)
    {
        m_pFini();
    }
}

Elf::Addr RoModule::BindLazy(uint32_t index) NN_NOEXCEPT
{
    Elf::Addr symAddr;
    if (m_IsPltRela)
    {
        Elf::Rela& rela = m_pPlt.rela[index];
        Elf::Sym sym = m_pDynSym[rela.GetSym()];
        if (!ResolveSym(&symAddr, sym))
        {
            NN_RO_PUTS("[rtld] Unresolved symbol: '");
            const char* pName = &m_pStrTab[sym.GetName()];
            NN_UNUSED(pName);
            NN_RO_PUTS(pName);
            NN_RO_PUTS("'\n");
            symAddr = 0;
        }
        else if (symAddr)
        {
            symAddr += rela.GetAddend();
        }
    }
    else
    {
        Elf::Rel& rel = m_pPlt.rel[index];
        Elf::Sym sym = m_pDynSym[rel.GetSym()];
        if(!ResolveSym(&symAddr, sym))
        {
            NN_RO_PUTS("[rtld] Unresolved symbol: '");
            const char* pName = &m_pStrTab[sym.GetName()];
            NN_UNUSED(pName);
            NN_RO_PUTS(pName);
            NN_RO_PUTS("'\n");
            symAddr = 0;
        }
    }
    return symAddr;
}

bool RoModule::ResolveSym(Elf::Addr* pOutAddress, const Elf::Sym sym) const NN_NOEXCEPT
{
    const char* pName = &m_pStrTab[sym.GetName()];

    Elf::Addr symAddr = 0;

    if (sym.GetVisibility() != Elf::STV_DEFAULT)
    {
        Elf::Sym* pSym = Lookup(pName);
        if (pSym)
        {
            symAddr= GetBase() + pSym->GetValue();
        }
    }
    else
    {
        symAddr = g_pLookupGlobalAutoFunctionPointer(pName);
        if( symAddr == 0 )
        {
            symAddr = nn::ro::detail::LookupGlobalManual(pName);
        }
    }

    if (!(symAddr || sym.GetBind() == Elf::STB_WEAK))
    {
        return false;
    }
    else
    {
        *pOutAddress = symAddr;
        return true;
    }
}

void RoModule::Relocation(bool lazyBind) NN_NOEXCEPT
{
    for (size_t i = 0; i < m_DynRelSz / sizeof(Elf::Rel); i++)
    {
        Elf::Rel& rel = m_pDyn.rel[i];

        switch (rel.GetType())
        {
#if defined NN_BUILD_CONFIG_ABI_LP64
        case Elf::R_ABS64:
#endif
        case Elf::R_ABS32:
        case Elf::R_GLOB_DAT:
            {
                Elf::Addr symAddr;
                Elf::Sym sym = m_pDynSym[rel.GetSym()];
                if (ResolveSym(&symAddr, sym))
                {
                    Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                    *pTarget += symAddr;
                }
                else
                {
                    if (*g_pRoDebugFlag)
                    {
                        const char* pName = &m_pStrTab[sym.GetName()];
                        NN_UNUSED(pName);
                        NN_RO_PUTS("[ro] warning: unresolved symbol = '");
                        NN_RO_PUTS(pName);
                        NN_RO_PUTS("'\n");
                    }
                }
            }
            break;
        case Elf::R_RELATIVE:
            {
                Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                *pTarget += m_Base;
            }
            break;
        default:
            break;
        }
    }
    for (size_t i = 0; i < m_DynRelaSz / sizeof(Elf::Rela); i++)
    {
        Elf::Rela& rela = m_pDyn.rela[i];

        switch (rela.GetType())
        {
#if defined NN_BUILD_CONFIG_ABI_LP64
        case Elf::R_ABS64:
#endif
        case Elf::R_ABS32:
        case Elf::R_GLOB_DAT:
            {
                Elf::Addr symAddr;
                Elf::Sym sym = m_pDynSym[rela.GetSym()];
                if (ResolveSym(&symAddr, sym))
                {
                    Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                    *pTarget = symAddr + rela.GetAddend();
                }
                else
                {
                    if (*g_pRoDebugFlag)
                    {
                        const char* pName = &m_pStrTab[sym.GetName()];
                        NN_UNUSED(pName);
                        NN_RO_PUTS("[ro] warning: unresolved symbol = '");
                        NN_RO_PUTS(pName);
                        NN_RO_PUTS("'\n");
                    }
                }
            }
            break;
        case Elf::R_RELATIVE:
            {
                Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                *pTarget = m_Base + rela.GetAddend();
            }
            break;
        default:
            break;
        }
    }

    if (lazyBind)
    {
        if (m_IsPltRela)
        {
            for (size_t i = 0; i < m_PltRelSz / sizeof(Elf::Rela); i++)
            {
                Elf::Rela& rela = m_pPlt.rela[i];
                switch (rela.GetType())
                {
                case Elf::R_JUMP_SLOT:
                    {
                        Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                        *pTarget += m_Base;
                        if (m_DefaultPltGot == 0)
                        {
                            m_DefaultPltGot = *pTarget;
                        }
                        else
                        {
                            NN_RO_ABORT_UNLESS(m_DefaultPltGot == *pTarget);
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }
        else
        {
            for (size_t i = 0; i < m_PltRelSz / sizeof(Elf::Rel); i++)
            {
                Elf::Rel& rel = m_pPlt.rel[i];
                switch (rel.GetType())
                {
                case Elf::R_JUMP_SLOT:
                    {
                        Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                        *pTarget += m_Base;
                        if (m_DefaultPltGot == 0)
                        {
                            m_DefaultPltGot = *pTarget;
                        }
                        else
                        {
                            NN_RO_ABORT_UNLESS(m_DefaultPltGot == *pTarget);
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }
    }
    else
    {
        if (m_IsPltRela)
        {
            for (size_t i = 0; i < m_PltRelSz / sizeof(Elf::Rela); i++)
            {
                Elf::Rela& rela = m_pPlt.rela[i];

                switch (rela.GetType())
                {
                case Elf::R_JUMP_SLOT:
                    {
                        Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                        if (m_DefaultPltGot == 0)
                        {
                            m_DefaultPltGot = *pTarget + m_Base;
                        }
                        else
                        {
                            NN_RO_ABORT_UNLESS(m_DefaultPltGot == *pTarget + m_Base);
                        }
                        Elf::Addr symAddr;
                        Elf::Sym sym = m_pDynSym[rela.GetSym()];
                        if(ResolveSym(&symAddr, sym))
                        {
                            *pTarget = symAddr + rela.GetAddend();
                        }
                        else
                        {
                            if (*g_pRoDebugFlag)
                            {
                                const char* pName = &m_pStrTab[sym.GetName()];
                                NN_UNUSED(pName);
                                NN_RO_PUTS("[rtld] warning: unresolved symbol = '");
                                NN_RO_PUTS(pName);
                                NN_RO_PUTS("'\n");
                            }
                            *pTarget += m_Base;
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }
        else
        {
            for (size_t i = 0; i < m_PltRelSz / sizeof(Elf::Rel); i++)
            {
                Elf::Rel& rel = m_pPlt.rel[i];

                switch (rel.GetType())
                {
                case Elf::R_JUMP_SLOT:
                    {
                        Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                        if (m_DefaultPltGot == 0)
                        {
                            m_DefaultPltGot = *pTarget + m_Base;
                        }
                        else
                        {
                            NN_RO_ABORT_UNLESS(m_DefaultPltGot == *pTarget + m_Base);
                        }
                        Elf::Addr symAddr;
                        Elf::Sym sym = m_pDynSym[rel.GetSym()];
                        if(ResolveSym(&symAddr, sym))
                        {
                            *pTarget = symAddr;
                        }
                        else
                        {
                            if (*g_pRoDebugFlag)
                            {
                                const char* pName = &m_pStrTab[sym.GetName()];
                                NN_UNUSED(pName);
                                NN_RO_PUTS("[rtld] warning: unresolved symbol = '");
                                NN_RO_PUTS(pName);
                                NN_RO_PUTS("'\n");
                            }
                            *pTarget += m_Base;
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }
    }
    if (m_pGot)
    {
        m_pGot[2] = reinterpret_cast<Elf::Addr>(&nn::ro::BindEntry);
        m_pGot[1] = reinterpret_cast<Elf::Addr>(this);
    }
} // NOLINT(readability/fn_size)

void RoModule::BindVariables(const RoModule* pNewModule) NN_NOEXCEPT
{
    for (size_t i = m_RelCount; i < m_DynRelSz / sizeof(Elf::Rel); i++)
    {
        Elf::Rel& rel = m_pDyn.rel[i];

        switch (rel.GetType())
        {
#if defined NN_BUILD_CONFIG_ABI_LP64
        case Elf::R_ABS64:
#endif
        case Elf::R_ABS32:
        case Elf::R_GLOB_DAT:
            {
                const Elf::Sym& sym = m_pDynSym[rel.GetSym()];
                const char* pSymName = &m_pStrTab[sym.GetName()];
                Elf::Sym* pSym = pNewModule->Lookup(pSymName);
                if (pSym)
                {
                    Elf::Addr symAddr;
                    if (ResolveSym(&symAddr, sym))
                    {
                        if (symAddr == pSym->GetValue() + pNewModule->GetBase())
                        {
                            if (*g_pRoDebugFlag)
                            {
                                NN_RO_PUTS("[ro] warning: resolve '");
                                NN_RO_PUTS(pSymName);
                                NN_RO_PUTS("'\n");
                            }
                            Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                            *pTarget += symAddr;
                        }
                    }
                }
            }
            break;
        default:
            break;
        }
    }
    for (size_t i = m_RelaCount; i < m_DynRelaSz / sizeof(Elf::Rela); i++)
    {
        Elf::Rela& rela = m_pDyn.rela[i];

        switch (rela.GetType())
        {
#if defined NN_BUILD_CONFIG_ABI_LP64
        case Elf::R_ABS64:
#endif
        case Elf::R_ABS32:
        case Elf::R_GLOB_DAT:
            {
                const Elf::Sym& sym = m_pDynSym[rela.GetSym()];
                const char* pSymName = &m_pStrTab[sym.GetName()];
                Elf::Sym* pSym = pNewModule->Lookup(pSymName);
                if (pSym)
                {
                    Elf::Addr symAddr;
                    if (ResolveSym(&symAddr, sym))
                    {
                        if (symAddr == pSym->GetValue() + pNewModule->GetBase())
                        {
                            if (*g_pRoDebugFlag)
                            {
                                NN_RO_PUTS("[ro] warning: resolve '");
                                NN_RO_PUTS(pSymName);
                                NN_RO_PUTS("'\n");
                            }
                            Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                            *pTarget = symAddr + rela.GetAddend();
                        }
                    }
                }
            }
            break;
        default:
            break;
        }
    }
}

void RoModule::RestoreFunction(RoModule* module) NN_NOEXCEPT
{
    if (m_IsPltRela)
    {
        for (size_t i = 0; i < m_PltRelSz / sizeof(Elf::Rela); i++)
        {
            Elf::Rela& rela = m_pPlt.rela[i];
            switch (rela.GetType())
            {
                case Elf::R_JUMP_SLOT:
                    {
                        Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                        if (module->IsAddressWithinModule(*pTarget))
                        {
                            if (*g_pRoDebugFlag)
                            {
                                Elf::Sym sym = m_pDynSym[rela.GetSym()];
                                const char* pSymName = &m_pStrTab[sym.GetName()];
                                NN_UNUSED(pSymName);
                                NN_RO_PUTS("[ro] warning: reset '");
                                NN_RO_PUTS(pSymName);
                                NN_RO_PUTS("'\n");
                            }
                            *pTarget = m_DefaultPltGot;
                        }
                    }
                    break;
                default:
                    break;
            }
        }
    }
    else
    {
        for (size_t i = 0; i < m_PltRelSz / sizeof(Elf::Rel); i++)
        {
            Elf::Rel& rel = m_pPlt.rel[i];
            switch (rel.GetType())
            {
                case Elf::R_JUMP_SLOT:
                    {
                        Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                        if (module->IsAddressWithinModule(*pTarget))
                        {
                            if (*g_pRoDebugFlag)
                            {
                                Elf::Sym sym = m_pDynSym[rel.GetSym()];
                                const char* pSymName = &m_pStrTab[sym.GetName()];
                                NN_UNUSED(pSymName);
                                NN_RO_PUTS("[ro] warning: reset '");
                                NN_RO_PUTS(pSymName);
                                NN_RO_PUTS("'\n");
                            }
                            *pTarget = m_DefaultPltGot;
                        }
                    }
                    break;
                default:
                    break;
            }
        }
    }

}

void RoModule::RestoreVariable(RoModule* module) NN_NOEXCEPT
{
    for (size_t i = m_RelCount; i < m_DynRelSz / sizeof(Elf::Rel); i++)
    {
        Elf::Rel& rel = m_pDyn.rel[i];

        switch (rel.GetType())
        {
#if defined NN_BUILD_CONFIG_ABI_LP64
        case Elf::R_ABS64:
#endif
        case Elf::R_ABS32:
        case Elf::R_GLOB_DAT:
            {
                Elf::Sym targetSym =  m_pDynSym[rel.GetSym()];
                // アンロードされないモジュール内のシンボルのうち、UND でないものだとしても、同じ名前のシンボルが
                // アンロードするモジュールに存在している場合そちらを参照している可能性があるので、処理はスキップしない
                const char* pTargetName = &m_pStrTab[targetSym.GetName()];
                Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rel.GetOffset());
                if (module->IsAddressWithinModule(*pTarget))
                {
                    Elf::Sym* pUnloadSym = module->Lookup(pTargetName);
                    if (pUnloadSym != nullptr)
                    {
                        Elf::Addr symbolAddress = module->GetBase() + pUnloadSym->GetValue(); // symbolAddress は、アンロードする NRO 内のシンボル名, m_Base はアンロードするモジュール
                        if (*g_pRoDebugFlag)
                        {
                            NN_RO_PUTS("[ro] warning: reset '");
                            NN_RO_PUTS(pTargetName);
                            NN_RO_PUTS("'\n");
                        }
                        *pTarget -= symbolAddress;
                        Elf::Addr symAddr;
                        if (ResolveSym(&symAddr, targetSym))
                        {
                            *pTarget += symAddr;
                        }
                        break;
                    }
                }
            }
            break;
        default:
            break;
        }
    }
    for (size_t i = m_RelaCount; i < m_DynRelaSz / sizeof(Elf::Rela); i++)
    {
        Elf::Rela& rela = m_pDyn.rela[i];

        switch (rela.GetType())
        {
#if defined NN_BUILD_CONFIG_ABI_LP64
        case Elf::R_ABS64:
#endif
        case Elf::R_ABS32:
        case Elf::R_GLOB_DAT:
            {
                Elf::Sym targetSym = m_pDynSym[rela.GetSym()];
                // アンロードされないモジュール内のシンボルのうち、UND でないものだとしても、同じ名前のシンボルが
                // アンロードするモジュールに存在している場合そちらを参照している可能性があるので、処理はスキップしない
                Elf::Addr* pTarget = reinterpret_cast<Elf::Addr*>(m_Base + rela.GetOffset());
                if (module->IsAddressWithinModule(*pTarget))
                {
                    // rela は AArch64 の ELF の仕様上、GetAddend に SymbolAddress を加えたものにすればよいだけであるため、
                    // 元々入っていた値を求める必要はない。BindVariable も同様の処理となっています。
                    *pTarget = 0;
                    Elf::Addr symAddr;
                    if (ResolveSym(&symAddr, targetSym))
                    {
                        *pTarget = symAddr + rela.GetAddend();
                    }
                }
            }
            break;
        default:
            break;
        }
    }
}

void RoModule::RevertReference(RoModule* module) NN_NOEXCEPT
{
    RestoreVariable(module);
    RestoreFunction(module);
}

}
}}
