﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ovln/ovln_Receiver.h>

#include <nn/ovln/ovln_Services.sfdl.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <utility>
#include <nn/ovln/ovln_ResultPrivate.h>
#include <cstring>
#include <nn/sf/sf_ShimLibraryUtility.h>

namespace nn { namespace ovln {

namespace detail {

sf::SharedPointer<IReceiverService> GetReceiverServicePrepared() NN_NOEXCEPT;

} // detail

namespace {

sf::ShimLibraryObjectHolder<IReceiverService> g_ReceiverService = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

} // anon

Result InitializeReceiverLibrary() NN_NOEXCEPT
{
    #if defined(NN_OVLN_USES_DIRECT_AGENT)
        return g_ReceiverService.InitializeHolder([](sf::SharedPointer<IReceiverService>* pOut) -> Result
        {
            *pOut = detail::GetReceiverServicePrepared();
            NN_RESULT_SUCCESS;
        });
    #elif defined(NN_OVLN_USES_HIPC_AGENT)
        static nn::sf::SimpleAllInOneHipcSubDomainClientManager<10> g_HipcSub = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;
        return g_HipcSub.InitializeShimLibraryHolder(&g_ReceiverService, "ovln:rcv");
    #else
        #error "not supported"
    #endif
}

void FinalizeReceicerLibrary() NN_NOEXCEPT
{
    return g_ReceiverService.FinalizeHolder();
}

sf::ShimLibraryObjectHolder<IReceiverService>& GetReceiverServiceShimLibraryObjectHolder() NN_NOEXCEPT
{
    return g_ReceiverService;
}

Receiver::Receiver() NN_NOEXCEPT
    : m_pReceiver(nullptr)
{
}

bool Receiver::IsValid() const NN_NOEXCEPT
{
    return m_pReceiver != nullptr;
}

Result Receiver::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsValid());

    auto service = g_ReceiverService.GetObject();
    sf::SharedPointer<IReceiver> receiver;
    NN_RESULT_DO(service->OpenReceiver(&receiver));
    sf::NativeHandle handle;
    NN_RESULT_DO(receiver->GetReceiveEventHandle(&handle));

    this->m_pReceiver = receiver.Detach();
    os::AttachReadableHandleToSystemEvent(&this->m_Event, handle.GetOsHandle(), handle.IsManaged(), os::EventClearMode_ManualClear);
    this->m_EventHandleManaged = handle.IsManaged();
    handle.Detach();

    NN_SDK_ASSERT(IsValid());
    NN_RESULT_SUCCESS;
}

void Receiver::Finalize() NN_NOEXCEPT
{
    if (IsValid())
    {
        sf::ReleaseSharedObject(m_pReceiver);
        os::DestroySystemEvent(&m_Event);
        this->m_pReceiver = nullptr;
    }
    NN_SDK_ASSERT(!IsValid());
}

Receiver::~Receiver() NN_NOEXCEPT
{
    Finalize();
}

void Receiver::swap(Receiver& other) NN_NOEXCEPT
{
    auto thisHandle = os::NativeHandle();
    if (this->IsValid())
    {
        thisHandle = os::DetachReadableHandleOfSystemEvent(&this->m_Event);
        os::DestroySystemEvent(&this->m_Event);
    }
    auto otherHandle = os::NativeHandle();
    if (other.IsValid())
    {
        otherHandle = os::DetachReadableHandleOfSystemEvent(&other.m_Event);
        os::DestroySystemEvent(&other.m_Event);
    }
    if (this->IsValid())
    {
        os::AttachReadableHandleToSystemEvent(&other.m_Event, thisHandle, this->m_EventHandleManaged, os::EventClearMode_ManualClear);
    }
    if (other.IsValid())
    {
        os::AttachReadableHandleToSystemEvent(&this->m_Event, otherHandle, other.m_EventHandleManaged, os::EventClearMode_ManualClear);
    }
    using std::swap;
    swap(this->m_pReceiver, other.m_pReceiver);
    swap(this->m_EventHandleManaged, other.m_EventHandleManaged);
}

Receiver::Receiver(Receiver&& other) NN_NOEXCEPT
    : m_pReceiver(nullptr)
{
    NN_SDK_ASSERT(!IsValid());
    this->swap(other);
    NN_SDK_ASSERT(!other.IsValid());
}

Receiver& Receiver::operator=(Receiver&& rhs) NN_NOEXCEPT
{
    Receiver(std::move(rhs)).swap(*this);
    return *this;
}

Result Receiver::AddSource(const SourceName& sourceName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    return m_pReceiver->AddSource(sourceName);
}

void Receiver::RemoveSource(const SourceName& sourceName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pReceiver->RemoveSource(sourceName));
}

bool Receiver::TryReceive(Message* pMessage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());

    RawMessage rawMessage;
    NN_RESULT_TRY(m_pReceiver->Receive(&rawMessage))
        NN_RESULT_CATCH(ovln::ResultNoMessage)
        {
            return false;
        }
    NN_RESULT_END_TRY

    pMessage->tag = rawMessage.tag;
    pMessage->dataSize = rawMessage.dataSize;
    NN_SDK_ASSERT(rawMessage.dataSize <= sizeof(pMessage->data));
    std::memcpy(&pMessage->data, &rawMessage.data, rawMessage.dataSize);
    return true;
}

bool Receiver::TryReceive(Message* pMessage, os::Tick* pOutTick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());

    RawMessage rawMessage;
    int64_t tick;
    NN_RESULT_TRY(m_pReceiver->ReceiveWithTick(&rawMessage, &tick))
        NN_RESULT_CATCH(ovln::ResultNoMessage)
        {
            return false;
        }
    NN_RESULT_END_TRY

    pMessage->tag = rawMessage.tag;
    pMessage->dataSize = rawMessage.dataSize;
    NN_SDK_ASSERT(rawMessage.dataSize <= sizeof(pMessage->data));
    std::memcpy(&pMessage->data, &rawMessage.data, rawMessage.dataSize);
    *pOutTick = os::Tick(tick);
    return true;
}

void Receiver::Receive(Message* pMessage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    do
    {
        os::WaitSystemEvent(&m_Event);
    } while (!TryReceive(pMessage));
}

void Receiver::Receive(Message* pMessage, os::Tick* pOutTick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    do
    {
        os::WaitSystemEvent(&m_Event);
    } while (!TryReceive(pMessage, pOutTick));
}

bool Receiver::HasMessageToReceive() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    return os::TryWaitSystemEvent(&m_Event);
}

void Receiver::WaitForMessageToReceive() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    os::WaitSystemEvent(&m_Event);
}

void Receiver::AttachWaitHolder(os::MultiWaitHolderType* pWaitHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid());
    os::InitializeMultiWaitHolder(pWaitHolder, &m_Event);
}

}}
