﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_SdkLog.h>

#include <nn/os/os_ThreadTypes.h>

#include <nn/osdbg/osdbg_Result.h>
#include <nn/osdbg/osdbg_ThreadTypes.h>
#include <nn/osdbg/osdbg_ThreadApi.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_ThreadLocalRegion.h>

#include "osdbg_ThreadInfo-os.horizon.h"
#include "osdbg_ThreadType-os.horizon.h"
#include "osdbg_ThreadLocalRegion-os.horizon.h"
#include "../../os/detail/os_ThreadManager-os.horizon.h"


namespace nn { namespace osdbg {
namespace detail {

namespace {

//-----------------------------------------------------------------------------
//  デバッグ対象スレッドの現在の優先度を返す
//  svc 呼出しでエラー発生時は _basePriority をそのまま返す
//
inline int ConvertToSigloPriority(int horizonPriority)
{
    return horizonPriority - os::detail::UserThreadPriorityOffset;
}

inline int GetCurrentThreadPriorityImpl(ThreadInfo* info) NN_NOEXCEPT
{

    Bit64  outDummy;
    Bit32  outHorizonPriority;
    Result result = svc::GetDebugThreadParam( &outDummy,
                                              &outHorizonPriority,
                                              info->_debugHandle,
                                              info->_debugInfoCreateThread.id,
                                              svc::DebugThreadParam_Priority);
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("[osdbg] GetCurrentThreadPriorityImpl(tid=%lld) result=0x%08x\n", info->_debugInfoCreateThread.id, result.GetInnerValueForDebug());
        return info->_basePriority;
    }

    return ConvertToSigloPriority( static_cast<int>(outHorizonPriority) );
}

}   // namespace

//-----------------------------------------------------------------------------
//  デバッグ対象からのスレッド情報の取得
//  info->_threadType == NULL の場合、最初に ThreadType ポインタを取得する
//
Result ThreadInfoImplByHorizon::FillWithCurrentInfo(ThreadInfo* info) NN_NOEXCEPT
{
    bool isLp64 = IsLp64(info);

    if (info->_threadType == NULL)
    {
        // 初呼出し時にターゲットスレッドの ThreadType アドレスの取得を行なう
        ThreadTypeCommon* p = GetTargetThreadType(info);
        if (p == NULL)
        {
            return osdbg::ResultCannotGetThreadInfo();
        }

        info->_threadType = p;
    }

    // デバッグ対象スレッドの ThreadType 構造体の中身を取得する
    ThreadTypeCommon threadType;
    Result result = svc::ReadDebugProcessMemory(
                            reinterpret_cast<uintptr_t>( &threadType ),
                            info->_debugHandle,
                            reinterpret_cast<uintptr_t>( info->_threadType ),
                            (isLp64 ? sizeof(ThreadTypeLp64) : sizeof(ThreadTypeIlp32))
                        );
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("[osdbg] FillWithCurrentInfo(tid=%lld) %s address=%p result=0x%08x\n", info->_debugInfoCreateThread.id, isLp64 ? "lp64" : "ilp32", info->_threadType, result.GetInnerValueForDebug());
        return result;
    }

    // Fill info parameter
    if (isLp64)
    {
        info->_basePriority     = threadType.lp64._basePriority;
        info->_currentPriority  = GetCurrentThreadPriorityImpl(info);

        info->_stackSize        = threadType.lp64._stackSize;
        info->_stack            = threadType.lp64._stack;
        info->_argument         = threadType.lp64._argument;
        info->_function         = threadType.lp64._threadFunction;
        info->_namePointer      = threadType.lp64._namePointer;
    }
    else
    {
        info->_basePriority     = threadType.ilp32._basePriority;
        info->_currentPriority  = GetCurrentThreadPriorityImpl(info);

        info->_stackSize        = threadType.ilp32._stackSize;
        info->_stack            = threadType.ilp32._stack;
        info->_argument         = threadType.ilp32._argument;
        info->_function         = threadType.ilp32._threadFunction;
        info->_namePointer      = threadType.ilp32._namePointer;
    }

    return ResultSuccess();
}

//-----------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::osdbg

