﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/os/os_TimerEventApi.h>

#include "detail/os_Diag.h"
#include "detail/os_TimerEventHelper.h"
#include "detail/os_TickManager.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderOfTimerEvent.h"
#include "detail/os_MultipleWaitHolderImpl.h"


namespace nn { namespace os {

namespace {

inline uint64_t GetBroadcastCounterUnsafe(TimerEventType* p) NN_NOEXCEPT
{
    uint64_t upper = p->_broadcastCounterUpper;
    return (upper << (sizeof(p->_broadcastCounterLower) * 8)) | p->_broadcastCounterLower;
}

inline void IncrementBroadcastCounterUnsafe(TimerEventType* p) NN_NOEXCEPT
{
    if (0 == (++p->_broadcastCounterLower))
    {
        ++p->_broadcastCounterUpper;
    }
}

void SignalTimerEventImplUnsafe(TimerEventType* event) NN_NOEXCEPT
{
    // シグナル状態にセット
    event->_signalState = true;

    if (event->_clearMode == EventClearMode_ManualClear)
    {
        IncrementBroadcastCounterUnsafe(event);
        Get(event->_cvSignaled).Broadcast();
    }
    else /* AutoClear */
    {
        Get(event->_cvSignaled).Signal();
    }

    // 多重待ちスレッドを起床（ロック取得状態で呼出す）
    Get(event->_multiWaitObjectList).WakeupAllMultiWaitThreadsUnsafe();
}

}   // namespace

//---------------------------------------------------------------------------
//  イニシャライズ
void InitializeTimerEvent(TimerEventType* event, EventClearMode clearMode) NN_NOEXCEPT
{
    // 排他リソースを初期化
    new( &event->_csTimerEvent ) detail::InternalCriticalSection;
    new( &event->_cvSignaled )   detail::InternalConditionVariable;

    // IntrusiveList のコンストラクタを呼ぶ
    new( &event->_multiWaitObjectList ) detail::MultiWaitObjectList;

    // メンバの初期化
    event->_clearMode           = static_cast<uint8_t>(clearMode);
    event->_signalState         = false;
    event->_timerState          = TimerEventType::TimerState_Stop;
    event->_broadcastCounterLower = 0;
    event->_broadcastCounterUpper = 0;

    Get(event->_nextTimeToWakeup)   = 0;
    Get(event->_first)              = 0;
    Get(event->_interval)           = 0;

    // タイマーイベントを Initialized 状態へ（最後に行なう）
    event->_state               = TimerEventType::State_Initialized;
}


//---------------------------------------------------------------------------
//  ファイナライズ
void FinalizeTimerEvent(TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    // タイマーイベントを NotInitialized 状態へ（最初に行なう）
    event->_state  = TimerEventType::State_NotInitialized;

    // IntrusiveList のデストラクタを呼ぶ
    Get(event->_multiWaitObjectList).~MultiWaitObjectList();

    // 排他リソースを破棄
    Get(event->_csTimerEvent).~InternalCriticalSection();
    Get(event->_cvSignaled).~InternalConditionVariable();
}


//---------------------------------------------------------------------------
//  ワンショットタイマーの開始
void StartOneShotTimerEvent(TimerEventType* event, nn::TimeSpan first) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( first >= 0, NN_TEXT_OS("nn::os::StartOneShotTimerEvent(): first は 0 以上でなければなりません。") );
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::StartOneShotTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        TimeSpan currTime = detail::GetCurrentTick().ToTimeSpan();

        Get(event->_nextTimeToWakeup) = detail::SaturatedAdd(currTime, first);
        Get(event->_first)            = first;
        Get(event->_interval)         = 0;
        event->_timerState            = TimerEventType::TimerState_OneShot;

        // タイマーのシグナル化時間が変わるので全待機スレッドを一旦起床
        Get(event->_cvSignaled).Broadcast();

        // 多重待ちスレッドもタイムアウト時間再計算のため一旦起床
        Get(event->_multiWaitObjectList).BroadcastToUpdateObjectStateUnsafe();
    }
}


//---------------------------------------------------------------------------
//  周期タイマーの開始
void StartPeriodicTimerEvent(TimerEventType* event, nn::TimeSpan first, nn::TimeSpan interval) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( first >= 0, NN_TEXT_OS("nn::os::StartPeriodicTimerEvent(): first は 0 以上でなければなりません。") );
    NN_SDK_REQUIRES( interval > 0, NN_TEXT_OS("nn::os::StartPeriodicTimerEvent(): interval は 0 より大きくなければなりません。") );
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::StartPeriodicTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        TimeSpan currTime = detail::GetCurrentTick().ToTimeSpan();

        Get(event->_nextTimeToWakeup) = detail::SaturatedAdd(currTime, first);
        Get(event->_first)            = first;
        Get(event->_interval)         = interval;
        event->_timerState            = TimerEventType::TimerState_Periodic;

        // タイマーのシグナル化時間が変わるので全待機スレッドを一旦起床
        Get(event->_cvSignaled).Broadcast();

        // 多重待ちスレッドもタイムアウト時間再計算のため一旦起床
        Get(event->_multiWaitObjectList).BroadcastToUpdateObjectStateUnsafe();
    }
}


//---------------------------------------------------------------------------
//  タイマーの停止
void StopTimerEvent(TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::StopTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        detail::StopTimerUnsafe(event);

        // タイマー待ちが不要になるので全待機スレッドを一旦起床
        Get(event->_cvSignaled).Broadcast();

        // 多重待ちスレッドもタイムアウト時間再計算のため一旦起床
        Get(event->_multiWaitObjectList).BroadcastToUpdateObjectStateUnsafe();
    }
}


//---------------------------------------------------------------------------
//  シグナル状態にセット
void SignalTimerEvent(TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::SignalTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );


    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // 既にシグナル状態であれば何もしない
        if ( event->_signalState == true )
        {
            return;
        }

        // タイマー動作によるシグナル化状況を _signalState へ反映させ、
        // 次のタイマーシグナル化時間を算出しておく
        auto currTime = detail::GetCurrentTick().ToTimeSpan();
        detail::UpdateSignalStateAndRecalcNextTimeToWakeupUnsafe(event, currTime);

        // シグナル状態にセット
        SignalTimerEventImplUnsafe(event);
    }
}


//---------------------------------------------------------------------------
//  非シグナル状態へクリア
void ClearTimerEvent(TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::ClearTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // タイマー動作によるシグナル化状況を _signalState へ反映させ、
        // 次のタイマーシグナル化時間を算出しておく
        auto currTime = detail::GetCurrentTick().ToTimeSpan();
        if (detail::UpdateSignalStateAndRecalcNextTimeToWakeupUnsafe(event, currTime))
        {
            SignalTimerEventImplUnsafe(event);
        }

        // クリアモードに関係なく非シグナル化
        event->_signalState = false;
    }
}


//---------------------------------------------------------------------------
//  シグナル状態になるまで待機
//  タイマーによるシグナル化は、誰からも通知が来ないので、
//  SVC のタイムアウト機能を使って実現する
void WaitTimerEvent(TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::WaitTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection& cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection> lock( cs );

        auto currentCounter = GetBroadcastCounterUnsafe(event);
        while (event->_signalState == false)
        {
            // ManualClear の場合、カウンタが進んでいたら起床する
            // AutoClear   の場合、カウンタが進むことはない
            if (currentCounter != GetBroadcastCounterUnsafe(event))
            {
                break;
            }

            // タイマー動作によるシグナル化状況を _signalState へ反映させ、
            // 次のタイマーシグナル化時間を算出しておく
            auto currTime = detail::GetCurrentTick().ToTimeSpan();
            if (detail::UpdateSignalStateAndRecalcNextTimeToWakeupUnsafe(event, currTime))
            {
                SignalTimerEventImplUnsafe(event);
                break;
            }

            // Wait() から起床後はループして再判定する。
            if (event->_timerState == TimerEventType::TimerState_Stop)
            {
                Get(event->_cvSignaled).Wait(&cs);
            }
            else
            {
                // タイマー機能実現のため、タイムアウトを使って待機する
                TimeSpan nextTime = Get(event->_nextTimeToWakeup);
                detail::TimeoutHelper tmout(nextTime - currTime);

                Get(event->_cvSignaled).TimedWait(&cs, tmout);
            }
        }

        // AutoClear の場合だけ自動クリアする
        if (event->_clearMode == EventClearMode_AutoClear)
        {
            event->_signalState = false;
        }
    }
}


//---------------------------------------------------------------------------
//  シグナル状態になるまで待機（ポーリング）
bool TryWaitTimerEvent(TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::TryWaitTimerEvent(): 指定されたタイマーイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csTimerEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // タイマー動作によるシグナル化状況を _signalState へ反映させ、
        // 次のタイマーシグナル化時間を算出しておく
        auto currTime = detail::GetCurrentTick().ToTimeSpan();
        if (detail::UpdateSignalStateAndRecalcNextTimeToWakeupUnsafe(event, currTime))
        {
            SignalTimerEventImplUnsafe(event);
        }

        bool prevState = event->_signalState;

        // 自動クリアモードなら非シグナル状態へ
        if (event->_clearMode == EventClearMode_AutoClear)
        {
            event->_signalState = false;
        }

        return prevState;
    }
}


//---------------------------------------------------------------------------
//  多重待ちホルダーの初期化（TimerEventType を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, TimerEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == TimerEventType::State_Initialized, NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): 指定されたタイマーイベントが初期化されていません。") );

    // コンストラクタ呼出し
    new(&Get(pHolder->_holderImpl)) detail::MultiWaitHolderOfTimerEvent(event);

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

