﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_SharedMemoryApi.h>
#include <nn/os/os_SharedMemoryTypes.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_MemoryHeapManager.h"
#include "detail/os_SharedMemoryImpl.h"
#include "detail/os_AslrSpaceManagerTypes.h"
#include "detail/os_AslrSpaceManager.h"


namespace nn { namespace os {

//--------------------------------------------------------------------------

namespace {

inline void SetupSharedMemoryType(SharedMemoryType* sharedMemory, size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
{
    // メンバの初期化
    sharedMemory->_handle       = handle;
    sharedMemory->_size         = size;
    sharedMemory->_address      = NULL;
    sharedMemory->_allocated    = false;

    // このメンバが true の場合、_handle メンバの管理責任も同時に保有している。
    // そのため、DestroySharedMemory() 時にクローズする必要がある。
    sharedMemory->_isHandleManaged  = managed;

    // コンストラクタを呼ぶ
    new( &sharedMemory->_csSharedMemory ) detail::InternalCriticalSection;

    // 共有メモリを初期化状態へ
    sharedMemory->_state    = SharedMemoryType::State_Initialized;
}

}   // namespace

//--------------------------------------------------------------------------
//  共有メモリの作成（ハンドルの生成）
nn::Result CreateSharedMemory(SharedMemoryType* sharedMemory, size_t size, MemoryPermission myPermission, MemoryPermission otherPermission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( size > 0, NN_TEXT_OS("nn::os::CreateSharedMemory(): size > 0 ではありません（size=0x%zx）"), size );
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("nn::os::CreateSharedMemory(): size が MemoryPageSize の整数倍ではありません（size=0x%zx）"), size );

    NativeHandle  handle;
    auto result = detail::SharedMemoryImpl::Create(&handle, size, myPermission, otherPermission);
    if (!result.IsSuccess())
    {
        return result;
    }

    // メンバの初期化
    SetupSharedMemoryType(sharedMemory, size, handle, true);

    return nn::ResultSuccess();
}


//--------------------------------------------------------------------------
//  共有メモリオブジェクトへのハンドルのアタッチ
void AttachSharedMemory(SharedMemoryType* sharedMemory, size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( detail::IsAligned( size, MemoryPageSize ), NN_TEXT_OS("nn::os::AttachSharedMemory(): size が MemoryPageSize の整数倍ではありません（size=0x%zx）"), size );
    NN_SDK_REQUIRES( handle != nn::os::InvalidNativeHandle, NN_TEXT_OS("nn::os::AttachSharedMemory(): handle が有効なハンドルではありません。") );

    // メンバの初期化
    SetupSharedMemoryType(sharedMemory, size, handle, managed);
}


//--------------------------------------------------------------------------
//  共有メモリの破棄
void DestroySharedMemory(SharedMemoryType* sharedMemory) NN_NOEXCEPT
{
    if ( sharedMemory->_state == SharedMemoryType::State_Mapped )
    {
        UnmapSharedMemory(sharedMemory);
    }

    NN_SDK_REQUIRES( sharedMemory->_state == SharedMemoryType::State_Initialized, NN_TEXT_OS("nn::os::DestroySharedMemory(): 共有メモリが初期化されていません。") );

    sharedMemory->_state    = SharedMemoryType::State_NotInitialized;

    if (sharedMemory->_isHandleManaged)
    {
        detail::SharedMemoryImpl::Close(sharedMemory->_handle);
    }
    sharedMemory->_isHandleManaged  = false;

    sharedMemory->_address  = NULL;
    sharedMemory->_size     = 0;
    sharedMemory->_handle   = 0;

    // デストラクタを呼ぶ
    Get(sharedMemory->_csSharedMemory).~InternalCriticalSection();
}


//--------------------------------------------------------------------------
//  共有メモリをマップ
void* MapSharedMemory(SharedMemoryType* sharedMemory, MemoryPermission myPermission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( sharedMemory->_state == SharedMemoryType::State_Initialized, NN_TEXT_OS("nn::os::MapSharedMemory(): 共有メモリが初期化状態ではありません。") );

    // TORIAEZU: 使用中の領域にマップしようとした場合、64 回までリトライする
    for (int loopCount = 0; loopCount < 64; ++loopCount)
    {   // クリティカルセクション
        std::lock_guard<detail::InternalCriticalSection> lockForSuspend( Get(detail::GetCurrentThread()->_csThread) );
        std::lock_guard<detail::InternalCriticalSection> lock( Get(sharedMemory->_csSharedMemory) );

        // フリーマップ領域からマップ先を確保
        void* mappingAddress = detail::GetAslrSpaceManagerInstance()->AllocateSpace(sharedMemory->_size);
        if (!mappingAddress)
        {
            // 候補となる空き空間すらない
            return NULL;
        }
        sharedMemory->_allocated = true;

        void* mappedAddress = NULL;
        auto result = detail::SharedMemoryImpl::Map(&mappedAddress,
                                                    sharedMemory->_handle,
                                                    mappingAddress,
                                                    sharedMemory->_size,
                                                    myPermission);

        // 成功した場合
        if (result.IsSuccess())
        {
            // 前後にガード空間があるかチェック
            if (!detail::GetAslrSpaceManagerInstance()->CheckGuardSpace(reinterpret_cast<uintptr_t>(mappedAddress), sharedMemory->_size))
            {
                // なかった場合、アンマップしてリトライ
                detail::SharedMemoryImpl::Unmap(sharedMemory->_handle, mappedAddress, sharedMemory->_size);
            }
            else
            {
                sharedMemory->_address   = mappedAddress;
                sharedMemory->_state     = SharedMemoryType::State_Mapped;
                return mappedAddress;
            }
        }

        // 失敗した場合はロールバック（空間の返却）
        sharedMemory->_allocated = false;

        // ResultInvalidCurrentMemoryState 以外のエラーの場合、すぐに返る
        if (!ResultInvalidCurrentMemoryState::Includes(result))
        {
            return NULL;
        }
    }
    return NULL;
}


//--------------------------------------------------------------------------
//  共有メモリをアンマップ
void UnmapSharedMemory(SharedMemoryType* sharedMemory) NN_NOEXCEPT
{
    // クリティカルセクション
    // ロック区間が長いが、通常使用では全く問題にならない。
    // そもそも、複数スレッドから同時に呼ばれるような使い方は使用方法が
    // 間違っている可能性が高い。ここでのロックは安全に abort するためである。
    std::lock_guard<detail::InternalCriticalSection> lock( Get(sharedMemory->_csSharedMemory) );

    if ( sharedMemory->_state != SharedMemoryType::State_Mapped )
    {
        return;
    }

    detail::SharedMemoryImpl::Unmap(sharedMemory->_handle, sharedMemory->_address, sharedMemory->_size);

    // 空間の返却
    if (sharedMemory->_allocated)
    {
        sharedMemory->_allocated = false;
    }

    // メンバの初期化
    sharedMemory->_address  = NULL;
    sharedMemory->_state    = SharedMemoryType::State_Initialized;
}


//--------------------------------------------------------------------------
//  共有メモリのマップ先アドレスの取得
void* GetSharedMemoryAddress(const SharedMemoryType* sharedMemory) NN_NOEXCEPT
{
    auto state = sharedMemory->_state;
    NN_SDK_REQUIRES( ( state == SharedMemoryType::State_Initialized ) ||
                     ( state == SharedMemoryType::State_Mapped ),
        NN_TEXT_OS("nn::os::GetSharedMemoryAddress(): 共有メモリが初期化されていません。") );
    NN_UNUSED(state);

    return sharedMemory->_address;
}

//--------------------------------------------------------------------------
//  共有メモリのサイズの取得
size_t GetSharedMemorySize(const SharedMemoryType* sharedMemory) NN_NOEXCEPT
{
    auto state = sharedMemory->_state;
    NN_SDK_REQUIRES( ( state == SharedMemoryType::State_Initialized ) ||
                     ( state == SharedMemoryType::State_Mapped ),
        NN_TEXT_OS("nn::os::GetSharedMemorySize(): 共有メモリが初期化されていません。") );
    NN_UNUSED(state);

    return sharedMemory->_size;
}

//--------------------------------------------------------------------------
//  共有メモリのハンドルの取得
NativeHandle GetSharedMemoryHandle(const SharedMemoryType* sharedMemory) NN_NOEXCEPT
{
    auto state = sharedMemory->_state;
    NN_SDK_REQUIRES( ( state == SharedMemoryType::State_Initialized ) ||
                     ( state == SharedMemoryType::State_Mapped ),
        NN_TEXT_OS("nn::os::GetSharedMemoryHandle(): 共有メモリが初期化されていません。") );
    NN_UNUSED(state);

    return sharedMemory->_handle;
}

//--------------------------------------------------------------------------

}} // namespace nn::os

