﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkConditionVariable.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_TimeoutHelper.h"

namespace nn { namespace os {

//-----------------------------------------------------------------------------
// os::SdkMutexType 用の Wait() と TimedWait()
//
void SdkConditionVariableType::Wait(SdkMutexType& mutex) NN_NOEXCEPT
{
    NN_ABORT_UNLESS( mutex.IsLockedByCurrentThread() );
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = nullptr;
#endif
    Get(this->_condition).Wait( &Get(mutex._mutex) );
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = detail::GetCurrentThread();
#endif
}

bool SdkConditionVariableType::TimedWait(SdkMutexType& mutex, TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0 );
    NN_ABORT_UNLESS( mutex.IsLockedByCurrentThread() );

    // 条件変数待ちを発行（Mutex も一旦解放される）
    if (timeout == TimeSpan(0))
    {
        Get(mutex._mutex).Leave();
        Get(mutex._mutex).Enter();
        return false;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = nullptr;
#endif
    detail::TimeoutHelper tmout( timeout );
    auto ret = Get(this->_condition).TimedWait( &Get(mutex._mutex), tmout );
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = detail::GetCurrentThread();
#endif
    return ConditionVariableStatus_NoTimeout == ret;
}

//-----------------------------------------------------------------------------
// os::SdkRecursiveMutexType 用の Wait() と TimedWait()
//
void SdkConditionVariableType::Wait(SdkRecursiveMutexType& mutex) NN_NOEXCEPT
{
    NN_ABORT_UNLESS( mutex.IsLockedByCurrentThread() );
    NN_ABORT_UNLESS( mutex._recursiveCount == 1 );
    mutex._recursiveCount = 0;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = nullptr;
#endif
    Get(this->_condition).Wait( &Get(mutex._mutex) );
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = detail::GetCurrentThread();
#endif
    NN_ABORT_UNLESS( ++mutex._recursiveCount );
}

bool SdkConditionVariableType::TimedWait(SdkRecursiveMutexType& mutex, TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0 );
    NN_ABORT_UNLESS( mutex.IsLockedByCurrentThread() );

    // 条件変数待ちを発行（Mutex も一旦解放される）
    if (timeout == TimeSpan(0))
    {
        Get(mutex._mutex).Leave();
        Get(mutex._mutex).Enter();
        return false;
    }

    NN_ABORT_UNLESS( mutex._recursiveCount == 1 );
    mutex._recursiveCount = 0;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = nullptr;
#endif
    detail::TimeoutHelper tmout( timeout );
    auto ret = Get(this->_condition).TimedWait( &Get(mutex._mutex), tmout );
#if defined(NN_BUILD_CONFIG_OS_WIN)
    mutex._ownerThread = detail::GetCurrentThread();
#endif
    NN_ABORT_UNLESS( ++mutex._recursiveCount );
    return ConditionVariableStatus_NoTimeout == ret;
}

}}  // namespace nn::os

