﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/util/util_TypedStorage.h>
#include "os_Common.h"
#include "os_TickManager.h"
#include "os_Diag.h"

namespace nn { namespace os {
namespace detail {

//---------------------------------------------------------------------------
//  Tick 値を TimeSpan 値に変換して返す
TimeSpan TickManager::ConvertToTimeSpan(Tick tickValue) const NN_NOEXCEPT
{
    int64_t tick     = tickValue.GetInt64Value();
    int64_t tickFreq = GetTickFrequency();

    // tickFreq が 32bit 以上である可能性は低く、実質的にこのループは回らない。
    // 将来的に、上記 tickFreq が定数値になれば、
    // 下記ループも最適化されるため、このループ実装は残しておきます。
    while ( tickFreq >= (1LL << 31) )
    {
        tick     >>= 1;
        tickFreq >>= 1;
    }

    // この時点で、tickFreq は確実に 31bit 以下（2,147,483,647 以下）である。
    NN_SDK_ASSERT( tickFreq < (1LL << 31), NN_TEXT_OS("nn::os::ConvertToTimeSpan(): 演算過程で桁あふれを検出しました。") );

    // 算出する TimeSpan 型は「ナノ秒」を最小単位とする型なので、
    // この時点での tickFreq が 10 桁以上あれば、精度が落ちることはない。

    const int64_t timeSpanUnit  =  1 * 1000 * 1000 * 1000;
    int64_t seconds  = tick / tickFreq; // 「秒」以上を算出
    int64_t fraction = tick % tickFreq; // 「秒」未満を算出
                                        //  fraction は 31bit 以下になる

    return TimeSpan::FromSeconds(seconds) +
           TimeSpan::FromNanoSeconds(fraction * timeSpanUnit / tickFreq);
}

//---------------------------------------------------------------------------
//  TimeSpan 値を Tick 値に変換して返す
Tick TickManager::ConvertToTick(TimeSpan timeSpan) const NN_NOEXCEPT
{
    // Tick の周期を獲得しておく。
    // Tick の周期は 31bit で表現可能な範囲を前提とする。
    int64_t tickFreq = GetTickFrequency();
    NN_SDK_ASSERT( tickFreq <= (1LL << 31), NN_TEXT_OS("nn::os::ConvertToTick(): 演算過程で桁あふれを検出しました。") );

    // timeSpan 時間情報を 秒 と 小数点以下 に分ける
    const int64_t timeSpanUnit        = 1 * 1000 * 1000 * 1000;
    int64_t       timeSpanNanoSeconds = timeSpan.GetNanoSeconds();
    int64_t       seconds             = timeSpanNanoSeconds / timeSpanUnit;
    int64_t       fraction            = timeSpanNanoSeconds % timeSpanUnit;
    bool          isMinusSign         = timeSpanNanoSeconds < 0;

    if (isMinusSign)
    {
        // -ConvertToTick( TimeSpan ) == ConvertToTick( -TimeSpan) となるよう、
        // timeSpan が負の場合、絶対値の Tick を計算し、最後に符号を付けて返す。
        // この処理を入れない場合、timeSpan の正負によって丸めの方向が変わってしまう。
        // timeSpanNanoSeconds の絶対値を取ると負の最大値の場合にオーバーフローするため
        // seconds と fraction の絶対値を取る。
        seconds = -seconds;
        fraction = -fraction;
    }

    // timeSpanUnit で割ったものを tickFreq で乗算するため、
    // 元の TimeSpan 値が 63bit 以下なら演算結果も 63bit 以下で表現できる。

    int64_t tick = (seconds * tickFreq) +
                   ((fraction * tickFreq + timeSpanUnit - 1) / timeSpanUnit);

    // 符号を付けて返す
    if (isMinusSign)
    {
        tick = -tick;
    }

    return Tick( tick );
}

}  // namespace detail
}} // namespace nn::os

