﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma warning( disable : 4127 )
#endif

#include <nn/nn_Common.h>
#include <iterator>
#include <sys/tree.h>

namespace nn { namespace os { namespace detail {

struct IntrusiveRbTreeNode
{
private:
    NN_DISALLOW_COPY(IntrusiveRbTreeNode);

public:
    IntrusiveRbTreeNode() NN_NOEXCEPT
    {
    }

private:
    template <class, class> friend class IntrusiveRbTree;
    RB_ENTRY(IntrusiveRbTreeNode) m_Entry;
};


template <class T, class Tag = T>
class IntrusiveRbTreeBaseNode : public IntrusiveRbTreeNode
{
};


template<class T, class Compare>
class IntrusiveRbTree
{
public:
    IntrusiveRbTree() NN_NOEXCEPT
    {
        RB_INIT(&m_Tree);
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        return RB_EMPTY(&m_Tree);
    }

    static T* Next(T* pItem) NN_NOEXCEPT
    {
        return GetItemPtr(NextImpl(pItem));
    }

    static T* Prev(T* pItem) NN_NOEXCEPT
    {
        return GetItemPtr(PrevImpl(pItem));
    }

    static const T* Next(const T* pItem) NN_NOEXCEPT
    {
        IntrusiveRbTreeNode* nonConst = const_cast<IntrusiveRbTreeNode*>(pItem);
        return GetItemPtr(NextImpl(nonConst));
    }

    static const T* Prev(const T* pItem) NN_NOEXCEPT
    {
        IntrusiveRbTreeNode* nonConst = const_cast<IntrusiveRbTreeNode*>(pItem);
        return GetItemPtr(PrevImpl(nonConst));
    }

    T* Remove(T* pItem) NN_NOEXCEPT
    {
        return GetItemPtr(RB_REMOVE(IntrusiveRbTreeInternal, &m_Tree, pItem));
    }

    T* Insert(T* pItem) NN_NOEXCEPT
    {
        return GetItemPtr(RB_INSERT(IntrusiveRbTreeInternal, &m_Tree, pItem));
    }

    T* Find(const T* pItem) const NN_NOEXCEPT
    {
        return GetItemPtr(
            RB_FIND(
                IntrusiveRbTreeInternal,
                const_cast<IntrusiveRbTreeInternal*>(&m_Tree),
                const_cast<IntrusiveRbTreeNode*>(static_cast<const IntrusiveRbTreeNode*>(pItem)) ));
    }

    T* NearFind(const T* pItem) const NN_NOEXCEPT
    {
        return GetItemPtr(
            RB_NFIND(
                IntrusiveRbTreeInternal,
                const_cast<IntrusiveRbTreeInternal*>(&m_Tree),
                const_cast<IntrusiveRbTreeNode*>(static_cast<const IntrusiveRbTreeNode*>(pItem)) ));
    }

    T* GetMin() const NN_NOEXCEPT
    {
        return GetItemPtr(RB_MIN(IntrusiveRbTreeInternal, const_cast<IntrusiveRbTreeInternal*>(&m_Tree)));
    }

    T* GetMax() const NN_NOEXCEPT
    {
        return GetItemPtr(RB_MAX(IntrusiveRbTreeInternal, const_cast<IntrusiveRbTreeInternal*>(&m_Tree)));
    }

private:
    RB_HEAD(IntrusiveRbTreeInternal, IntrusiveRbTreeNode);
    NN_DISALLOW_COPY(IntrusiveRbTree);

    static IntrusiveRbTreeNode* NextImpl(IntrusiveRbTreeNode* pItem) NN_NOEXCEPT
    {
        return RB_NEXT(IntrusiveRbTreeInternal, nullptr, pItem);
    }

    static IntrusiveRbTreeNode* PrevImpl(IntrusiveRbTreeNode* pItem) NN_NOEXCEPT
    {
        return RB_PREV(IntrusiveRbTreeInternal, nullptr, pItem);
    }

    static int IntrusiveRbTreeCompare(const IntrusiveRbTreeNode* a, const IntrusiveRbTreeNode* b) NN_NOEXCEPT
    {
        return Compare()(*GetItemPtr(a), *GetItemPtr(b));
    }

    static const T* GetItemPtr(const IntrusiveRbTreeNode* node) NN_NOEXCEPT
    {
        return static_cast<const T*>(static_cast<const IntrusiveRbTreeBaseNode<T>*>(node));
    }
    static T* GetItemPtr(IntrusiveRbTreeNode* node) NN_NOEXCEPT
    {
        return static_cast<T*>(static_cast<IntrusiveRbTreeBaseNode<T>*>(node));
    }

private:
    RB_GENERATE_STATIC(IntrusiveRbTreeInternal, IntrusiveRbTreeNode, m_Entry, IntrusiveRbTreeCompare);
    IntrusiveRbTreeInternal m_Tree;
};


}}}
