﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdint.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_InterruptEventCommon.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_InterruptEventImpl.h"
#include "os_TimeoutHelper.h"

#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Types.h>

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// コンストラクタ
InterruptEventImplByHorizon::InterruptEventImplByHorizon(InterruptName name, EventClearMode clearMode) NN_NOEXCEPT
{
    m_ManualReset = (clearMode == EventClearMode_ManualClear);

    auto interruptType = m_ManualReset ? nn::svc::InterruptType_Level
                                       : nn::svc::InterruptType_Edge;

    nn::svc::Handle handle;
    auto result = nn::svc::CreateInterruptEvent(&handle, static_cast<int32_t>(name), interruptType);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    m_Handle = handle;
}

//--------------------------------------------------------------------------
// デストラクタ
InterruptEventImplByHorizon::~InterruptEventImplByHorizon() NN_NOEXCEPT
{
    auto result = nn::svc::CloseHandle( m_Handle );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------
// クリア
void InterruptEventImplByHorizon::Clear() NN_NOEXCEPT
{
    auto result = nn::svc::ClearEvent( m_Handle );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------
// 割込みイベント待機
void InterruptEventImplByHorizon::Wait() NN_NOEXCEPT
{
    for(;;)
    {
        int32_t index;
        Result ret = nn::svc::WaitSynchronization(&index, &m_Handle, 1, nn::svc::WAIT_INFINITE );
        if ( ret.IsSuccess() )
        {
            if (!m_ManualReset)
            {
                auto result = svc::ResetSignal( m_Handle );
                if (result <= svc::ResultInvalidState())
                {
                    // アトミックなクリアに失敗したら再度 Wait する
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            return;
        }

        // 通常、単独の InterruptEvent では CancelSynchronization されることは
        // ないが、多重待ちのスレッド起床用に CancelSynchronization されると、
        // その内部フラグが残っている場合に ResultCancelled() が返る可能性
        // があるため、これが返ってきた場合はループして再度待ちに入る。
        NN_SDK_ASSERT( ret <= nn::svc::ResultCancelled(), NN_TEXT_OS("nn::os::WaitInterruptEvent(): イベントの待機中に異常を検出しました。"));
    }
}

//--------------------------------------------------------------------------
// 割込みイベントのポーリング
bool InterruptEventImplByHorizon::TryWait() NN_NOEXCEPT
{
    if (!m_ManualReset)
    {
        Result ret = nn::svc::ResetSignal( m_Handle );
        return ret.IsSuccess();
    }

    for (;;)
    {
        int32_t index;
        Result ret = nn::svc::WaitSynchronization(&index, &m_Handle, 1, 0 );
        if ( ret.IsSuccess() )
        {
            return true;
        }
        if ( ret <= nn::svc::ResultTimeout() )
        {
            return false;
        }

        // ResultCancelled ならばループして再度ポーリングする
        NN_SDK_ASSERT( ret <= nn::svc::ResultCancelled(), NN_TEXT_OS("nn::os::TryWaitInterruptEvent(): イベントのポーリング中に異常を検出しました。"));
    }
}

//--------------------------------------------------------------------------
// 割込みイベント待機（タイムアウト付き）
bool InterruptEventImplByHorizon::TimedWait(TimeSpan timeSpan) NN_NOEXCEPT
{
    TimeoutHelper   tmout( timeSpan );

    for (;;)
    {
        int32_t index;
        Result ret = nn::svc::WaitSynchronization(&index, &m_Handle, 1,
                                tmout.GetLeftTimeOnTarget().GetNanoSeconds() );
        if ( ret.IsSuccess() )
        {
            if (!m_ManualReset)
            {
                auto result = svc::ResetSignal( m_Handle );
                if (result <= svc::ResultInvalidState())
                {
                    // アトミックなクリアに失敗したら再度 Wait する
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            return true;
        }
        if ( ret <= nn::svc::ResultTimeout() )
        {
            return false;
        }

        // ResultCancelled ならばループして再度待ちに入る
        NN_SDK_ASSERT( ret <= nn::svc::ResultCancelled(), NN_TEXT_OS("nn::os::TimedWaitInterruptEvent(): イベントの時限付き待機中に異常を検出しました。"));
    }
}

}   // namespace detail
}}  // namespace nn::os

