﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Windows.h>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_SystemEventTypes.h>

#include "os_Diag.h"
#include "os_Utility.h"
#include "os_InterProcessEventImpl.h"
#include "os_TimeoutHelper.h"
#include "os_GiantLock.h"


namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// Win32 Event をアトミックにクリアする。クリア前の状態が返値となる。
bool InterProcessEventImplByWin32::ResetEventSignal(NativeHandle handle) NN_NOEXCEPT
{
    std::lock_guard<GiantLock> lock(*GetGiantLockInstance());

    DWORD ret = ::WaitForSingleObject(handle, 0);
    if (ret == WAIT_OBJECT_0)
    {
        // シグナル状態だった場合はクリアして true を返す
        ::ResetEvent( handle );
        return true;
    }

    return false;
}

//--------------------------------------------------------------------------
// Event ハンドル生成
Result InterProcessEventImplByWin32::Create(NativeHandle* pOutWritableHandle, NativeHandle* pOutReadableHandle) NN_NOEXCEPT
{
    // Win32 イベントの生成（必ずマニュアルリセットで生成する）
    HANDLE writableHandle = ::CreateEvent( NULL, TRUE, FALSE, NULL );
    NN_SDK_ASSERT( writableHandle != NULL, NN_TEXT_OS("nn::os::CreateSystemEvent(): Win32 イベントの生成に失敗しました。\n") );

    HANDLE readableHandle;
    HANDLE myProcess = ::GetCurrentProcess();
    BOOL ret = ::DuplicateHandle(myProcess, writableHandle, myProcess, &readableHandle, 0, FALSE, DUPLICATE_SAME_ACCESS);
    NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::CreateSystemEvent(): Win32 イベントの複製に失敗しました。\n") );
    NN_UNUSED(ret);

    *pOutWritableHandle = writableHandle;
    *pOutReadableHandle = readableHandle;

    return ResultSuccess();
}

//--------------------------------------------------------------------------
// Event ハンドルのクローズ
void InterProcessEventImplByWin32::Close(NativeHandle handle) NN_NOEXCEPT
{
    if (handle != InvalidNativeHandle)
    {
        BOOL ret = ::CloseHandle( handle );
        NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::DestroySystemEvent(): Win32 イベントのクローズに失敗しました。"));
        NN_UNUSED(ret);
    }
}

//--------------------------------------------------------------------------
// シグナル
void InterProcessEventImplByWin32::Signal(NativeHandle handle) NN_NOEXCEPT
{
    BOOL ret = ::SetEvent( handle );
    NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::SignalSystemEvent(): Win32 イベントのセットに失敗しました。"));
    NN_UNUSED(ret);
}

//--------------------------------------------------------------------------
// クリア
void InterProcessEventImplByWin32::Clear(NativeHandle handle) NN_NOEXCEPT
{
    BOOL ret = ::ResetEvent( handle );
    NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::ClearSystemEvent(): Win32 イベントのクリアに失敗しました。"));
    NN_UNUSED(ret);
}

//--------------------------------------------------------------------------
// プロセス間同期イベント待機
void InterProcessEventImplByWin32::Wait(NativeHandle handle, bool autoClear) NN_NOEXCEPT
{
    for (;;)
    {
        DWORD ret = ::WaitForSingleObject( handle, INFINITE );
        NN_SDK_ASSERT( ret == WAIT_OBJECT_0, NN_TEXT_OS("nn::os::WaitSystemEvent(): Win32 イベントの待機でエラーが発生しました。"));
        NN_UNUSED(ret);

        if (!autoClear)
        {
            return;
        }

        bool result = ResetEventSignal(handle);
        if (result == true)
        {
            return;
        }
    }
}

//--------------------------------------------------------------------------
// プロセス間同期イベントのポーリング
bool InterProcessEventImplByWin32::TryWait(NativeHandle handle, bool autoClear) NN_NOEXCEPT
{
    if (autoClear)
    {
        return ResetEventSignal(handle);
    }

    DWORD result = ::WaitForSingleObject( handle, 0 );
    NN_SDK_ASSERT( (result == WAIT_OBJECT_0) || (result == WAIT_TIMEOUT), NN_TEXT_OS("nn::os::TryWaitSystemEvent(): Win32 イベントのポーリングでエラーが発生しました。"));

    return result == WAIT_OBJECT_0;
}

//--------------------------------------------------------------------------
// プロセス間同期イベント待機（タイムアウト付き）
bool InterProcessEventImplByWin32::TimedWait(NativeHandle handle, bool autoClear, TimeSpan timeout) NN_NOEXCEPT
{
    detail::TimeoutHelper   tmout( timeout );

    do
    {
        DWORD result = ::WaitForSingleObject( handle, tmout.GetLeftTimeOnTarget() );
        if (result == WAIT_OBJECT_0)
        {
            if (!autoClear)
            {
                return true;
            }

            bool ret = ResetEventSignal(handle);
            if (ret == true)
            {
                return true;
            }
        }

    } while ( !tmout.IsTimedout() );

    return false;
}

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

