﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"

#include <nn/nn_Windows.h>
#pragma warning( push )
#pragma warning ( disable : 4668 )
#include <winnt.h>
#pragma warning( pop )
#include <Psapi.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include "os_Diag.h"
#include "os_DebugImpl.h"
#include "os_ThreadManager.h"
#include "os_MemoryHeapManager.h"
#include "os_TransferMemoryImpl.h"
#include "os_SharedMemoryImpl.h"

#include <algorithm>

namespace nn { namespace os { namespace detail {

//-----------------------------------------------------------------------------
//  スタック情報の取得
void DebugImpl::GetCurrentStackInfo(uintptr_t* outStack, size_t* outStackSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( outStack     != NULL );
    NN_SDK_ASSERT( outStackSize != NULL );

    NT_TIB* tib     = reinterpret_cast<NT_TIB*>( NtCurrentTeb() );

    *outStack       = reinterpret_cast<uintptr_t>(tib->StackLimit);
    *outStackSize   = reinterpret_cast<uintptr_t>(tib->StackBase) -
                      reinterpret_cast<uintptr_t>(tib->StackLimit);
}

//-----------------------------------------------------------------------------
//  メモリ情報の取得
void DebugImpl::QueryMemoryInfo(MemoryInfo* info) NN_NOEXCEPT
{
    // 情報取得その１
    HANDLE hProcess = ::OpenProcess( PROCESS_QUERY_INFORMATION,
                                     FALSE,
                                     ::GetCurrentProcessId() );
    NN_SDK_ASSERT( hProcess != NULL );

    PROCESS_MEMORY_COUNTERS pmc = { 0 };
    auto ret = ::GetProcessMemoryInfo( hProcess, &pmc, sizeof(pmc) );
    NN_SDK_ASSERT( ret != 0 );
    CloseHandle( hProcess );

    // 情報取得その２
    MEMORYSTATUSEX mstatex;
    mstatex.dwLength = sizeof(mstatex);
    ret = ::GlobalMemoryStatusEx(&mstatex);
    NN_SDK_ASSERT( ret != 0 );

    // 収集した情報をセット
    auto* memoryHeap = GetMemoryHeapManagerInstance();
    info->totalAvailableMemorySize  = std::min<uint64_t>(mstatex.ullAvailPhys, memoryHeap->GetReservedSize());
    info->totalUsedMemorySize       = pmc.PeakWorkingSetSize;
    info->programSize               = 0;

    info->totalMemoryHeapSize       = memoryHeap->GetHeapSize();
    info->allocatedMemoryHeapSize   = memoryHeap->GetUsedHeapSize();

    auto* threadManager = GetThreadManagerInstance();
    info->totalThreadStackSize      = threadManager->GetTotalThreadStackSize();
    info->threadCount               = threadManager->GetThreadCountForDebug();
}

//-----------------------------------------------------------------------------

}}} // namespace nn::os::detail

