﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/util/util_Optional.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include "ncm_FileSystemUtility.h"

#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fsa/fs_Registrar.h>

namespace nn { namespace ncm {

class SubmissionPackageInstallTask::Impl
{
public:
    explicit Impl(fs::FileHandle file) NN_NOEXCEPT : m_Storage(file){}
    ~Impl()
    {
        if(m_MountName)
        {
            nn::fs::fsa::Unregister(m_MountName->string);
        }
    }

    Result Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(! m_MountName);

        std::unique_ptr<nn::fssystem::PartitionFileSystem> fileSystem(new nn::fssystem::PartitionFileSystem());
        NN_RESULT_THROW_UNLESS(fileSystem, ResultAllocationMemoryFailed());
        NN_RESULT_DO(fileSystem->Initialize(&m_Storage));
        auto mountName = detail::CreateUniqueMountName();
        NN_RESULT_DO(nn::fs::fsa::Register(mountName.string, std::move(fileSystem)));
        m_MountName = mountName;

        NN_RESULT_SUCCESS;
    }

    const detail::MountName& GetMountName() const NN_NOEXCEPT
    {
        return *m_MountName;
    }

private:
    nn::fs::FileHandleStorage m_Storage;
    util::optional<detail::MountName> m_MountName;
};

SubmissionPackageInstallTask::SubmissionPackageInstallTask() NN_NOEXCEPT{}
SubmissionPackageInstallTask::~SubmissionPackageInstallTask() NN_NOEXCEPT{}

Result SubmissionPackageInstallTask::Initialize(fs::FileHandle file, ncm::StorageId storage, void* buffer, size_t bufferSize, bool ignoreTicket) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(! m_Impl);

    m_Impl.reset(new Impl(file));
    NN_RESULT_THROW_UNLESS(m_Impl, ResultAllocationMemoryFailed());
    NN_RESULT_DO(m_Impl->Initialize());
    PackageInstallTask::Initialize(detail::GetRootDirectoryPath(m_Impl->GetMountName()).string, storage, buffer, bufferSize, ignoreTicket);

    NN_RESULT_SUCCESS;
}

}}
