﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_MountPrivate.h>

#include <nn/ncm/detail/ncm_Log.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_PlaceHolderAccessor.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_UuidApi.h>
#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {

namespace
{

static const char BASE_PLACEHOLDER_DIRECTORY[] = "/placehld";
static const char PLACEHOLDER_FILE_NAME[] = "00001111222233334444555566667777.nca";
static const char PLACEHOLDER_FILE_EXTENSION[] = ".nca";

// #define NN_NCM_ENABLE_DETECTING_WRITE_AFTER_GETPATH

#if defined(NN_NCM_ENABLE_DETECTING_WRITE_AFTER_GETPATH)
PlaceHolderId s_RecentlyUsed = {};
PlaceHolderId s_WriteAfterGetPath = {};
#endif

void MakeBasePlaceHolderDirectoryPath(PathString* outValue, const char* rootPath) NN_NOEXCEPT
{
    outValue->AssignFormat("%s%s", rootPath, BASE_PLACEHOLDER_DIRECTORY);
}

void MakePlaceHolderFilePath(PathString* outValue, PlaceHolderId id, MakePlaceHolderPathFunction func, const char* rootPath) NN_NOEXCEPT
{
    PathString dirPath;
    MakeBasePlaceHolderDirectoryPath(&dirPath, rootPath);
    func(outValue, id, dirPath);
}

}

void PlaceHolderAccessor::MakePath(PathString* outPath, PlaceHolderId id) const NN_NOEXCEPT
{
    MakePlaceHolderFilePath(outPath, id, m_Func, *m_RootPath);
}
void PlaceHolderAccessor::MakeBaseDirectoryPath(PathString* outPath, const char* rootPath) NN_NOEXCEPT
{
    MakeBasePlaceHolderDirectoryPath(outPath, rootPath);
}


Result PlaceHolderAccessor::GetPlaceHolderIdFromFileName(PlaceHolderId* outValue, const char* name) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(util::Strnlen(name, sizeof(PLACEHOLDER_FILE_NAME)) == util::Strnlen(PLACEHOLDER_FILE_NAME, sizeof(PLACEHOLDER_FILE_NAME)), ResultInvalidPlaceHolderFile());
    NN_RESULT_THROW_UNLESS(util::Strncmp(&name[sizeof(PLACEHOLDER_FILE_NAME) - sizeof(PLACEHOLDER_FILE_EXTENSION)], PLACEHOLDER_FILE_EXTENSION, static_cast<int>(sizeof(PLACEHOLDER_FILE_EXTENSION))) == 0, ResultInvalidPlaceHolderFile());

    PlaceHolderId placeHolderId = {};
    for (int i = 0; i < sizeof(placeHolderId.uuid.data); i++)
    {
        char buffer[3];
        util::Strlcpy(buffer, &name[i * 2], 3);

        char* endPtr;
        auto byte = static_cast<Bit8>(std::strtoul(buffer, &endPtr, 16));
        NN_RESULT_THROW_UNLESS(*endPtr == '\0', ResultInvalidPlaceHolderFile());

        placeHolderId.uuid.data[i] = byte;
    }

    *outValue = placeHolderId;

    NN_RESULT_SUCCESS;
}

Result PlaceHolderAccessor::CreatePlaceHolderFile(PlaceHolderId id, int64_t size) NN_NOEXCEPT
{
    NN_RESULT_DO(EnsurePlaceHolderDirectory(id));

    PathString filePath;
    GetPath(&filePath, id);
    NN_RESULT_TRY(fs::CreateFile(filePath, size, fs::CreateFileOptionFlag_BigFile))
        NN_RESULT_CATCH(fs::ResultPathAlreadyExists){ NN_RESULT_THROW(ResultPlaceHolderAlreadyExists()); }
    NN_RESULT_END_TRY

#if defined(NN_NCM_ENABLE_DETECTING_WRITE_AFTER_GETPATH)
    s_RecentlyUsed = {};
#endif
    NN_RESULT_SUCCESS;
}
Result PlaceHolderAccessor::SetPlaceHolderFileSize(PlaceHolderId id, int64_t size) NN_NOEXCEPT
{
    fs::FileHandle handle;
    NN_RESULT_TRY(Open(&handle, id))
        NN_RESULT_CATCH(fs::ResultPathNotFound) { NN_RESULT_THROW(ResultPlaceHolderNotFound()); }
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    NN_RESULT_DO(fs::SetFileSize(handle, size));

    NN_RESULT_SUCCESS;
}
Result PlaceHolderAccessor::TryGetPlaceHolderFileSize(bool* outIsFileSizeSet, int64_t* outFileSize, PlaceHolderId id) NN_NOEXCEPT
{
    fs::FileHandle handle;
    auto found = LoadCache(&handle, id);
    if (found)
    {
        StoreCache(id, handle);
        NN_RESULT_DO(fs::GetFileSize(outFileSize, handle));
        *outIsFileSizeSet = true;
        NN_RESULT_SUCCESS;
    }
    *outIsFileSizeSet = false;
    NN_RESULT_SUCCESS;
}
Result PlaceHolderAccessor::DeletePlaceHolderFile(PlaceHolderId id) NN_NOEXCEPT
{
    PathString filePath;

    GetPath(&filePath, id);
    NN_RESULT_TRY(fs::DeleteFile(filePath))
        NN_RESULT_CATCH(fs::ResultPathNotFound) { NN_RESULT_THROW(ResultPlaceHolderNotFound()); }
    NN_RESULT_END_TRY

#if defined(NN_NCM_ENABLE_DETECTING_WRITE_AFTER_GETPATH)
    s_RecentlyUsed = {};
#endif
    NN_RESULT_SUCCESS;
}
Result PlaceHolderAccessor::WritePlaceHolderFile(PlaceHolderId id, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
#if defined(NN_NCM_ENABLE_DETECTING_WRITE_AFTER_GETPATH)
    if (s_RecentlyUsed.uuid == id.uuid)
    {
        NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] Write after GetPath detected  %02x%02x%02x%02x\n", id.uuid.data[0], id.uuid.data[1], id.uuid.data[2], id.uuid.data[3]);
        s_WriteAfterGetPath.uuid = s_RecentlyUsed.uuid;
    }
#endif
    //    NN_DETAIL_NCM_TRACE("[ncm] WritePlaceHolder offset: %llx, size: %llx\n", offset, size);
    fs::FileHandle handle;
    NN_RESULT_TRY(Open(&handle, id))
        NN_RESULT_CATCH(fs::ResultPathNotFound) { NN_RESULT_THROW(ResultPlaceHolderNotFound()); }
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT { StoreCache(id, handle); };

    fs::WriteOption writeOption = fs::WriteOption::MakeValue(m_DelayFlush ? 0 : fs::WriteOptionFlag_Flush);
    NN_RESULT_DO(fs::WriteFile(handle, offset, buffer, size, writeOption));

    NN_RESULT_SUCCESS;
}

Result PlaceHolderAccessor::Open(fs::FileHandle* outHandle, PlaceHolderId id) NN_NOEXCEPT
{
    auto found = LoadCache(outHandle, id);
    if (found)
    {
        // NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] cache hit %02x%02x%02x%02x\n", id.uuid.data[0], id.uuid.data[1], id.uuid.data[2], id.uuid.data[3]);
        NN_RESULT_SUCCESS;
    }
    PathString path;
    MakePlaceHolderFilePath(&path, id, m_Func, *m_RootPath);
    NN_RESULT_DO(fs::OpenFile(outHandle, path, fs::OpenMode_Write));

    NN_RESULT_SUCCESS;
}

bool PlaceHolderAccessor::LoadCache(fs::FileHandle* outHandle, PlaceHolderId id) NN_NOEXCEPT
{
    // NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] load cache\n");
    std::lock_guard<os::Mutex> guard(m_CacheMutex);
    auto cache = FindCache(id);
    if (cache == nullptr)
    {
        return false;
    }
    cache->id.uuid = util::InvalidUuid;
    *outHandle = cache->handle;

    return true;
}

void PlaceHolderAccessor::StoreCache(PlaceHolderId id, fs::FileHandle handle) NN_NOEXCEPT
{
    // NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] store cache\n");
    std::lock_guard<os::Mutex> guard(m_CacheMutex);
    auto cache = GetCacheSpace();
    cache->id      = id;
    cache->handle  = handle;
    cache->counter = m_CurrentCounter;
    m_CurrentCounter++;
}

void PlaceHolderAccessor::GetPath(PathString* outPath, PlaceHolderId id) NN_NOEXCEPT
{
    {
        std::lock_guard<os::Mutex> guard(m_CacheMutex);
        Invalidate(FindCache(id));
    }
    MakePlaceHolderFilePath(outPath, id, m_Func, *m_RootPath);
#if defined(NN_NCM_ENABLE_DETECTING_WRITE_AFTER_GETPATH)
    s_RecentlyUsed = id;
    if (s_WriteAfterGetPath.uuid == id.uuid)
    {
        NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] GetPath is re-called after detecing Write after GetPath  %02x%02x%02x%02x\n", id.uuid.data[0], id.uuid.data[1], id.uuid.data[2], id.uuid.data[3]);
        s_WriteAfterGetPath = {};
    }
#endif
}
void PlaceHolderAccessor::Invalidate(Cache* cache) NN_NOEXCEPT
{
    if (cache != nullptr)
    {
        // NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] Invalidate %02x%02x%02x%02x\n", cache->id.uuid.data[0], cache->id.uuid.data[1], cache->id.uuid.data[2], cache->id.uuid.data[3]);
        fs::FlushFile(cache->handle);
        fs::CloseFile(cache->handle);
        cache->id.uuid = util::InvalidUuid;
    }
}
void PlaceHolderAccessor::InvalidateAll() NN_NOEXCEPT
{
    for (auto& cache : m_Caches)
    {
        if (cache.id.uuid != util::InvalidUuid)
        {
            Invalidate(&cache);
        }
    }
}

PlaceHolderAccessor::Cache* PlaceHolderAccessor::FindCache(PlaceHolderId id) NN_NOEXCEPT
{
    if (id.uuid == util::InvalidUuid)
    {
        // id = 0 で Delete してくるものが観測されたため
        NN_DETAIL_NCM_TRACE("[PlaceHolderAccessor] invalid uuid is specified\n");
        return nullptr;
    }
    for(auto& cache : m_Caches)
    {
        if (cache.id.uuid == id.uuid)
        {
            return &cache;
        }
    }
    return nullptr;
}

PlaceHolderAccessor::Cache* PlaceHolderAccessor::GetCacheSpace() NN_NOEXCEPT
{
    for(auto& cache : m_Caches)
    {
        if (cache.id.uuid == util::InvalidUuid)
        {
            return &cache;
        }
    }
    // 空いているものがない場合、カウンタが一番小さいものを使う
    {
        Cache* cache = &(m_Caches[0]);
        for (int i = 1; i < CacheCount; ++i)
        {
            if (cache->counter < m_Caches[i].counter)
            {
                cache = &(m_Caches[i]);
            }
        }
        Invalidate(cache);
        return cache;
    }
}

Result PlaceHolderAccessor::EnsurePlaceHolderDirectory(PlaceHolderId id) NN_NOEXCEPT
{
    PathString filePath;
    MakePlaceHolderFilePath(&filePath, id, m_Func, *m_RootPath);
    NN_RESULT_DO(detail::EnsureParentDirectoryRecursively(filePath));

    NN_RESULT_SUCCESS;
}

}}
