﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_MakeContentPathFunction.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {

    namespace {
        void MakeContentName(PathString* outValue, ContentId id) NN_NOEXCEPT
        {
            char idString[ContentIdStringLength + 1];
            GetStringFromContentId(idString, sizeof(idString), id);

            outValue->AssignFormat("%s.nca", idString);
        }
        Bit16 GenerateBit16Sha256ContentIdHashHead(ContentId id)
        {
            Bit8 hash[crypto::Sha256Generator::HashSize];
            crypto::GenerateSha256Hash(hash, sizeof(hash), &id, sizeof(id));
            return static_cast<Bit16>(hash[0]) + (static_cast<Bit16>(hash[1]) << 8);
        }
        Bit8 GenerateBit8Sha256ContentIdHashHead(ContentId id)
        {
            Bit8 hash[crypto::Sha256Generator::HashSize];
            crypto::GenerateSha256Hash(hash, sizeof(hash), &id, sizeof(id));
            return hash[0];
        }
    }

    void MakeFlatContentFilePath(PathString* outValue, ContentId id, const char* contentRootPath)
    {
        PathString contentName;
        MakeContentName(&contentName, id);
        outValue->AssignFormat("%s/%s", contentRootPath, contentName.Get());
    }

    void MakeSha256HierarchicalContentFilePath_ForFat4KCluster(PathString* outValue, ContentId id, const char* contentRootPath)
    {
        auto hashHead = GenerateBit16Sha256ContentIdHashHead(id);
        Bit32 hash0 = (hashHead >> 10) & 0x3F;
        Bit32 hash1 = (hashHead >> 4) & 0x3F;

        PathString contentName;
        MakeContentName(&contentName, id);

        outValue->AssignFormat("%s/%08X/%08X/%s", contentRootPath, hash0, hash1, contentName.Get());
    }

    void MakeSha256HierarchicalContentFilePath_ForFat32KCluster(PathString* outValue, ContentId id, const char* contentRootPath)
    {
        auto hashHead = GenerateBit16Sha256ContentIdHashHead(id);
        Bit32 hash = (hashHead >> 6) & 0x3FF;

        PathString contentName;
        MakeContentName(&contentName, id);

        outValue->AssignFormat("%s/%08X/%s", contentRootPath, hash, contentName.Get());
    }

    void MakeSha256HierarchicalContentFilePath_ForFat16KCluster(PathString* outValue, ContentId id, const char* contentRootPath)
    {
        Bit32 hash = GenerateBit8Sha256ContentIdHashHead(id);

        PathString contentName;
        MakeContentName(&contentName, id);

        outValue->AssignFormat("%s/%08X/%s", contentRootPath, hash, contentName.Get());
    }

    int GetHierarchicalContentDirectoryDepth(MakeContentPathFunction func) NN_NOEXCEPT
    {
        if (func == MakeFlatContentFilePath)
        {
            return 1;
        }
        else if (func == MakeSha256HierarchicalContentFilePath_ForFat4KCluster)
        {
            return 3;
        }
        else if (func == MakeSha256HierarchicalContentFilePath_ForFat32KCluster ||
                 func == MakeSha256HierarchicalContentFilePath_ForFat16KCluster)
        {
            return 2;
        }
        else
        {
            NN_ABORT();
        }
    }
}}
