﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_InstallTaskData.h>
#include <nn/kvdb/kvdb_BoundedString.h>

namespace nn { namespace ncm {
    namespace {
        typedef kvdb::BoundedString<64> BoundedPath;

        bool Includes(const ContentMetaKey& key, const ContentMetaKey list[], int count) NN_NOEXCEPT
        {
            for (int i = 0; i < count; i++)
            {
                if (key == list[i])
                {
                    return true;
                }
            }

            return false;
        }
    }

    Result InstallTaskDataBase::Get(InstallContentMeta* outValue, int index) NN_NOEXCEPT
    {
        size_t dataSize;
        NN_RESULT_DO(GetSize(&dataSize, index));

        std::unique_ptr<char[]> dataBuffer(new char[dataSize]);
        NN_RESULT_THROW_UNLESS(dataBuffer, ResultAllocationMemoryFailed());;
        NN_RESULT_DO(Get(index, dataBuffer.get(), dataSize));

        outValue->data = std::move(dataBuffer);
        outValue->size = dataSize;

        NN_RESULT_SUCCESS;
    }

    Result InstallTaskDataBase::Update(const InstallContentMeta& data, int index) NN_NOEXCEPT
    {
        return Update(index, data.data.get(), data.size);
    }

    Result InstallTaskDataBase::Has(bool* outValue, Bit64 id) NN_NOEXCEPT
    {
        int count;
        NN_RESULT_DO(Count(&count));
        for (int i = 0; i < count; i++)
        {
            InstallContentMeta data;
            NN_RESULT_DO(Get(&data, i));
            if (data.GetReader().GetKey().id == id)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    MemoryInstallTaskData::MemoryInstallTaskData() NN_NOEXCEPT : m_State(InstallProgressState::NotPrepared), m_LastResult(ResultSuccess()) {}

    MemoryInstallTaskData::~MemoryInstallTaskData() NN_NOEXCEPT
    {
        Cleanup();
    }

    Result MemoryInstallTaskData::GetProgress(InstallProgress* outValue) NN_NOEXCEPT
    {
        InstallProgress progress = {};
        progress.state = m_State;
        std::memcpy(&progress.lastResult, &m_LastResult, sizeof(m_LastResult));
        if (m_State != InstallProgressState::NotPrepared &&
            m_State != InstallProgressState::DataPrepared)
        {
            for (auto& data : m_DataList)
            {
                auto reader = data.GetReader();
                auto count = reader.CountContent();
                for (int i = 0; i < count; i++)
                {
                    auto info = reader.GetContentInfo(i);
                    progress.installedSize += info->written;
                    progress.totalSize += info->info.GetSize();
                }
            }
        }

        *outValue = progress;
        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT
    {
        *outValue = m_SystemUpdateTaskApplyInfo;
        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::SetState(InstallProgressState state) NN_NOEXCEPT
    {
        m_State = state;
        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::SetLastResult(Result result) NN_NOEXCEPT
    {
        m_LastResult = result;
        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::SetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo info) NN_NOEXCEPT
    {
        m_SystemUpdateTaskApplyInfo = info;
        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::Push(const void* data, size_t size) NN_NOEXCEPT
    {
        auto dataHolder = std::unique_ptr<DataHolder>(new DataHolder());
        NN_RESULT_THROW_UNLESS(dataHolder, ResultAllocationMemoryFailed());

        dataHolder->data = std::unique_ptr<char[]>(new char[size]);
        NN_RESULT_THROW_UNLESS(dataHolder->data, ResultAllocationMemoryFailed());
        dataHolder->size = size;

        std::memcpy(dataHolder->data.get(), data, size);

        m_DataList.push_back(*dataHolder.get());
        dataHolder.release();

        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::Count(int* outValue) NN_NOEXCEPT
    {
        *outValue = m_DataList.size();
        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::GetSize(size_t* outValue, int index) NN_NOEXCEPT
    {
        int count = 0;
        for (auto& data : m_DataList)
        {
            if (count == index)
            {
                *outValue = data.size;
                NN_RESULT_SUCCESS;
            }

            count++;
        }

        NN_ABORT("must not come here.");
    }

    Result MemoryInstallTaskData::Get(int index, void* buffer, size_t size) NN_NOEXCEPT
    {
        int count = 0;
        for (auto& data : m_DataList)
        {
            if (count == index)
            {
                NN_RESULT_THROW_UNLESS(data.size <= size, ResultBufferNotEnough());
                std::memcpy(buffer, data.data.get(), data.size);
                NN_RESULT_SUCCESS;
            }

            count++;
        }

        NN_ABORT("must not come here.");
    }

    Result MemoryInstallTaskData::Update(int index, const void* buffer, size_t size) NN_NOEXCEPT
    {
        int count = 0;
        for (auto& data : m_DataList)
        {
            if (count == index)
            {
                NN_RESULT_THROW_UNLESS(size == data.size, ResultBufferNotEnough());
                std::memcpy(data.data.get(), buffer, size);
                NN_RESULT_SUCCESS;
            }

            count++;
        }

        NN_ABORT("must not come here.");
    }

    Result MemoryInstallTaskData::Delete(const ContentMetaKey list[], int count) NN_NOEXCEPT
    {
        for (int i = 0; i < count; i++)
        {
            auto& key = list[i];
            for (auto& data : m_DataList)
            {
                if (key == data.GetReader().GetKey())
                {
                    m_DataList.erase(m_DataList.iterator_to(data));
                    delete &data;
                    break;
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result MemoryInstallTaskData::Cleanup() NN_NOEXCEPT
    {
        while (!m_DataList.empty())
        {
            auto& data = m_DataList.front();
            m_DataList.pop_front();
            delete &data;
        }

        NN_RESULT_SUCCESS;
    }

    FileInstallTaskData::Header FileInstallTaskData::MakeInitialHeader(int maxEntryCount) NN_NOEXCEPT
    {
        Header header = {
            static_cast<uint32_t>(maxEntryCount),
            0,
            static_cast<int64_t>(sizeof(Header)) + static_cast<int64_t>(sizeof(EntryInfo)) * maxEntryCount,
            ResultSuccess(),
            InstallProgressState::NotPrepared,
            SystemUpdateTaskApplyInfo::CannotJudgeYet
        };

        return header;
    }

    Result FileInstallTaskData::Create(const char* path, int maxEntryCount) NN_NOEXCEPT
    {
        NN_RESULT_DO(fs::CreateFile(path, 0));
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        auto header = MakeInitialHeader(maxEntryCount);
        NN_RESULT_DO(fs::WriteFile(file, 0, &header, sizeof(header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Initialize(const char* path) NN_NOEXCEPT
    {
        util::Strlcpy(m_Path, path, static_cast<int>(sizeof(m_Path)));
        return Read(&m_Header, sizeof(m_Header), 0);
    }


    Result FileInstallTaskData::GetProgress(InstallProgress* outValue) NN_NOEXCEPT
    {
        InstallProgress progress = {};
        progress.state = m_Header.state;
        std::memcpy(&progress.lastResult, &m_Header.lastResult, sizeof(m_Header.lastResult));
        if (m_Header.state != InstallProgressState::NotPrepared &&
            m_Header.state != InstallProgressState::DataPrepared)
        {
            for (uint32_t i = 0; i < m_Header.entryCount; i++)
            {
                EntryInfo info;
                NN_RESULT_DO(GetEntryInfo(&info, i));
                auto dataSize = static_cast<size_t>(info.size);

                std::unique_ptr<char[]> data(new char[dataSize]);
                NN_RESULT_THROW_UNLESS(data, ResultAllocationMemoryFailed());;

                NN_RESULT_DO(Read(data.get(), dataSize, info.offset));

                InstallContentMetaReader reader(data.get(), dataSize);
                auto count = reader.CountContent();
                for (int j = 0; j < count; j++)
                {
                    auto contentInfo = reader.GetContentInfo(j);
                    progress.installedSize += contentInfo->written;
                    progress.totalSize += contentInfo->info.GetSize();
                }
            }
        }

        *outValue = progress;
        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT
    {
        *outValue = m_Header.applyInfo;
        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::SetState(InstallProgressState state) NN_NOEXCEPT
    {
        m_Header.state = state;
        NN_RESULT_DO(WriteHeader());

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::SetLastResult(Result result) NN_NOEXCEPT
    {
        m_Header.lastResult = result;
        NN_RESULT_DO(WriteHeader());

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::SetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo info) NN_NOEXCEPT
    {
        m_Header.applyInfo = info;
        NN_RESULT_DO(WriteHeader());

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Push(const void* data, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Header.entryCount < m_Header.maxEntryCount, ResultBufferNotEnough());
        EntryInfo info = { m_Header.lastDataOffset, static_cast<int64_t>(size) };

        NN_RESULT_DO(Write(&info, sizeof(info), GetEntryInfoOffset(m_Header.entryCount)));
        NN_RESULT_DO(Write(data, size, info.offset));

        m_Header.lastDataOffset += size;
        m_Header.entryCount++;

        NN_RESULT_DO(WriteHeader());

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Count(int* outValue) NN_NOEXCEPT
    {
        *outValue = m_Header.entryCount;

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::GetSize(size_t* outValue, int index) NN_NOEXCEPT
    {
        EntryInfo info;
        NN_RESULT_DO(GetEntryInfo(&info, index));

        *outValue = static_cast<size_t>(info.size);
        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Get(int index, void* buffer, size_t size) NN_NOEXCEPT
    {
        EntryInfo info;
        NN_RESULT_DO(GetEntryInfo(&info, index));
        NN_RESULT_THROW_UNLESS(info.size <= static_cast<int64_t>(size), ResultBufferNotEnough());
        NN_RESULT_DO(Read(buffer, size, info.offset));

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Update(int index, const void* buffer, size_t size) NN_NOEXCEPT
    {
        EntryInfo info;
        NN_RESULT_DO(GetEntryInfo(&info, index));
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(size) == info.size, ResultBufferNotEnough());
        NN_RESULT_DO(Write(buffer, size, info.offset));

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Delete(const ContentMetaKey list[], int count) NN_NOEXCEPT
    {
        BoundedPath path(m_Path);
        path.Append(".tmp");

        FileInstallTaskData newData;
        NN_RESULT_DO(FileInstallTaskData::Create(path, m_Header.maxEntryCount));
        NN_RESULT_DO(newData.Initialize(path));

        int dataCount;
        NN_RESULT_DO(Count(&dataCount));
        for (int i = 0; i < dataCount; i++)
        {
            size_t dataSize;
            NN_RESULT_DO(GetSize(&dataSize, i));

            std::unique_ptr<char[]> dataBuffer(new char[dataSize]);
            NN_RESULT_THROW_UNLESS(dataBuffer, ResultAllocationMemoryFailed());;
            NN_RESULT_DO(Get(i, dataBuffer.get(), dataSize));

            InstallContentMetaReader reader(dataBuffer.get(), dataSize);
            if (Includes(reader.GetKey(), list, count))
            {
                continue;
            }

            newData.Push(dataBuffer.get(), dataSize);
        }

        m_Header = newData.m_Header;
        NN_RESULT_DO(fs::DeleteFile(m_Path));
        NN_RESULT_DO(fs::RenameFile(path, m_Path));

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Cleanup() NN_NOEXCEPT
    {
        m_Header = MakeInitialHeader(m_Header.maxEntryCount);
        NN_RESULT_DO(WriteHeader());

        NN_RESULT_SUCCESS;
    }

    int64_t FileInstallTaskData::GetEntryInfoOffset(int index) NN_NOEXCEPT
    {
        return sizeof(Header) + sizeof(EntryInfo) * index;
    }

    Result FileInstallTaskData::GetEntryInfo(EntryInfo* outValue, int index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(static_cast<uint32_t>(index) < m_Header.entryCount);
        return Read(outValue, sizeof(EntryInfo), GetEntryInfoOffset(index));
    }

    Result FileInstallTaskData::Write(const void* buffer, size_t bufferSize, int64_t offset) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, m_Path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
        NN_RESULT_DO(fs::WriteFile(file, offset, buffer, bufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::Read(void* buffer, size_t bufferSize, int64_t offset) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, m_Path, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
        NN_RESULT_DO(fs::ReadFile(file, offset, buffer, bufferSize));

        NN_RESULT_SUCCESS;
    }

    Result FileInstallTaskData::WriteHeader() NN_NOEXCEPT
    {
        return Write(&m_Header, sizeof(m_Header), 0);
    }
}}
