﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_Optional.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include "ncm_ContentMetaDatabaseUtil.h"

namespace nn { namespace ncm {
    ContentMetaDatabaseImpl::ContentMetaDatabaseImpl(ContentMetaKeyValueStore* kvs, const char* mountName) NN_NOEXCEPT
        : m_Kvs(kvs), m_IsDisabled(false)
    {
        std::strcpy(m_MountName, mountName);
    }

    Result ContentMetaDatabaseImpl::Set(const ContentMetaKey& key, const sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        NN_RESULT_DO(m_Kvs->Put(key, buffer.GetPointerUnsafe(), buffer.GetSize()));

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::Get(sf::Out<::std::uint64_t> outValue, const ContentMetaKey& key, const sf::OutBuffer& buffer) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        size_t outSize;
        NN_RESULT_TRY(m_Kvs->Get(&outSize, key, buffer.GetPointerUnsafe(), buffer.GetSize()))
            NN_RESULT_CATCH(nn::kvdb::ResultKeyNotFound)
            {
                NN_RESULT_THROW(ResultContentMetaNotFound());
            }
        NN_RESULT_END_TRY

        outValue.Set(outSize);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::Remove(const ContentMetaKey& key) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        NN_RESULT_TRY(m_Kvs->Delete(key))
            NN_RESULT_CATCH(nn::kvdb::ResultKeyNotFound)
            {
                NN_RESULT_THROW(ResultContentMetaNotFound());
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetContentIdImpl(sf::Out<ContentId> outValue, const ContentMetaKey& key, ContentType type, util::optional<uint8_t> idOffset) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        auto iter = m_Kvs->GetLowerBoundIterator(key);
        NN_RESULT_THROW_UNLESS(! iter.IsEnd(), ResultContentMetaNotFound());
        NN_RESULT_THROW_UNLESS(iter.Get().id == key.id, ResultContentMetaNotFound());

        auto foundKey = iter.Get();
        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, foundKey, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        const ContentInfo* contentInfo{};
        if (idOffset)
        {
            contentInfo = reader.GetContentInfo(type, *idOffset);
        }
        else
        {
            contentInfo = reader.GetContentInfo(type);
        }
        NN_RESULT_THROW_UNLESS(contentInfo, ResultContentNotFound());

        outValue.Set(contentInfo->id);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetContentIdByType(sf::Out<ContentId> outValue, const ContentMetaKey& key, ContentType type) const NN_NOEXCEPT
    {
        return GetContentIdImpl(outValue, key, type, util::nullopt);
    }

    Result ContentMetaDatabaseImpl::GetContentIdByTypeAndIdOffset(sf::Out<ContentId> outValue, const ContentMetaKey& key, ContentType type, uint8_t idOffset) const NN_NOEXCEPT
    {
        return GetContentIdImpl(outValue, key, type, idOffset);
    }

    Result ContentMetaDatabaseImpl::ListContentInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ContentInfo>& outArray, const ContentMetaKey& key, int offset) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        int count = 0;
        for (int i = 0; i < static_cast<int>(outArray.GetLength()) && i + offset < reader.CountContent(); i++)
        {
            outArray[i] = *reader.GetContentInfo(i + offset);
            count++;
        }

        outCount.Set(count);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::ListContentMetaInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::ContentMetaInfo>& outArray, const ncm::ContentMetaKey& key, std::int32_t offset) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        int count = 0;
        for (int i = 0; i < static_cast<int>(outArray.GetLength()) && i + offset < reader.CountContentMeta(); i++)
        {
            outArray[i] = *reader.GetContentMetaInfo(i + offset);
            count++;
        }

        outCount.Set(count);
        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::List(sf::Out<std::int32_t> outTotal, sf::Out<std::int32_t> outCount, const sf::OutArray<ContentMetaKey>& outArray, ContentMetaType type, ApplicationId appId, Bit64 min, Bit64 max, ContentInstallType installType) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        auto iter = m_Kvs->GetBeginIterator();

        size_t count = 0;
        int totalCount = 0;
        for (; ! iter.IsEnd(); iter.Next())
        {
            auto iterated = iter.Get();
            if((type == ContentMetaType::Unknown || iterated.type == type) &&
                (min <= iterated.id && iterated.id <= max) &&
                (installType == ContentInstallType::Unknown || iterated.installType == installType))
            {
                if (appId.value != 0)
                {
                    size_t metaSize;
                    const void* meta;
                    NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, iterated, m_Kvs));
                    ContentMetaReader reader(meta, metaSize);

                    if (reader.GetApplicationId(iterated) && appId.value != reader.GetApplicationId(iterated)->value)
                    {
                        continue;
                    }
                }

                if(count < outArray.GetLength())
                {
                    outArray[count] = iterated;
                    count++;
                }
                totalCount++;
            }
        }

        outTotal.Set(totalCount);
        outCount.Set(static_cast<int32_t>(count)); // TODO: SIGLO-12092 でチェック

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetLatestContentMetaKey(sf::Out<ContentMetaKey> outValue, Bit64 id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        ContentMetaKey key;
        NN_RESULT_DO(FindLatestKeyById(&key, id));

        outValue.Set(key);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::ListApplication(sf::Out<std::int32_t> outTotal, sf::Out<std::int32_t> outCount, const sf::OutArray<ApplicationContentMetaKey>& outArray, ContentMetaType type) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        auto iter = m_Kvs->GetBeginIterator();

        size_t count = 0;
        int totalCount = 0;
        for (; ! iter.IsEnd(); iter.Next())
        {
            auto iterated = iter.Get();
            if(type == ContentMetaType::Unknown || iterated.type == type)
            {
                ContentMetaReader reader(iter.GetPointer(), iter.GetSize());
                auto appId = reader.GetApplicationId(iterated);
                if (appId)
                {
                    if(count < outArray.GetLength())
                    {
                        ApplicationContentMetaKey key = { iterated, *appId };
                        outArray[count] = key;
                        count++;
                    }
                    totalCount++;
                }
            }
        }

        outTotal.Set(totalCount);
        outCount.Set(static_cast<int32_t>(count));

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::Has(sf::Out<bool> outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        size_t size;
        NN_RESULT_TRY(m_Kvs->GetSize(&size, key))
            NN_RESULT_CATCH(kvdb::ResultKeyNotFound)
            {
                outValue.Set(false);
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        outValue.Set(true);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::HasAll(sf::Out<bool> outValue, const sf::InArray<ncm::ContentMetaKey>& inArray) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        for (size_t i = 0; i < inArray.GetLength(); i++)
        {
            bool has;
            NN_RESULT_DO(Has(&has, inArray[i]));
            if (!has)
            {
                outValue.Set(false);
                NN_RESULT_SUCCESS;
            }
        }
        outValue.Set(true);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::HasContent(sf::Out<bool> outValue, const ncm::ContentMetaKey& key, const ncm::ContentId& contentId) const NN_NOEXCEPT
    {
        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);
        for (int i = 0; i < reader.CountContent(); i++)
        {
            if (reader.GetContentInfo(i)->GetId() == contentId)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::FindLatestKeyById(ContentMetaKey* outValue, Bit64 id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        auto iter = m_Kvs->GetLowerBoundIterator(ContentMetaKey::MakeUnknownType(id, 0));

        util::optional<ContentMetaKey> foundKey;
        while (! iter.IsEnd())
        {
            auto iterated = iter.Get();
            if (iterated.id != id)
            {
                break;
            }
            if (iterated.installType == ContentInstallType::Full)
            {
                foundKey = iterated;
            }

            iter.Next();
        }
        NN_RESULT_THROW_UNLESS(foundKey, ResultContentMetaNotFound());

        *outValue = *foundKey;

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetSize(nn::sf::Out<std::uint64_t> outValue, const nn::ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        size_t outSize;
        NN_RESULT_DO(GetContentMetaSize(&outSize, key, m_Kvs));

        outValue.Set(outSize);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetRequiredSystemVersion(nn::sf::Out<std::uint32_t> outValue, const nn::ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());
        NN_RESULT_THROW_UNLESS(key.type == ContentMetaType::Application || key.type == ContentMetaType::Patch, ResultInvalidContentMetaKey());

        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        if (key.type == ContentMetaType::Application)
        {
            outValue.Set(reader.GetExtendedHeader<ApplicationMetaExtendedHeader>()->requiredSystemVersion);
        }
        else if (key.type == ContentMetaType::Patch)
        {
            outValue.Set(reader.GetExtendedHeader<PatchMetaExtendedHeader>()->requiredSystemVersion);
        }
        else
        {
            NN_ABORT("Must not come here.\n");
        }

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetPatchId(nn::sf::Out<nn::ncm::PatchId> outValue, const nn::ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());
        NN_RESULT_THROW_UNLESS(key.type == ContentMetaType::Application, ResultInvalidContentMetaKey());

        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        outValue.Set(reader.GetExtendedHeader<ApplicationMetaExtendedHeader>()->patchId);

        NN_RESULT_SUCCESS;
    }
    Result ContentMetaDatabaseImpl::GetAttributes(nn::sf::Out<Bit8> outAttributes, const nn::ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());

        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        outAttributes.Set(reader.GetHeader()->attributes);

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::LookupOrphanContent(const sf::OutArray<bool>& outList, const sf::InArray<ncm::ContentId>& contentList) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());
        NN_RESULT_THROW_UNLESS(outList.GetLength() >= contentList.GetLength(), ResultBufferNotEnough());

        for (size_t i = 0; i < outList.GetLength(); i++)
        {
            outList[i] = true;
        }

        auto find = [](const sf::InArray<ncm::ContentId>& list, const ncm::ContentId& id)->util::optional<size_t>
        {
            for (size_t i = 0; i < list.GetLength(); i++)
            {
                if (list[i] == id)
                {
                    return i;
                }
            }

            return util::nullopt;
        };

        for (auto iter = m_Kvs->GetBeginIterator(); !iter.IsEnd(); iter.Next())
        {
            ContentMetaReader reader(iter.GetPointer(), iter.GetSize());
            auto count = reader.CountContent();
            for (int i = 0; i < count; i++)
            {
                auto found = find(contentList, reader.GetContentInfo(i)->GetId());
                if (found)
                {
                    outList[*found] = false;
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::Commit() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());
        NN_RESULT_DO(m_Kvs->Save());
        NN_RESULT_DO(fs::CommitSaveData(m_MountName));
        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::DisableForcibly() NN_NOEXCEPT
    {
        m_IsDisabled = true;
        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseImpl::GetRequiredApplicationVersion(nn::sf::Out<std::uint32_t> outValue, const nn::ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentMetaDatabase());
        NN_RESULT_THROW_UNLESS(key.type == ContentMetaType::AddOnContent, ResultInvalidContentMetaKey());

        const void* meta;
        size_t metaSize;
        NN_RESULT_DO(GetContentMetaPointer(&meta, &metaSize, key, m_Kvs));

        ContentMetaReader reader(meta, metaSize);

        if (key.type == ContentMetaType::AddOnContent)
        {
            outValue.Set(reader.GetExtendedHeader<AddOnContentMetaExtendedHeader>()->requiredApplicationVersion);
        }
        else
        {
            NN_ABORT("Must not come here.\n");
        }

        NN_RESULT_SUCCESS;
    }
}}
