﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/detail/ncm_Log.h>
#include <nn/os/os_Mutex.h>

#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {
    namespace {
        static const int MaxContentMetaCountPerId = 64;
        static const int MaxPathLength = 256;

        Result ConvertToFsCommonPath(char* outPath, size_t length, const char* packageRootPath, const char* entryPath) NN_NOEXCEPT
        {
            char packagePath[MaxPathLength];

            auto packagePathLength = util::SNPrintf(packagePath, sizeof(packagePath), "%s%s", packageRootPath, entryPath);
            NN_ABORT_UNLESS_LESS(packagePathLength, MaxPathLength);
            NN_RESULT_DO(fs::ConvertToFsCommonPath(outPath, length, packagePath));

            NN_RESULT_SUCCESS;
        }

        Result LoadContentMeta(ncm::AutoBuffer* outBuffer, const char* packageRootPath, const fs::DirectoryEntry entry) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(detail::PathView(entry.name).HasSuffix(".cnmt.nca"));
            char path[MaxPathLength];
            NN_RESULT_DO(ConvertToFsCommonPath(path, sizeof(path), packageRootPath, entry.name));
            NN_RESULT_DO(ncm::ReadContentMetaPath(outBuffer, path));
            NN_RESULT_SUCCESS;
        }

        template <class FuncT>
        Result ForEachDirectory(const char* rootPath, FuncT func) NN_NOEXCEPT
        {
            fs::DirectoryHandle directory;
            NN_RESULT_DO(fs::OpenDirectory(&directory, rootPath, fs::OpenDirectoryMode_File));
            NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(directory); };

            while (NN_STATIC_CONDITION(true))
            {
                fs::DirectoryEntry entry;
                int64_t count;
                NN_RESULT_DO(fs::ReadDirectory(&count, &entry, directory, 1));
                if (count == 0)
                {
                    break;
                }

                bool isEnd;
                NN_RESULT_DO(func(&isEnd, entry));
                if (isEnd)
                {
                    NN_RESULT_SUCCESS;
                }
            }

            NN_RESULT_SUCCESS;
        }
    }

    Result ContentManagerAccessor::DeleteRedundantAll(const ncm::ContentMetaKey& latest) NN_NOEXCEPT
    {
        ContentMetaKey metaList[MaxContentMetaCountPerId];
        auto listCount = m_Db->ListContentMeta(metaList, sizeof(metaList) / sizeof(metaList[0]), ContentMetaType::Unknown, ContentMetaDatabase::AnyApplicationId, latest.id, latest.id);
        for (int i = 0; i < listCount.listed; i++)
        {
            auto& key = metaList[i];
            if (key < latest)
            {
                NN_RESULT_DO(DeleteRedundant(key, &latest));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ContentManagerAccessor::DeleteAll(Bit64 id) NN_NOEXCEPT
    {
        ContentMetaKey metaList[MaxContentMetaCountPerId];
        auto listCount = m_Db->ListContentMeta(metaList, sizeof(metaList) / sizeof(metaList[0]), ContentMetaType::Unknown, ContentMetaDatabase::AnyApplicationId, id, id);
        for (int i = 0; i < listCount.listed; i++)
        {
            NN_RESULT_DO(DeleteRedundant(metaList[i], nullptr));
        }

        NN_RESULT_SUCCESS;
    }


    Result ContentManagerAccessor::DeleteRedundant(const ncm::ContentMetaKey& redundant, const ncm::ContentMetaKey* latest) NN_NOEXCEPT
    {
        static os::Mutex s_Mutex(false);
        std::lock_guard<os::Mutex> guard(s_Mutex);
        static ncm::ContentInfo buffer[MaxContentInfoCount];

        // MaxContentInfoCount 以上のコンテンツはないはず
        // 仮にあったとしたら孤児コンテンツになるが、どこかのタイミングで消えるはず
        int offset = 0;
        int outCount;
        NN_RESULT_DO(m_Db->ListContentInfo(&outCount, buffer, MaxContentInfoCount, redundant, offset));

        NN_RESULT_DO(m_Db->Remove(redundant));

        for (int i = 0; i < outCount; i++)
        {
            auto& contentId = buffer[i].id;

            bool hasInLatest;
            if (latest)
            {
                NN_RESULT_DO(m_Db->HasContent(&hasInLatest, *latest, contentId));
            }
            else
            {
                hasInLatest = false;
            }

            if (!hasInLatest)
            {
                NN_RESULT_TRY(m_Storage->Delete(contentId))
                    NN_RESULT_CATCH(ncm::ResultContentNotFound) {}
                NN_RESULT_END_TRY
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseBuilder::BuildFromPackage(const char* packageRoot) NN_NOEXCEPT
    {
        NN_DETAIL_NCM_TRACE("[ContentMetaDatabaseBuilder] Build from package %s\n", packageRoot);

        NN_RESULT_DO(ForEachDirectory(packageRoot, [this, &packageRoot](bool* outValue, fs::DirectoryEntry entry) NN_NOEXCEPT -> Result
        {
            *outValue = false;

            if (! detail::PathView(entry.name).HasSuffix(".cnmt.nca"))
            {
                NN_RESULT_SUCCESS;
            }

            ncm::AutoBuffer packageMeta;
            NN_RESULT_DO(LoadContentMeta(&packageMeta, packageRoot, entry));

            auto contentId = GetContentIdFromString(entry.name, sizeof(entry.name));
            NN_RESULT_THROW_UNLESS(contentId, ResultInvalidPackageFormat());
            NN_RESULT_DO(BuildFromPackageContentMeta(packageMeta.Get(), packageMeta.GetSize(), ContentInfo::Make(*contentId, entry.fileSize, ContentType::Meta, 0)));

            NN_RESULT_SUCCESS;

        }));

        NN_RESULT_DO(m_Db->Commit());

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseBuilder::Cleanup() NN_NOEXCEPT
    {
        NN_DETAIL_NCM_TRACE("[ContentMetaDatabaseBuilder] Cleanup\n");

        while (NN_STATIC_CONDITION(true))
        {
            static const int KeyCount = 64;
            ncm::ContentMetaKey keyList[KeyCount];
            auto listCount = m_Db->ListContentMeta(keyList, KeyCount);
            for (int i = 0; i < listCount.listed; i++)
            {
                NN_RESULT_DO(m_Db->Remove(keyList[i]));
            }
            if (listCount.listed < KeyCount)
            {
                break;
            }
        }

        NN_RESULT_DO(m_Db->Commit());

        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabaseBuilder::BuildFromPackageContentMeta(void* buffer, size_t size, const ContentInfo& metaInfo) NN_NOEXCEPT
    {
        ncm::PackagedContentMetaReader packageMetaReader(buffer, size);

        size_t metaSize = packageMetaReader.CalculateConvertContentMetaSize();
        std::unique_ptr<char[]> meta(new char[metaSize]);

        packageMetaReader.ConvertToContentMeta(meta.get(), metaSize, metaInfo);
        ncm::ContentMetaReader metaReader(meta.get(), metaSize);

        auto key = packageMetaReader.GetKey();
        NN_RESULT_DO(m_Db->Set(key, metaReader.GetData(), metaReader.GetSize()));

        NN_DETAIL_NCM_TRACE("[ContentMetaDatabaseBuilder] Build 0x%016llx version %u size %zu\n", key.id, key.version, metaReader.GetSize());

        NN_RESULT_SUCCESS;
    }

    Result ListApplicationFromPackage(int* outCount, ApplicationId* outList, int numList, const char* packageRoot) NN_NOEXCEPT
    {
        int count = 0;
        NN_RESULT_DO(ForEachDirectory(packageRoot, [&packageRoot, &outList, &numList, &count](bool* outValue, fs::DirectoryEntry entry) NN_NOEXCEPT -> Result
        {
            *outValue = false;

            if (! detail::PathView(entry.name).HasSuffix(".cnmt.nca"))
            {
                NN_RESULT_SUCCESS;
            }

            ncm::AutoBuffer packageMeta;
            NN_RESULT_DO(LoadContentMeta(&packageMeta, packageRoot, entry));

            ncm::PackagedContentMetaReader packageMetaReader(packageMeta.Get(), packageMeta.GetSize());
            const auto& key = packageMetaReader.GetKey();
            if (key.type == ncm::ContentMetaType::Application)
            {
                NN_RESULT_THROW_UNLESS(count < numList, ResultBufferNotEnough());
                ApplicationId appId = { key.id };
                outList[count] = appId;
                count++;
            }

            NN_RESULT_SUCCESS;
        }));

        *outCount = count;
        NN_RESULT_SUCCESS;
    }
}}
