﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentIdUtil.h>

namespace nn { namespace ncm {
    namespace {
        bool ConvertBinaryFromString(Bit8* buffer, size_t bufferSize, const char* str, size_t length) NN_NOEXCEPT
        {
            if ((length % 2 != 0) || (bufferSize * 2 < length))
            {
                return false;
            }

            for (size_t i = 0; i < length; i += 2)
            {
                char byteString[3];
                char* errptr = NULL;
                util::Strlcpy(byteString, &str[i], sizeof(byteString));

                buffer[i / 2] = static_cast<Bit8>(std::strtoul(byteString, &errptr, 16));
                if (*errptr != '\0')
                {
                    return false;
                }
            }

            return true;
        }
    }

    ContentIdString GetContentIdString(ContentId id) NN_NOEXCEPT
    {
        ContentIdString str;

        for (size_t i = 0; i < sizeof(id.data); i++)
        {
            util::SNPrintf(&(str.data[i * 2]), 3, "%02x", id.data[i]);
        }

        return str;
    }

    void GetStringFromContentId(char* outValue, size_t length, ContentId id) NN_NOEXCEPT
    {
        NN_UNUSED(length);
        NN_SDK_REQUIRES(length > ContentIdStringLength);

        for (size_t i = 0; i < sizeof(id.data); i++)
        {
            util::SNPrintf(&(outValue[i * 2]), 3, "%02x", id.data[i]);
        }
    }

    void GetStringFromRightsId(char* outValue, size_t length, nn::fs::RightsId id) NN_NOEXCEPT
    {
        NN_UNUSED(length);
        NN_SDK_REQUIRES(length > RightsIdStringLength);

        for (size_t i = 0; i < sizeof(id.data); i++)
        {
            util::SNPrintf(&(outValue[i * 2]), 3, "%02x", id.data[i]);
        }
    }

    void GetTicketFileStringFromRightsId(char* outValue, size_t length, nn::fs::RightsId id) NN_NOEXCEPT
    {
        NN_UNUSED(length);
        NN_SDK_REQUIRES(length > TicketFileStringLength);

        ContentIdString str;
        GetStringFromRightsId(str.data, sizeof(str), id);
        util::SNPrintf(outValue, length, "%s.tik", str.data);
    }
    void GetCertificateFileStringFromRightsId(char* outValue, size_t length, nn::fs::RightsId id) NN_NOEXCEPT
    {
        NN_UNUSED(length);
        NN_SDK_REQUIRES(length > CertFileStringLength);

        ContentIdString str;
        GetStringFromRightsId(str.data, sizeof(str), id);
        util::SNPrintf(outValue, length, "%s.cert", str.data);
    }

    util::optional<ContentId> GetContentIdFromString(const char* string, size_t length) NN_NOEXCEPT
    {
        if (length < ContentIdStringLength)
        {
            return util::nullopt;
        }

        ContentId contentId;
        return ConvertBinaryFromString(contentId.data, sizeof(contentId.data), string, ContentIdStringLength) ? util::optional<ContentId>(contentId) : util::nullopt;
    }

    util::optional<Hash> GetHashFromString(const char* string, size_t length) NN_NOEXCEPT
    {
        if (length < sizeof(Hash) * 2)
        {
            return util::nullopt;
        }

        Hash hash;
        return ConvertBinaryFromString(hash.data, sizeof(hash.data), string, sizeof(Hash) * 2) ? util::optional<Hash>(hash) : util::nullopt;
    }
}}
