﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <complex>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/mbuf/mbuf_Mbuf.h>
#include "mbuf_MbufAllocation.h"
#include "mbuf_MbufImpl.h"

namespace nn { namespace mbuf { namespace
{
    NN_STATIC_ASSERT(offsetof(Mbuf, _data) == MbufHeaderSize);

    Mbuf* AllocateMbuf(size_t len, MbufAllocationMode how, int type,
                       AllocationSizePolicty policy) NN_NOEXCEPT
    {
        Mbuf* pMbuf = NULL;
        if( how == MbufAllocationMode_Wait )
        {
            pMbuf = Allocate(len, policy, type);
        }
        else
        {
            pMbuf = TryAllocate(len, policy, type);
        }

        if (pMbuf != nullptr)
        {
            SetTypeMbuf(pMbuf, type);
        }
        return pMbuf;
    }

    inline void FreeMbuf(Mbuf* pMbuf) NN_NOEXCEPT
    {
        Free(pMbuf);
    }

    Mbuf* UnlinkAndFreeMbuf(Mbuf* pMbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pMbuf);
        /* 次を覚えておいてリンクからmbufを切り離し */
        Mbuf* pNextMbuf = UnlinkMbuf(pMbuf);

        /* 解放 */
        Free(pMbuf);

        /* 次へ */
        return pNextMbuf;
    }

    int GetFreeCount( int type ) NN_NOEXCEPT
    {
        return GetFreeCountFromPool(type);
    }

}}} // end of namespace nn::mbuf::<unnamed>

namespace nn { namespace mbuf
{
    Mbuf* MbufGetm(Mbuf* orig, size_t len, MbufAllocationMode how, int type) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(orig == nullptr, "orig is not implemented.");
        NN_UNUSED(orig);

        Mbuf* top_mbuf;
        Mbuf* prev_mbuf;
        Mbuf* mbuf;

        top_mbuf  = NULL;
        prev_mbuf = NULL;
        size_t allocated = 0;
        while(allocated < len)
        {
            mbuf = AllocateMbuf(len - allocated, how, type, AllocationSizePolicty_NotRequired);
            if( mbuf == NULL )
            {
                if( top_mbuf != NULL )
                {
                    MbufFreem(top_mbuf);
                }
                return NULL;
            }

            if( prev_mbuf != NULL )
            {
                LinkMbuf(prev_mbuf, mbuf);
            }

            if( top_mbuf == NULL )
            {
                /* 新mbufの先頭を覚えておく */
                top_mbuf = mbuf;
            }
            allocated += GetCapacityMbuf(mbuf);
            prev_mbuf = mbuf;
        }
        return top_mbuf;
    }


    void MbufFreem(Mbuf* pMbuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);

        // 既に解放された mbuf でないことを確認します。
        MbufPacketHeaderFlag flag = GetFlagsMbuf(pMbuf);
        if ((flag & MbufPacketHeaderFlag_FreeList) != 0)
        {
            NN_SDK_ASSERT(false, "Freeing of already freed mbuf");
            return;
        }

        // チェイン分だけループ
        Mbuf* pWorkMbuf = pMbuf;
        Mbuf* pMbufNext;
        while( pWorkMbuf != NULL )
        {
            pMbufNext = UnlinkMbuf(pWorkMbuf);
            FreeMbuf(pWorkMbuf);
            pWorkMbuf = pMbufNext;
        }
        return;
    }

    bool MbufAdj(Mbuf* pMbuf, int len) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);

        // データサイズよりも大きなトリムサイズは指定できません。
        size_t totalLen = MbufLength(pMbuf, NULL);
        size_t absLen   = ::std::abs(len);
        if( totalLen < absLen )
        {
            return false;
        }

        // 先頭から削除します。
        if( len > 0 )
        {
            // チェインは残す必要があるため、データが空であることだけ記録します。
            Mbuf* pNow = pMbuf;
            while( absLen > GetLengthMbuf(pNow) )
            {
                absLen -= GetLengthMbuf(pNow);
                SetLengthMbuf(pNow, 0);
                SetTopMbuf(pNow, 0);
                pNow = GetNextMbuf(pNow);
            }

            // 先頭ポインタを進めることでトリム扱いとします。
            SetTopMbuf(pNow, GetTopMbuf(pNow) + absLen);
            SetLengthMbuf(pNow, GetLengthMbuf(pNow) - absLen);
        }
        else if( len < 0 )
        {
            // まず最後尾の mbuf を発見します。
            Mbuf* pNow = GetTailMbuf(pMbuf);

            // 最後尾から順にトリムしていきます。
            // ただし、空になった mbuf は (先頭でなければ) 解放します。
            while (0 < absLen)
            {
                size_t dataLength = GetLengthMbuf(pNow);
                size_t trimLength = std::min(dataLength, absLen);
                absLen -= trimLength;
                SetLengthMbuf(pNow, dataLength - trimLength);

                Mbuf* pPrev = GetPreviousMbuf(pNow);
                if(pPrev != nullptr && GetLengthMbuf(pNow) == 0)
                {
                    UnlinkMbuf(pPrev);
                    FreeMbuf(pNow);
                }
                pNow = pPrev;
            }
        }
        return true;
    }

    bool MbufAppend(Mbuf* pMbuf, size_t len, const void* cp) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        NN_SDK_REQUIRES_NOT_NULL(cp);
        return MbufCopyback(pMbuf, MbufLength(pMbuf, nullptr), len, cp);
    }

    Mbuf* MbufPrependWithAllocation(Mbuf* pMbuf, size_t len, MbufAllocationMode how) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);

        Mbuf* pWorkMbuf = pMbuf;
        Mbuf* newMbuf;

        // まず先頭に挿入する追加分の mbuf を確保します。
        const int type = GetTypeMbuf(pMbuf);
        newMbuf = AllocateMbuf(len, how, type, AllocationSizePolicty_Required);
        if( newMbuf == nullptr )
        {
            return nullptr;
        }

        // 必要最小限の情報をコピーします。
        SetProtocolMbuf(newMbuf, GetProtocolMbuf(pWorkMbuf));
        SetFlagsMbuf(newMbuf, GetFlagsMbuf(pWorkMbuf));

        // 指定されたデータサイズを設定します。
        SetLengthMbuf(newMbuf, len);

        // 与えられた mbuf チェインの先頭に割り当てた mbuf を結合します。
        LinkMbuf(newMbuf, pWorkMbuf);
        return newMbuf;
    }

    Mbuf* MbufPrepend(Mbuf* pMbuf, size_t len, MbufAllocationMode how) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        if (pMbuf->_top >= len)
        {
            pMbuf->_top -= static_cast<uint16_t>(len);
            pMbuf->_len += static_cast<uint16_t>(len);
            return pMbuf;
        }
        else
        {
            return MbufPrependWithAllocation(pMbuf, len, how);
        }
    }

    Mbuf* MbufPullup(Mbuf* pMbuf, size_t len) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);

        Mbuf*  pWorkMbuf = pMbuf;
        Mbuf*  newMbuf;
        size_t copyLen;
        int    protocol;
        MbufPacketHeaderFlag flags;
        int type;

        //NDEBUG_Printf("MbufPullup: %08x len = %d\n", pWorkMbuf, len);
        if( len <= GetLengthMbuf(pWorkMbuf) )
        {
            /* そのままで大丈夫 */
            return pWorkMbuf;
        }

        size_t totalLen = MbufLength(pWorkMbuf, NULL);
        if( len > pWorkMbuf->_capacity || len > totalLen )
        {
            /* それは無理 */
            MbufFreem(pWorkMbuf);
            //NN_LOG_WARN("MbufPullup: invalid len = %d, pWorkMbuf_len=%d\n", len, totalLen);
            return NULL;
        }

        /* 情報の退避 */
        protocol = GetProtocolMbuf(pWorkMbuf);
        flags    = GetFlagsMbuf(pWorkMbuf);
        type     = GetTypeMbuf(pWorkMbuf);
        /* まずm_lenがゼロのpWorkMbufを全て解放 */
        while( pWorkMbuf != NULL )
        {
            if( GetLengthMbuf(pWorkMbuf) > 0 )
            {
                break;
            }
            pWorkMbuf = UnlinkAndFreeMbuf(pWorkMbuf);
        }
        NN_SDK_ASSERT(pWorkMbuf != NULL);

        if( len <= GetLengthMbuf(pWorkMbuf) )
        {
            /* そのままで大丈夫 */
            //NDEBUG_Printf("MbufPullup: no alloc pWorkMbuf:%d, len:%d\n", len, pWorkMbuf->GetLength());

            /* 情報の回復 */
            SetProtocolMbuf(pWorkMbuf, protocol);
            SetFlagsMbuf(pWorkMbuf, flags);
            return pWorkMbuf;
        }

        /* 集めたデータを保存するmbufを確保 */
        newMbuf = MbufGetm(NULL, len, MbufAllocationMode_Wait, type);
        if( newMbuf == NULL )
        {
            MbufFreem(pWorkMbuf);
            //NDEBUG_Printf("MbufPullup: no memory len=%d\n", len);
            return NULL;
        }

        /* 情報の回復 */
        SetProtocolMbuf(pWorkMbuf, protocol);
        SetFlagsMbuf(pWorkMbuf, flags);

        SetLengthMbuf(newMbuf, 0);    /* 一旦0にしておく */
        while( len > 0 )
        {
            //NDEBUG_Printf("MbufPullup: pWorkMbuf=%08x top=%d, len=%d\n", pWorkMbuf, pWorkMbuf->m_top, pWorkMbuf->GetLength());
            if( GetLengthMbuf(pWorkMbuf) > 0 )
            {
                /* データを集める */
                copyLen = (GetLengthMbuf(pWorkMbuf) > len) ? len : GetLengthMbuf(pWorkMbuf);
                // TORIAEZU: nn::nstd::MemCpy から std::memcpy に変更
                Bit8* pTop = static_cast<Bit8*>(GetTopPtrMbuf(newMbuf));
                std::memcpy(&(pTop[GetLengthMbuf(newMbuf)]), GetTopPtrMbuf(pWorkMbuf), copyLen);
                SetLengthMbuf(newMbuf, GetLengthMbuf(newMbuf) + copyLen);
                len -= copyLen;
                SetTopMbuf(pWorkMbuf, GetTopMbuf(pWorkMbuf) + copyLen);
                SetLengthMbuf(pWorkMbuf, GetLengthMbuf(pWorkMbuf) - copyLen);
            }

            /* 上記処理でm_lenがゼロになったら解放していく */
            if( GetLengthMbuf(pWorkMbuf) == 0 )
            {
                pWorkMbuf = UnlinkAndFreeMbuf(pWorkMbuf);
                NN_SDK_ASSERT((len == 0) || (pWorkMbuf != NULL));/*newMbufへのコピーが完了していない場合(=lenが0より大きい場合)は、次のmbufが必ず存在する。*/
            }
        }

        /* 新しく作ったmbufに、残っているmbufをリンクさせる */
        if( pWorkMbuf != NULL )
        {
            LinkMbuf(newMbuf, pWorkMbuf);
        }
        return newMbuf;
    }

    Mbuf* MbufDup(const Mbuf* pMbuf, MbufAllocationMode how) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        const Mbuf* pMbufSrc     = pMbuf;
        Mbuf* pMbufDst           = NULL;
        Mbuf* pMbufDstTop        = NULL;
        Mbuf* pMbufDstPrev       = NULL;

        while( pMbufSrc != NULL )
        {
            pMbufDst = AllocateMbuf(static_cast<int32_t>(GetLengthMbuf(pMbufSrc)), how,
                                    GetTypeMbuf(pMbufSrc), AllocationSizePolicty_Required);
            if ( pMbufDst == NULL )
            {
                if (pMbufDstTop != NULL )
                {
                    MbufFreem(pMbufDstTop);
                }
                return NULL;
            }
            CopyFromMbuf(pMbufDst, pMbufSrc);

            //NDEBUG_Printf("MbufDup: new=%08x(%d), org=%08x(%d) len=%d\n", newMbuf, newMbuf->m_top, mbuf, mbuf->m_top, mbuf->m_len);

            if( pMbufDstPrev != NULL )
            {
                LinkMbuf(pMbufDstPrev, pMbufDst);
            }

            pMbufDstPrev = pMbufDst;
            if( pMbufDstTop == NULL )
            {
                pMbufDstTop = pMbufDst;
            }

            pMbufSrc = GetNextMbuf(pMbufSrc);
        }

        return pMbufDstTop;
    }

    bool MbufCopydata(const Mbuf* pMbuf, size_t offset, size_t len, void* buf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        NN_SDK_REQUIRES_NOT_NULL(buf);

        const Mbuf* pWorkMbuf = pMbuf;

        /* コピーすべき mbuf を探す */
        while( NN_STATIC_CONDITION(true) )
        {
            if( pWorkMbuf == NULL )
            {
                /* 終端 */
                //NN_LOG_WARN("MbufCopydata: Could not find the head.\n");
                return false;
            }
            if( offset < GetLengthMbuf(pWorkMbuf) )
            {
                /* コピー対象の先頭の mbuf を発見 */
                break;
            }
            offset -= GetLengthMbuf(pWorkMbuf);
            pWorkMbuf = GetNextMbuf(pWorkMbuf);
        }

        /* コピー開始 */
        size_t dstOffset = 0;
        size_t restLen = len;
        while( restLen > 0 )
        {
            if( pWorkMbuf == NULL )
            {
                if( len == MbufCopyall )
                {
                    break;
                }
                else
                {
                    /* コピー中に終端にたどり着いてしまった */
                    //NN_LOG_WARN("MbufCopydata: too short (rest = %d)\n", len);
                    return false;
                }
            }

            /* コピー */
            size_t copyLen = ::std::min<size_t>(GetLengthMbuf(pWorkMbuf) - offset, restLen);
            // TORIAEZU: nn::nstd::MemCpy から std::memcpy に変更
            std::memcpy(&(static_cast<Bit8*>(buf)[dstOffset]), GetTopPtrMbuf(const_cast<Mbuf*>(pWorkMbuf), offset), copyLen);

            /* オフセットと残りコピー長を更新 */
            dstOffset += copyLen;
            restLen   -= copyLen;
            pWorkMbuf  = GetNextMbuf(pWorkMbuf);
            offset     = 0;
        }
        return true;
    }

    bool MbufCopyback(Mbuf* pMbuf, size_t offset, size_t len, const void* buf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        NN_SDK_REQUIRES_NOT_NULL(buf);

        // オフセットに到達するまで mbuf チェインを辿ります。
        // サイズが不足している場合は mbuf チェインを拡張します。
        Mbuf* pNow = pMbuf;
        size_t dstOffset = offset;
        while (GetLengthMbuf(pNow) < dstOffset)
        {
            dstOffset -= GetLengthMbuf(pNow);
            if (GetNextMbuf(pNow) == nullptr)
            {
                Mbuf* pNext = AllocateMbuf(
                    dstOffset + len, MbufAllocationMode_DontWait,
                    GetTypeMbuf(pMbuf), AllocationSizePolicty_NotRequired);
                if (pNext == nullptr)
                {
                    return false;
                }
                SetLengthMbuf(pNext, std::min(dstOffset + len, GetCapacityMbuf(pNext)));
                LinkMbuf(pNow, pNext);
            }
            pNow = GetNextMbuf(pNow);
        }

        // mbuf チェインにデータをコピーします。
        size_t srcOffset = 0;
        size_t restLength = len;
        Mbuf* pPrev = nullptr;
        bool isExtensible = false;
        while (0 < restLength)
        {
            // 末尾に到達した場合は mbuf チェインを拡張します。
            if (pNow == nullptr)
            {
                pNow = AllocateMbuf(
                    dstOffset + restLength, MbufAllocationMode_DontWait,
                    GetTypeMbuf(pMbuf), AllocationSizePolicty_NotRequired);
                if (pNow == nullptr)
                {
                    return false;
                }
                SetLengthMbuf(pNow, std::min(dstOffset + restLength, GetCapacityMbuf(pNow)));
                LinkMbuf(pPrev, pNow);
            }

            // mbuf クラスタのデータ領域を拡張します。
            Mbuf* pNext = GetNextMbuf(pNow);
            const size_t requiredSize = dstOffset + restLength;
            isExtensible = isExtensible || pNext == nullptr || MbufLength(pNext, nullptr) == 0;
            if (isExtensible && GetLengthMbuf(pNow) < requiredSize)
            {
                SetLengthMbuf(pNow, std::min(requiredSize, GetWritableLengthMbuf(pNow)));
            }

            // データをコピーします。
            size_t copyLength = std::min(GetLengthMbuf(pNow) - dstOffset, restLength);
            if (0 < copyLength)
            {
                std::memcpy(
                    GetTopPtrMbuf(pNow, dstOffset),
                    &(static_cast<const Bit8*>(buf)[srcOffset]),
                    copyLength);
                srcOffset  += copyLength;
                restLength -= copyLength;
            }
            dstOffset   = 0;
            pPrev = pNow;
            pNow = pNext;;
        }
        return true;
    }

    bool MbufCat(Mbuf* pMbuf, Mbuf* pMbufNext) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        NN_SDK_REQUIRES_NOT_NULL(pMbufNext);

        Mbuf* pWorkMbuf     = pMbuf;
        Mbuf* pWorkMbufNext = pMbufNext;

        /* 最後尾のmbufを発見 */
        pWorkMbuf = GetTailMbuf(pWorkMbuf);

        /* リンクの結合 */
        LinkMbuf(pWorkMbuf, pWorkMbufNext);
        return true;
    }

    Mbuf* MbufSplit(Mbuf* pMbuf, size_t len, MbufAllocationMode how) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);

        Mbuf* pWorkMbuf = pMbuf;
        Mbuf* remainMbuf;
        size_t offset;
        int type;
        int protocol;
        MbufPacketHeaderFlag flags;

        //NDEBUG_Printf("MbufSplit: %08x %d\n", pWorkMbuf, len);

        /* 情報の退避 */
        protocol = GetProtocolMbuf(pWorkMbuf);
        flags    = GetFlagsMbuf(pWorkMbuf);
        type     = GetTypeMbuf(pWorkMbuf);

        /* 最後の部分となるmbufを発見 */
        offset   = 0;
        while( pWorkMbuf != NULL )
        {
            offset += GetLengthMbuf(pWorkMbuf);
            if( len <= offset )
            {
                break;
            }
            pWorkMbuf = GetNextMbuf(pWorkMbuf);
        }
        if( pWorkMbuf == NULL )
        {
            /* mbufの長さがlenに満たなかった */
            //NDEBUG_Printf("MbufSplit: mbuf end\n");
            return NULL;
        }

        //NDEBUG_Printf("MbufSplit: remain=%d, m_len=%d\n", remain, pWorkMbuf->GetLength());
        if( offset > len )
        {
            size_t remain = offset - len;

            /* 新たに一つmbufを確保する */
            remainMbuf = MbufGetm(NULL, remain, how, type);
            if( remainMbuf == NULL )
            {
                /* 分離失敗 */
                //NDEBUG_Printf("MbufSplit: no memory\n");
                return NULL;
            }
            /* 残りデータをコピー */
            // TORIAEZU: nn::nstd::MemMove から std::memmove に変更
            std::memmove(GetTopPtrMbuf(remainMbuf), GetTopPtrMbuf(pWorkMbuf, GetLengthMbuf(pWorkMbuf) - remain), static_cast<uint32_t>(remain));
            SetLengthMbuf(remainMbuf, remain);
            SetLengthMbuf(pWorkMbuf, GetLengthMbuf(pWorkMbuf) - remain);

            /* 残りのmbufのリンク修正 */
            SetNextMbuf(remainMbuf, GetNextMbuf(pWorkMbuf));
            if( GetNextMbuf(pWorkMbuf) != NULL )
            {
                SetPreviousMbuf(GetNextMbuf(pWorkMbuf), remainMbuf);
            }

            /* mbufの後半を切り離し */
            SetNextMbuf(pWorkMbuf, NULL);

            //NDEBUG_Printf("MbufSplit: first  pWorkMbuf %08x next=%d, len=%d\n", pWorkMbuf, pWorkMbuf->m_pNext, pWorkMbuf->GetLength());
            //NDEBUG_Printf("MbufSplit: second pWorkMbuf %08x next=%d, len=%d\n", remainMbuf, remainMbuf->m_pNext, remainpWorkMbuf->GetLength());
        }
        else
        {
            if( GetNextMbuf(pWorkMbuf) )
            {
                /* 残りのmbufのリンク修正 */
                remainMbuf = GetNextMbuf(pWorkMbuf);
                SetPreviousMbuf(remainMbuf, NULL);

                /* mbufの後半を切り離し */
                SetNextMbuf(pWorkMbuf, NULL);

                //NDEBUG_Printf("MbufSplit: first  pWorkMbuf %08x next=%d, len=%d\n", pWorkMbuf, pWorkMbuf->m_pNext, pWorkMbuf->GetLength());
                //NDEBUG_Printf("MbufSplit: second pWorkMbuf %08x next=%d, len=%d\n", remainMbuf, remainMbuf->m_pNext, remainpWorkMbuf->GetLength());
            }
            else
            {
                /* もうあとがないので残りはNULL */
                remainMbuf = NULL;
                //NDEBUG_Printf("MbufSplit: all first pWorkMbuf %08x next=%d, len=%d\n", pWorkMbuf, pWorkMbuf->m_pNext, pWorkMbuf->GetLength());
            }
        }

        if( remainMbuf != NULL )
        {
            /* 情報の復元 */
            SetProtocolMbuf(remainMbuf, protocol);
            SetFlagsMbuf(remainMbuf, flags);
        }

        return remainMbuf;
    }

    size_t MbufLength(Mbuf* pMbuf, Mbuf** pOutLastMbuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);

        Mbuf*  pWorkMbuf        = pMbuf;
        Mbuf** pWorkOutMbufLast = pOutLastMbuf;

        size_t len;

        //NDEBUG_Printf("MbufLength: %08x ", mbuf);

        len = 0;
        for( ; ; )
        {
            len += GetLengthMbuf(pWorkMbuf);
            if( GetNextMbuf(pWorkMbuf) != NULL )
            {
                pWorkMbuf = GetNextMbuf(pWorkMbuf);
            }
            else
            {
                break;
            }
        }
        if( pWorkOutMbufLast != NULL )
        {
            *pWorkOutMbufLast = pWorkMbuf;
        }

        //NDEBUG_Printf("len = %d\n", len);

        return len;
    }

    int MbufApply(Mbuf* pMbuf, size_t offset, size_t len, int (* func)(void* arg, void* data, size_t len), void* arg) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        NN_SDK_REQUIRES_NOT_NULL(func);

        Mbuf* pWorkMbuf = pMbuf;
        size_t clOffset;
        size_t dataLen;
        int err;

        //NDEBUG_Printf("MbufApply: %08x offset=%d, len=%d\n", pWorkMbuf, offset, len);

        /* 先頭のmbufを発見 */
        clOffset = 0;
        while( pWorkMbuf != NULL )
        {
            clOffset += GetLengthMbuf(pWorkMbuf);
            if( offset < clOffset )
            {
                break;
            }
            pWorkMbuf = GetNextMbuf(pWorkMbuf);
        }
        if( pWorkMbuf == NULL )
        {
            /* mbufの長さがlenに満たなかった */
            //NDEBUG_Printf("MbufCopydata: mbuf end\n");
            return -1;
        }

        /* 先頭のmbufでのオフセット */
        clOffset = GetLengthMbuf(pWorkMbuf) - (clOffset - offset);
        while( len > 0 )
        {
            dataLen = (len > (GetLengthMbuf(pWorkMbuf) - clOffset)) ? (GetLengthMbuf(pWorkMbuf) - clOffset) : len;

            //NDEBUG_Printf("MbufApply: pWorkMbuf=%08x, top=%d, clOffset = %d, len=%d\n", pWorkMbuf, pWorkMbuf->m_top, clOffset, dataLen);

            err = func(arg, GetTopPtrMbuf(pWorkMbuf, clOffset), dataLen);
            if( err != 0 )
            {
                //NDEBUG_Printf("MbufApply: function error %d\n", err);
                return err;
            }

            len -= dataLen;
            pWorkMbuf = GetNextMbuf(pWorkMbuf);
            clOffset = 0;
            if( len > 0 && pWorkMbuf == NULL )
            {
                //NDEBUG_Printf("MbufApply: invalid len %d\n", len);
                return -1;
            }
        }

        return 0;
    }

    bool MbufAlignWithOffset(Mbuf* pMbuf, size_t align, size_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        NN_SDK_REQUIRES(0 < align);
        NN_SDK_REQUIRES((align & (align - 1)) == 0);
        NN_SDK_REQUIRES(offset < align);

        const uintptr_t topAddress     = reinterpret_cast<uintptr_t>(MbufTod(pMbuf));
        const uintptr_t alignedAddress = ((topAddress + align) & ~(align - 1)) + offset;
        size_t diff = static_cast<size_t>(alignedAddress - topAddress);


        //NN_MIN_TASSERT_(pNnMbuf->m_top, 0);
        NN_SDK_ASSERT(pMbuf->_pNext == NULL && pMbuf->_pPrev == NULL);

        //NN_LOG_DEBUG("align: %d, offset: %-d", align, offset);
        //MbufDump(pMbuf);
        //NN_LOG_DEBUG("%p => %p\n", topAddress, alignedAddress);
        if( pMbuf->_len != 0 )
        {
            if( MbufTrailingspace(pMbuf) < diff )
            {
                return false;
            }
            // TORIAEZU: nn::nstd::MemMove から std::memmove に変更
            std::memmove(reinterpret_cast<void*>(alignedAddress), reinterpret_cast<void*>(topAddress), pMbuf->_len);
        }
        pMbuf->_top = static_cast<uint16_t>(pMbuf->_top + diff);

        //NN_MIN_TASSERT_(pNnMbuf->m_top, 0);
        //NN_MAX_TASSERT_(pNnMbuf->m_top + pNnMbuf->m_len, pNnMbuf->m_capacity);
        return true;
    }

    bool MbufExpand(Mbuf* pMbuf, size_t len) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMbuf);
        if (pMbuf->_top + pMbuf->_len + len > pMbuf->_capacity)
        {
            NN_SDK_LOG("MbufExpand: out of capacity.(len = %d)\n", len);
            return false;
        }
        pMbuf->_len += static_cast<uint16_t>(len);
        return true;
    }

    int MbufGetFreeCount( int type ) NN_NOEXCEPT
    {
        return GetFreeCount(type);
    }

}} // end of namespace nn::mbuf
