﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ユニットヒープ API です。内部実装は mem_DetailUnitHeap.cpp 内にあります。
 */

#include "./detail/lmem_DetailUnitHeap.h"
#include "./detail/lmem_DetailCommonHeap.h"


namespace nn { namespace lmem {

HeapHandle CreateUnitHeap( void* startAddress, size_t size, size_t unitSize, int option) NN_NOEXCEPT
{
    HeapHandle heapHandle = detail::CreateUnitHeap(startAddress, size, unitSize, DefaultAlignment, static_cast<uint16_t>(option), InfoPlacement_Head, nullptr);
    if(option & CreationOption_ThreadSafe)
    {
        nn::os::InitializeMutex(&(heapHandle->mutex), false, 0);
    }
    return heapHandle;
}

HeapHandle CreateUnitHeap( void* startAddress, size_t size, size_t unitSize, int option, int alignment, InfoPlacement placement) NN_NOEXCEPT
{
    HeapHandle heapHandle = detail::CreateUnitHeap(startAddress, size, unitSize, alignment, static_cast<uint16_t>(option), placement, nullptr);
    if(option & CreationOption_ThreadSafe)
    {
        nn::os::InitializeMutex(&(heapHandle->mutex), false, 0);
    }
    return heapHandle;
}

HeapHandle CreateUnitHeap( void* startAddress, size_t size, size_t unitSize, int option, int alignment, HeapCommonHead* pHeapHead) NN_NOEXCEPT
{
    HeapHandle heapHandle = detail::CreateUnitHeap(startAddress, size, unitSize, alignment, static_cast<uint16_t>(option), InfoPlacement_Head, pHeapHead);
    if(option & CreationOption_ThreadSafe)
    {
        nn::os::InitializeMutex(&(heapHandle->mutex), false, 0);
    }
    return heapHandle;
}

void DestroyUnitHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    if(heapHandle->attribute & CreationOption_ThreadSafe)
    {
        nn::os::FinalizeMutex(&(heapHandle->mutex));
    }
    detail::DestroyUnitHeap(heapHandle);
}

void InvalidateUnitHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::InvalidateUnitHeap(heapHandle);
}

void ExtendUnitHeapArea(HeapHandle heapHandle, size_t size) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::ExtendUnitHeapArea(heapHandle, size);
}

void* AllocateFromUnitHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AllocFromUnitHeap(heapHandle);
}

void FreeToUnitHeap( HeapHandle heapHandle, void* pBlock ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::FreeToUnitHeap(heapHandle, pBlock);
}

size_t GetUnitHeapUnitSize( HeapHandle heapHandle ) NN_NOEXCEPT
{
    // ユニットサイズはヒープ作成後破棄まで変えられないため、
    // ロックを取る必要はない
    return detail::GetUnitHeapUnitSize(heapHandle);
}

int GetUnitHeapAllocatableCount( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetUnitHeapAllocatableCount(heapHandle);
}

int GetUnitHeapAllocatedCount( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetUnitHeapAllocatedCount( heapHandle );
}

int GetUnitHeapAlignment( HeapHandle heapHandle ) NN_NOEXCEPT
{
    // アライメントはヒープ作成後破棄まで変えられないため、
    // ロックを取る必要はない
    return detail::GetUnitHeapAlignment( heapHandle );
}

size_t GetRequiredUnitHeapSize( size_t unitSize, int unitNum, int alignment, bool hasHeadInternally ) NN_NOEXCEPT
{
    return detail::GetRequiredUnitHeapSize( unitSize, unitNum, alignment, hasHeadInternally);
}

void DumpUnitHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::DumpUnitHeap( heapHandle );
}

}}  // nn::mem
