﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include "lmem_DetailCommonHeap.h"
#include "lmem_DetailFrameHeap.h"
#include "lmem_DetailCommonImpl.h"

namespace nn { namespace lmem { namespace detail {

namespace {

const int MinimumAlignment = 4; // アライメントの最小値

/* ========================================================================
    static関数
   ======================================================================== */

/**
 * @brief       有効なフレームヒープハンドルであるかどうかシグニチャをチェックします。
 * @param[in]   handle  ヒープハンドル
 * @return      有効なヒープハンドルである場合、 true を返します。
 */
inline bool IsValidFrameHeapHandle( HeapHandle handle )
{
    if ( handle == nullptr )
    {
        return false;
    }

    {
        HeapHead* pHeapHead = handle;
        return pHeapHead->signature == MemFrameHeapSignature;
    }
}

/**
 * @brief       ヒープヘッダへのポインタから、フレームヒープヘッダへのポインタを取得します。
 * @param[in]   pHHead  ヒープヘッダへのポインタ
 * @return      フレームヒープヘッダへのポインタを返します。
 */
inline FrameHeapHead* GetFrameHeapHeadPtrFromHeapHead( HeapHead* pHead )
{
    return &pHead->specificHeapHead.frameHeapHead;
}

/**
 * @brief       フレームヒープヘッダへのポインタから、ヒープヘッダへのポインタを取得します。
 * @param[in]   pFrameHeapHead  フレームヒープヘッダへのポインタ
 * @return      ヒープヘッダへのポインタを返します。
 */
inline HeapHead* GetHeapHeadPtrFromFrameHeapHead( FrameHeapHead* pHead )
{
    return reinterpret_cast<HeapHead*>(SubSizeToPtr(pHead, sizeof(HeapHead) - sizeof(SpecificHeapHead)));
}

/**
 * @brief       フレームヒープの初期化を行います。
 * @param[in]   startAddress    フレームヒープとするメモリの開始アドレス
 * @param[in]   endAddress      フレームヒープとするメモリの終了アドレス +1
 * @param[in]   optFlag         オプションフラグ
 * @return      ヒープヘッダへのポインタを返します。
 * @details     ヒープの管理領域（HeapHead）をヒープ領域内に持ちます。
 */
HeapHead* InitFrameHeap(void* startAddress, void* endAddress, int option)
{
    HeapHead* pHeapHead = reinterpret_cast<HeapHead*>(startAddress);
    FrameHeapHead* pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead( pHeapHead );

    // ヒープ共通初期化
    InitHeapHead(pHeapHead,
                 MemFrameHeapSignature,
                 AddSizeToPtr( pFrameHeapHead, sizeof(SpecificHeapHead) ),   // heapStart
                 endAddress,                                                // heapEnd
                 static_cast<uint16_t>(option));

    pFrameHeapHead->nextBlockFront = pHeapHead->pHeapStart;
    pFrameHeapHead->nextBlockRear = pHeapHead->pHeapEnd;

    return pHeapHead;
}

/**
 * @brief       フレームヒープの初期化を行います。
 * @param[in]   startAddress    フレームヒープとするメモリの開始アドレス
 * @param[in]   endAddress      フレームヒープとするメモリの終了アドレス +1
 * @param[in]   optFlag         オプションフラグ
 * @return      ヒープヘッダへのポインタを返します。
 * @details     ヒープの管理領域（HeapHead）はヒープ領域外に持ち、引数として与えてもらいます。
 */
HeapHead* InitFrameHeap(void* startAddress, void* endAddress, int option, HeapHead* pHeapHead)
{
    FrameHeapHead* pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead( pHeapHead );

    // ヒープ共通初期化
    InitHeapHead(pHeapHead,
                 MemFrameHeapSignature,
                 startAddress,              // heapStart
                 endAddress,                // heapEnd
                 static_cast<uint16_t>(option));

    pFrameHeapHead->nextBlockFront = pHeapHead->pHeapStart;
    pFrameHeapHead->nextBlockRear = pHeapHead->pHeapEnd;

    return pHeapHead;
}

/**
 * @brief       ヒープの先頭からメモリブロックを確保します。
 * @param[in]   pFrameHeapHead  ヒープヘッダへのポインタ
 * @param[in]   size            確保するメモリブロックのサイズ
 * @param[in]   alignment       アライメント値
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
                失敗した場合、NULLが返ります。
 */
void* AllocFromHead(FrameHeapHead* pFrameHeapHead, size_t size, int alignment)
{
    void* newBlock = RoundUpPtr(pFrameHeapHead->nextBlockFront, alignment);
    void* endAddress = AddSizeToPtr(newBlock, size);

    if ( GetUIntPtr(endAddress) > GetUIntPtr(pFrameHeapHead->nextBlockRear) )
    {
        return NULL;
    }

#if defined(NN_DETAIL_HEAP_DEBUG)
    // メモリ充填
    FillAllocMemory(GetHeapHeadPtrFromFrameHeapHead(pFrameHeapHead),
                    pFrameHeapHead->nextBlockFront,
                    GetOffsetFromPtr(pFrameHeapHead->nextBlockFront, endAddress));
#endif

    pFrameHeapHead->nextBlockFront = endAddress;

    return newBlock;
}

/**
 * @brief       ヒープの末尾からメモリブロックを確保します。
 * @param[in]   pFrameHeapHead  ヒープヘッダへのポインタ
 * @param[in]   size            確保するメモリブロックのサイズ
 * @param[in]   alignment       アライメント値
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
                失敗した場合、NULLが返ります。
 */
void* AllocFromTail(FrameHeapHead* pFrameHeapHead, size_t size, int alignment)
{
    void* newBlock = RoundDownPtr(SubSizeToPtr(pFrameHeapHead->nextBlockRear, size), alignment);

    if ( GetUIntPtr(newBlock) < GetUIntPtr(pFrameHeapHead->nextBlockFront) )
    {
        return NULL;
    }

#if defined(NN_DETAIL_HEAP_DEBUG)
    // メモリ充填
    FillAllocMemory(GetHeapHeadPtrFromFrameHeapHead(pFrameHeapHead),
                    newBlock,
                    GetOffsetFromPtr(newBlock, pFrameHeapHead->nextBlockRear) );
#endif

    pFrameHeapHead->nextBlockRear = newBlock;

    return newBlock;
}

/**
 * @brief       ヒープ領域の先頭から確保したメモリブロックを一括して開放します。
 * @param[in]   pHead:  ヒープのヘッダへのポインタ
 */
void FreeFromHead( HeapHead* pHead )
{
    FrameHeapHead* pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead(pHead);

#if defined(NN_DETAIL_HEAP_DEBUG)
    FillFreeMemory(pHead,
                   pHead->pHeapStart,
                   GetOffsetFromPtr(pHead->pHeapStart, pFrameHeapHead->nextBlockFront) );
#endif

    pFrameHeapHead->nextBlockFront = pHead->pHeapStart;
}

/**
 * @brief       ヒープの後方から確保した全てのメモリブロックを一括して開放します。
 * @param[in]   pHead   ヒープのヘッダへのポインタ
 */
void FreeFromTail( HeapHead* pHead )
{
    FrameHeapHead* pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead(pHead);

#if defined(NN_DETAIL_HEAP_DEBUG)
    FillFreeMemory(pHead,
                   pFrameHeapHead->nextBlockRear,
                   GetOffsetFromPtr(pFrameHeapHead->nextBlockRear, pHead->pHeapEnd) );
#endif

    pFrameHeapHead->nextBlockRear = pHead->pHeapEnd;
}

/**
 * @brief       サイズとパーセンテージを出力します。
 * @param[in]   size        対象となるサイズ
 * @param[in]   wholeSize   全体のサイズ
 */
void PrintSize(size_t size, size_t wholeSize)
{
    NN_UNUSED(size);
    NN_UNUSED(wholeSize);
    // NN_SDK_LOG で %f が使えないため、 %d で代用
    // 小数点以下2位まで表示（切り捨て）
    int percent = static_cast<int>(static_cast<uint64_t>(size) * 10000 / wholeSize);
    NN_UNUSED(percent);
    NN_SDK_LOG("%zu (%d.%02d%%) ", size, percent / 100, percent % 100);
}

}   // unnamed namespace


/* ========================================================================
    外部関数(非公開)
   ======================================================================== */

/**
 * @brief       フレームヒープのフリーエリアの先頭アドレスを取得します。
 */
void* GetNextBlockFront( HeapHandle heap )
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));
    return GetFrameHeapHeadPtrFromHeapHead(heap)->nextBlockFront;
}

/**
 * @brief       フレームヒープのフリーエリアの末尾アドレスを取得します。
 */
void* GetNextBlockRear( HeapHandle heap )
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));
    return GetFrameHeapHeadPtrFromHeapHead(heap)->nextBlockRear;
}

/* ========================================================================
    外部関数(公開)
   ======================================================================== */

/**
 * @brief   フレームヒープを作成します。
 * @details ヒープの管理領域（HeapHead）はヒープ内部に作成されます。
 */
HeapHandle CreateFrameHeap( void* startAddress, size_t size, int option ) NN_NOEXCEPT
{
    void* endAddress;

    NN_SDK_ASSERT(startAddress != NULL);

    endAddress   = RoundDownPtr(AddSizeToPtr(startAddress, size), MinimumAlignment);
    startAddress = RoundUpPtr(startAddress, MinimumAlignment);

    if ( GetUIntPtr(startAddress) > GetUIntPtr(endAddress) ||
         GetOffsetFromPtr(startAddress, endAddress) < sizeof(HeapHead) + sizeof(SpecificHeapHead)
       )
    {
        return nullptr;
    }

    // フレームヒープ向け初期化
    HeapHead* pHead = InitFrameHeap( startAddress, endAddress, option );

    return pHead;  // ヒープヘッダへのポインタがそのままハンドル値とする
}

/**
 * @brief   フレームヒープを作成します。
 * @details ヒープの管理領域（HeapHead）はヒープ外部に作成され、引数で与えられます。
 */
HeapHandle CreateFrameHeap( void* startAddress, size_t size, int option, HeapCommonHead* pHeapHead ) NN_NOEXCEPT
{
    void* endAddress;

    NN_SDK_ASSERT(startAddress != NULL);

    endAddress   = RoundDownPtr(AddSizeToPtr(startAddress, size), MinimumAlignment);
    startAddress = RoundUpPtr(startAddress, MinimumAlignment);

    if ( GetUIntPtr(startAddress) > GetUIntPtr(endAddress))
    {
        return nullptr;
    }

    {   // Frame ヒープ向け初期化
        HeapHead* pHead = InitFrameHeap( startAddress, endAddress, option, reinterpret_cast<HeapHead*>(pHeapHead) );

        return pHead;  // ヒープヘッダへのポインタがそのままハンドル値とする
    }
}

/**
 * @brief       フレームヒープを破棄します。
 */
void DestroyFrameHeap( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    FinalizeHeap(heap);
}

/**
 * @brief       フレームヒープからメモリブロックを確保します。
 */
void* AllocFromFrameHeap( HeapHandle heap, size_t size, int alignment ) NN_NOEXCEPT
{
    void* memory = NULL;
    FrameHeapHead* pFrameHeapHead;

    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    // アライメントが 4 未満の 2 のべき乗の場合は、4 にする
    if(alignment == 1 || alignment == 2)
    {
        alignment = 4;
    }
    else if(alignment == -1 || alignment == -2)
    {
        alignment = -4;
    }

    // alignment のチェック
    NN_SDK_ASSERT(alignment % MinimumAlignment == 0);
    NN_SDK_ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    NN_SDK_ASSERT(MinimumAlignment <= abs(alignment));

    pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead(heap);

    if ( size == 0 )
    {
        size = 1;
    }

    size = RoundUp(size, MinimumAlignment);

    if ( alignment >= 0 )   // ヒープ前から確保
    {
        memory = AllocFromHead(pFrameHeapHead, size, alignment);
    }
    else                    // ヒープ後ろから確保
    {
        memory = AllocFromTail(pFrameHeapHead, size, -alignment);
    }

    return memory;
}

/**
 * @brief       フレームヒープへメモリブロックを返却します。
 */
void FreeToFrameHeap( HeapHandle heap, FreeMode mode ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    if (mode & FreeMode_Front)
    {
        FreeFromHead(heap);
    }

    if (mode & FreeMode_Rear)
    {
        FreeFromTail(heap);
    }
}

/**
 * @brief       フレームヒープ内の割り当て可能な最大サイズを取得します。
 */
size_t GetAllocatableSizeForFrameHeap( HeapHandle heap, int alignment ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    // alignment のチェック
    NN_SDK_ASSERT(alignment % MinimumAlignment == 0);
    NN_SDK_ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    NN_SDK_ASSERT(MinimumAlignment <= abs(alignment));

    alignment = abs(alignment); // 念のため正数化

    size_t returnValue;
    const FrameHeapHead* pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead(heap);
    const void* block = RoundUpPtr(pFrameHeapHead->nextBlockFront, alignment);

    if ( GetUIntPtr(block) > GetUIntPtr(pFrameHeapHead->nextBlockRear) )
    {
        returnValue = 0;
    }
    else
    {
        returnValue = GetOffsetFromPtr( block, pFrameHeapHead->nextBlockRear );
    }
    return returnValue;
}


/**
 * @brief       フレームヒープの現在のメモリ使用状態を記録します。@n
                後で記録したメモリ使用状況に戻すことができます。
 */
FrameHeapState GetFrameHeapState( HeapHandle heap ) NN_NOEXCEPT
{

    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    FrameHeapHead* pFrameHeapHead  = GetFrameHeapHeadPtrFromHeapHead(heap);

    // 現在の状態を格納
    FrameHeapState state;
    state.nextBlockFront = pFrameHeapHead->nextBlockFront;
    state.nextBlockRear = pFrameHeapHead->nextBlockRear;

    return state;
}

/**
 * @brief       フレームヒープのメモリブロックを記録された状態に従って返却します。
 */
void RestoreFrameHeapState( HeapHandle heap, const FrameHeapState& pState ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));
    NN_SDK_ASSERT(heap->pHeapStart <= pState.nextBlockFront && pState.nextBlockFront <= heap->pHeapEnd);
    NN_SDK_ASSERT(heap->pHeapStart <= pState.nextBlockRear && pState.nextBlockRear <= heap->pHeapEnd);

    FrameHeapHead*  pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead(heap);

    NN_SDK_ASSERT(pState.nextBlockFront <= pFrameHeapHead->nextBlockFront);
    NN_SDK_ASSERT(pState.nextBlockRear >= pFrameHeapHead->nextBlockRear);

    void* oldHeadAllocator = pFrameHeapHead->nextBlockFront;
    void* oldTailAllocator = pFrameHeapHead->nextBlockRear;

    NN_UNUSED(oldHeadAllocator);
    NN_UNUSED(oldTailAllocator);

    pFrameHeapHead->nextBlockFront = pState.nextBlockFront;
    pFrameHeapHead->nextBlockRear = pState.nextBlockRear;

#if defined(NN_DETAIL_HEAP_DEBUG)
    FillFreeMemory( heap,
                    pFrameHeapHead->nextBlockFront,
                    GetOffsetFromPtr(pFrameHeapHead->nextBlockFront, oldHeadAllocator) );
    FillFreeMemory( heap,
                    oldTailAllocator,
                    GetOffsetFromPtr(oldTailAllocator, pFrameHeapHead->nextBlockRear) );
#endif

}

/**
 * @brief       フレームヒープの空き領域をヒープ領域から開放し、その分ヒープ領域を縮小します。
 */
MemoryRange AdjustFrameHeap( HeapHandle heap, AdjustMode adjustMode ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    HeapHead* pHeapHead = heap;
    FrameHeapHead* pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead( pHeapHead );
    size_t adjustSize;

    if(adjustMode == AdjustMode_Tail)
    {
        // 後方に確保されたメモリブロックが存在する場合は失敗する
        if ( 0 < GetOffsetFromPtr( pFrameHeapHead->nextBlockRear, pHeapHead->pHeapEnd ) )
        {
            return MakeMemoryRange(pHeapHead->pHeapEnd, 0);
        }

        adjustSize = GetOffsetFromPtr( pFrameHeapHead->nextBlockFront, pHeapHead->pHeapEnd );
        pFrameHeapHead->nextBlockRear = pHeapHead->pHeapEnd = pFrameHeapHead->nextBlockFront;

#if defined(NN_DETAIL_HEAP_DEBUG)
        FillFreeMemory(pHeapHead, pHeapHead->pHeapEnd, adjustSize);
#endif
        return MakeMemoryRange(pHeapHead->pHeapEnd, adjustSize);
    }
    else if(adjustMode == AdjustMode_Head)
    {
        // 前方に確保されたメモリブロックが存在する、もしくはHeapHeadがヒープ領域内にある場合は失敗する
        if ( 0 < GetOffsetFromPtr( pHeapHead->pHeapStart, pFrameHeapHead->nextBlockFront ) ||
             GetUIntPtr(pHeapHead) == GetUIntPtr(pHeapHead->pHeapStart) - sizeof(HeapHead) )
        {
            return MakeMemoryRange(pHeapHead->pHeapEnd, 0);
        }

        void* oldFront = pHeapHead->pHeapStart;
        adjustSize = GetOffsetFromPtr( pHeapHead->pHeapStart, pFrameHeapHead->nextBlockRear );
        pFrameHeapHead->nextBlockFront = pHeapHead->pHeapStart = pFrameHeapHead->nextBlockRear;

#if defined(NN_DETAIL_HEAP_DEBUG)
        FillFreeMemory( pHeapHead, oldFront, adjustSize );
#endif

        return MakeMemoryRange(oldFront, adjustSize);
    }

    return MakeMemoryRange(pHeapHead->pHeapEnd, 0);
}

/**
 * @brief       フレームヒープから確保されたメモリブロックのサイズを変更します。
 */
size_t ResizeForMemoryBlockFrameHeap( HeapHandle heap, void* pBlock, size_t newSize ) NN_NOEXCEPT
{
    HeapHead*    pHeapHead    = NULL;
    FrameHeapHead* pFrameHeapHead = NULL;

    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));
    NN_SDK_ASSERT(pBlock == RoundDownPtr(pBlock, MinimumAlignment));  // 最低限、最小アライメントの境界にあるかチェック

    pHeapHead    = heap;
    pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead( pHeapHead );

    NN_SDK_ASSERT( CompareHigherPtr(pHeapHead->pHeapStart, pBlock) <= 0
               &&  CompareHigherPtr(pFrameHeapHead->nextBlockFront, pBlock) > 0 );   // メモリブロックは前方に存在すること

    /*
        newSizeを0することは認めないようにしている。
        0にしてしまうと、memBlockが指すメモリブロックが存在しなくなるため。
    */
    if ( newSize == 0 )
    {
        newSize = 1;
    }
    newSize = RoundUp( newSize, MinimumAlignment );

    {
        const size_t oldSize = GetOffsetFromPtr( pBlock, pFrameHeapHead->nextBlockFront );
        void* endAddress  = AddSizeToPtr( pBlock, newSize );

        if ( newSize == oldSize )  // ブロックサイズ変更なしの場合
        {
            goto ret_;
        }

        if ( newSize > oldSize )  // 拡大するとき
        {
            if ( CompareHigherPtr( endAddress, pFrameHeapHead->nextBlockRear ) > 0 )  // サイズが足りない場合
            {
                newSize = 0;
                goto ret_;
            }
#if defined(NN_DETAIL_HEAP_DEBUG)
            FillAllocMemory( heap, pFrameHeapHead->nextBlockFront, newSize - oldSize );
#endif
        }
        else                      // 縮小するとき
        {
#if defined(NN_DETAIL_HEAP_DEBUG)
            FillFreeMemory( heap, endAddress, oldSize - newSize );
#endif
        }

        pFrameHeapHead->nextBlockFront = endAddress;
    }
ret_:
    return newSize;
}

/**
 * @brief       フレームヒープ内部の情報を表示します。
 */

void DumpFrameHeap( HeapHandle heap )
{
    NN_SDK_ASSERT(IsValidFrameHeapHandle(heap));

    HeapHead *const    pHeapHead    = heap;
    FrameHeapHead *const pFrameHeapHead = GetFrameHeapHeadPtrFromHeapHead( pHeapHead );
    size_t heapSize = GetOffsetFromPtr( pHeapHead->pHeapStart, pHeapHead->pHeapEnd );
    size_t usedSize = heapSize - GetOffsetFromPtr( pFrameHeapHead->nextBlockFront, pFrameHeapHead->nextBlockRear );

    NN_UNUSED(usedSize);

    DumpHeapHead(pHeapHead);

    NN_SDK_LOG(  "     head [%p - %p) ", pHeapHead->pHeapStart, pFrameHeapHead->nextBlockFront);
    PrintSize(GetOffsetFromPtr(pHeapHead->pHeapStart, pFrameHeapHead->nextBlockFront), heapSize);
    NN_SDK_LOG("\n     free ");
    PrintSize(GetOffsetFromPtr(pFrameHeapHead->nextBlockFront, pFrameHeapHead->nextBlockRear), heapSize);
    NN_SDK_LOG("\n     tail [%p - %p) ", pFrameHeapHead->nextBlockRear, pHeapHead->pHeapEnd);
    PrintSize(GetOffsetFromPtr(pFrameHeapHead->nextBlockRear, pHeapHead->pHeapEnd), heapSize);
    NN_SDK_LOG("\n");

    NN_SDK_LOG("    %zu / %zu bytes (%d%%) used\n",
    usedSize, heapSize, static_cast<int>(static_cast<uint64_t>(usedSize) * 100 / heapSize));
    NN_SDK_LOG("\n");
}

}}} // nn::mem::detail
