﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include "lmem_DetailCommonHeap.h"
#include "lmem_DetailCommonImpl.h"

namespace nn { namespace lmem { namespace detail {

/**
 * @brief       拡張ヒープ内部の情報を表示します。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @details     デバッグ用関数です。
 */
void DumpExpHeap(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープを作成します。
 * @param[in]   pStartAddress   ヒープ領域の先頭アドレス
 * @param[in]   size            ヒープ領域のサイズ
 * @param[in]   option          オプションフラグ
 * @return      関数が成功した場合、作成された拡張ヒープのハンドルが返ります。
                関数が失敗すると、NULL が返ります。
 */
HeapHandle CreateExpHeap(void* pStartAddress, size_t size, int option) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープを破棄します。
 * @param[in]   heapHandle    拡張ヒープのハンドル
 */
void DestroyExpHeap(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープからメモリブロックを確保します。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @param[in]   size        確保するメモリブロックのサイズ(バイト単位)
 * @param[in]   alignment   確保するメモリブロックのアライメント
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
 *              失敗した場合、NULLが返ります。
 * @details     メモリブロックのアライメントを指定できます。@n
 *              アライメントには 正または負の 2 のべき乗の値が指定できます。@n
 *              アライメント値を負の値で指定すると、ヒープの空き領域を後方から探します。
 */
void* AllocFromExpHeap(HeapHandle heapHandle, size_t size, int alignment) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープから確保されたメモリブロックのサイズを変更します。
 * @param[in]   heapHandle      拡張ヒープのハンドル
 * @param[in]   pMemoryBlock    サイズを変更するメモリブロックへのポインタ
 * @param[in]   size            新しく割り当てるサイズ(バイト単位)
 * @return      関数が成功した場合、変更されたメモリブロックのサイズを返します(バイト単位)。@n
                関数が失敗した場合、0 が返ります。
 */
size_t ResizeExpHeapMemoryBlock(HeapHandle heapHandle, void* pMemoryBlock, size_t size) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープへメモリブロックを返却します。
 * @param[in]   heapHandle      拡張ヒープのハンドル
 * @param[in]   pMemoryBlock    返却するメモリブロックへのポインタ
 */
void FreeToExpHeap(HeapHandle heapHandle, void* pMemoryBlock) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープの空き領域のサイズの合計を取得します。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @return      拡張ヒープの空き領域のサイズの合計を返します(バイト単位)。
 */
size_t GetExpHeapTotalFreeSize(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープ内の割り当て可能な最大サイズを取得します。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @param[in]   alignment   確保するメモリブロックのアライメント@n
                            2 のべき乗の値が使用できます。
 * @return      拡張ヒープ内の割り当て可能な最大サイズを返します(バイト単位)。
 * @details     メモリブロックのアライメントを指定できます。
 */
size_t GetExpHeapAllocatableSize(HeapHandle heapHandle, int alignment) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープのメモリ確保モードをセットします。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @param[in]   mode        メモリ確保モード
 * @return      以前の拡張ヒープのメモリ確保モードを返します。
 */
uint16_t SetExpHeapAllocMode(HeapHandle heapHandle, uint16_t mode) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープのメモリ確保モードを取得します。
 * @param[in]   heapHandle    拡張ヒープのハンドル
 * @return      拡張ヒープのメモリ確保モードを返します。
 */
uint16_t GetExpHeapAllocMode(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       アライメントの際に発生する隙間の領域を再利用するかどうかを設定します。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @param[in]   isReuse     TRUEの場合、アライメントで発生する領域を再利用します。@n
                            FALSEの場合は再利用しません。
 * @return      以前の設定値が返ります。
 * @details     デフォルトではFALSEに設定されています。@n
                TRUEに設定した場合には小さなメモリ領域が有効に使用できる可能性がありますが、大量のフリーブロックが生成されメモリ確保時のパフォーマンスが悪化する危険性があります。
 */
bool SetUseExpHeapMarginOfAlignment(HeapHandle heapHandle, bool isReuse) NN_NOEXCEPT;

/**
 * @brief       アライメントの際に発生する隙間の領域を再利用するかどうかの値を返します。
 * @param[in]   heapHandle    拡張ヒープのハンドル
 * @return      再利用するかどうかの値が返ります。
 */
bool GetUseExpHeapMarginOfAlignment(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープのグループIDをセットします。
 * @param[in]   heapHandle    拡張ヒープのハンドル
 * @param[in]   groupId セットするグループID値
 * @return      以前のグループID値が返ります。
 */
uint16_t SetExpHeapGroupId(HeapHandle heapHandle, uint16_t groupId) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープのグループIDを取得します。
 * @param[in]   heapHandle    拡張ヒープのハンドル
 * @return      グループID値が返ります。
 */
uint16_t GetExpHeapGroupId(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープから割り当てられたメモリブロック全てに対して、ユーザが指定した関数を呼ばせます。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @param[in]   visitor     各メモリブロックに対して呼ばせる関数
 * @param[in]   userParam   visitor関数に渡すユーザ指定のパラメータ
 * @details     visitor関数で呼ばれるメモリブロックの順番は、確保した順番になります。@n
                visitor の型 HeapVisitor は次のように定義されています。@n
                typedef void (*HeapVisitor)(void* pMemBlock, HeapHandle heapHandle, uintptr_t userParam);@n
                @li pMemBlock   メモリブロックへのポインタ
                @li heapHandle  メモリブロックを含有するヒープ
                @li userParam   ユーザー用パラメータ
 */
void VisitExpHeapAllocated(HeapHandle heapHandle, HeapVisitor visitor, uintptr_t userParam) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープから確保されたメモリブロックのサイズを取得します。
 * @param[in]   pMemBlock    サイズを取得するメモリブロックへのポインタ
 * @return      指定したメモリブロックのサイズを返します(バイト単位)。
 */
size_t GetExpHeapMemoryBlockSize(const void* pMemBlock) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープから確保されたメモリブロックのグループIDを取得します。
 * @param[in]   pMemBlock    グループIDを取得するメモリブロックへのポインタ
 * @return      指定したメモリブロックのグループIDが返ります。
 */
uint16_t GetMemoryBlockHeapGroupId(const void* pMemBlock) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープから確保されたメモリブロックの確保方向を取得します。
 * @param[in]   pMemBlock    サイズを取得するメモリブロックへのポインタ
 * @return      指定したメモリブロックの確保方向が返ります。
 */
uint16_t GetMemoryBlockHeapAllocDir(const void* pMemBlock) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープの空き領域を解放し、拡張ヒープが使用するメモリ領域を縮小します。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @return      ヒープが縮小されることにより空いたメモリ領域の範囲を返します。
 * @details     ヒープメモリの後ろから確保されたメモリブロックが存在していてはいけません。
 */
MemoryRange AdjustExpHeap(HeapHandle heapHandle) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープが破壊されていないかどうかをチェックします。
 * @param[in]   heapHandle  拡張ヒープのハンドル
 * @param[in]   option      オプションフラグ
 * @return      ヒープが正常だった場合 true を返します。@n
                ヒープにエラーがあった場合、false を返します。
 */
bool CheckExpHeap(HeapHandle heapHandle, uint32_t option) NN_NOEXCEPT;

/**
 * @brief       拡張ヒープのメモリブロックが破壊されていないかどうかをチェックします。
 * @param[in]   pMemoryBlock    チェックするメモリブロックへのポインタ。
 * @param[in]   heapHandle      拡張ヒープのハンドル
 * @param[in]   option          オプションフラグ
 * @return      メモリブロックが正常だった場合 true を返します。@n
                メモリブロックにエラーがあった場合、false を返します。
 */
bool CheckExpHeapMemoryBlock(const void* pMemoryBlock, HeapHandle heapHandle, uint32_t option) NN_NOEXCEPT;

}}}
