﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_Result.h>

#include <nn/htcfs.h>
#include <nn/htcfs/detail/htcfs_Log.h>
#include <nn/htcfs/detail/htcfs_IServiceObject.h>

#include <nn/util/util_StringUtil.h>

#include "htcfs_ServiceObjectUtil.h"

namespace {
    nn::sf::SharedPointer<nn::htcfs::IFileSystem> g_Manager;
    bool g_InitializedByHipc = false;
}

namespace nn { namespace htcfs {

namespace detail {

void InitializeWith(nn::sf::SharedPointer<IFileSystem>&& manager) NN_NOEXCEPT
{
    if (g_Manager)
    {
        return;
    }

    g_Manager = manager;
    g_InitializedByHipc = false;
}

}

void Initialize() NN_NOEXCEPT
{
    if (g_Manager)
    {
        return;
    }

    g_Manager = nn::htcfs::CreateFileSystemServiceObjectByHipc();
    g_InitializedByHipc = true;
}

void Finalize() NN_NOEXCEPT
{
    g_Manager = nullptr;

    if (g_InitializedByHipc)
    {
        nn::htcfs::DeleteFileSystemServiceObjectByHipc();
        g_InitializedByHipc = false;
    }
}

nn::Result GetEntryType(nn::fs::DirectoryEntryType* pOutType, const char* pPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    int32_t type;
    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    NN_RESULT_DO(g_Manager->GetEntryType(nn::sf::Out<int32_t>(&type), path));
    *pOutType = static_cast<nn::fs::DirectoryEntryType>(type);

    NN_RESULT_SUCCESS;
}

nn::Result OpenFile(FileHandle* pOutHandle, const char* pPath, nn::fs::OpenMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    nn::sf::SharedPointer<IFile> handle;
    NN_RESULT_DO(g_Manager->OpenFile(&handle, path, mode));
    *pOutHandle = static_cast<FileHandle>(handle.Detach());

    NN_RESULT_SUCCESS;
}

void CloseFile(FileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    nn::sf::ReleaseSharedObject(static_cast<IFile*>(handle));
}

nn::Result GetPriorityForFile(int32_t* pOutPriority, FileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFile*>(handle)->GetPriorityForFile(nn::sf::Out<int32_t>(pOutPriority));
}

nn::Result SetPriorityForFile(int32_t priority, FileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFile*>(handle)->SetPriorityForFile(priority);
}

nn::Result CreateFile(const char* pPath, int64_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    return g_Manager->CreateFile(path, size);
}

nn::Result DeleteFile(const char* pPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    return g_Manager->DeleteFile(path);
}

nn::Result RenameFile(const char* pFromName, const char* pToName) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path fromPath;
    nn::util::Strlcpy(fromPath.str, pFromName, sizeof(Path::str));
    Path toPath;
    nn::util::Strlcpy(toPath.str, pToName, sizeof(Path::str));

    return g_Manager->RenameFile(fromPath, toPath);
}

nn::Result FileExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    return g_Manager->FileExists(nn::sf::Out<bool>(pOutExists), path);
}

nn::Result ReadFile(size_t* pOutSize, void* pOutBuffer, FileHandle handle, int64_t offset, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);

    auto outBuffer = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutBuffer), size);

    int64_t outSize;
    NN_RESULT_DO(static_cast<IFile*>(handle)->ReadFile(nn::sf::Out<int64_t>(&outSize), outBuffer, offset, option));

    if (outSize < 0 || static_cast<uint64_t>(outSize) > std::numeric_limits<size_t>::max())
    {
        return nn::htcfs::ResultNumericOverflow();
    }
    *pOutSize = static_cast<size_t>(outSize);

    NN_RESULT_SUCCESS;
}

nn::Result WriteFile(const void* pBuffer, FileHandle handle, int64_t offset, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);

    auto buffer = nn::sf::InBuffer(reinterpret_cast<const char*>(pBuffer), size);
    return static_cast<IFile*>(handle)->WriteFile(buffer, offset, option);
}

nn::Result FlushFile(FileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFile*>(handle)->FlushFile();
}

nn::Result GetFileTimeStamp(uint64_t* pOutCreateTime, uint64_t* pOutAccessTime, uint64_t* pOutModifyTime, const char* pPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pPath);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));
    uint64_t createTime, accessTime, modifyTime;

    NN_RESULT_DO(g_Manager->GetFileTimeStamp(nn::sf::Out<uint64_t>(&createTime), nn::sf::Out<uint64_t>(&accessTime), nn::sf::Out<uint64_t>(&modifyTime), path));

    if (pOutCreateTime)
    {
        *pOutCreateTime = createTime;
    }
    if (pOutAccessTime)
    {
        *pOutAccessTime = accessTime;
    }
    if (pOutModifyTime)
    {
        *pOutModifyTime = modifyTime;
    }

    NN_RESULT_SUCCESS;
}

nn::Result GetFileSize(int64_t* pOutSize, FileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFile*>(handle)->GetFileSize(nn::sf::Out<int64_t>(pOutSize));
}

nn::Result SetFileSize(int64_t size, FileHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFile*>(handle)->SetFileSize(size);
}

nn::Result OpenDirectory(DirectoryHandle* pOutHandle, const char* pPath, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    nn::htcfs::Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    nn::sf::SharedPointer<IDirectory> handle;
    NN_RESULT_DO(g_Manager->OpenDirectory(&handle, path, mode));
    *pOutHandle = handle.Detach();

    NN_RESULT_SUCCESS;
}

void CloseDirectory(DirectoryHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    nn::sf::ReleaseSharedObject(static_cast<IDirectory*>(handle));
}

nn::Result GetPriorityForDirectory(int32_t* pOutPriority, DirectoryHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IDirectory*>(handle)->GetPriorityForDirectory(nn::sf::Out<int32_t>(pOutPriority));
}

nn::Result SetPriorityForDirectory(int32_t priority, DirectoryHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IDirectory*>(handle)->SetPriorityForDirectory(priority);
}

nn::Result CreateDirectory(const char* pPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    return g_Manager->CreateDirectory(path);
}

nn::Result DeleteDirectory(const char* pPath, bool recursively) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    return g_Manager->DeleteDirectory(path, recursively);
}

nn::Result RenameDirectory(const char* pFromName, const char* pToName) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path fromPath;
    nn::util::Strlcpy(fromPath.str, pFromName, sizeof(Path::str));
    Path toPath;
    nn::util::Strlcpy(toPath.str, pToName, sizeof(Path::str));

    return g_Manager->RenameDirectory(fromPath, toPath);
}

nn::Result DirectoryExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy(path.str, pPath, sizeof(Path::str));

    return g_Manager->DirectoryExists(nn::sf::Out<bool>(pOutExists), path);
}

nn::Result ReadDirectory(int64_t* pOutCount, nn::fs::DirectoryEntry* pOutBuffer, int64_t count, DirectoryHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);

    if (count < 0 || static_cast<uint64_t>(count) > std::numeric_limits<size_t>::max())
    {
        return ResultInvalidArgument();
    }

    auto entries = nn::sf::OutArray<nn::fs::DirectoryEntry>(pOutBuffer, static_cast<size_t>(count));
    return static_cast<IDirectory*>(handle)->ReadDirectory(nn::sf::Out<int64_t>(pOutCount), entries);
}

nn::Result GetEntryCount(int64_t* pOutCount, DirectoryHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(handle);
    return static_cast<IDirectory*>(handle)->GetEntryCount(nn::sf::Out<int64_t>(pOutCount));
}

}} // namespace
