﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/htc/htc_Api.h>
#include <nn/htc/tenv/htc_Tenv.h>
#include <nn/htc/tenv/htc_TenvIService.h>
#include <nn/htc/tenv/htc_TenvIServiceManager.h>
#include <nn/htc/tenv/htc_TenvServiceManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_MemUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_HipcClientProxyByName.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace htc {

    namespace {
        struct HtcTenvTag;
        typedef nn::sf::ExpHeapStaticAllocator<1024 * 4, HtcTenvTag> HipcAllocator;

        class HipcAllocatorInitializer
        {
        public:

            HipcAllocatorInitializer() NN_NOEXCEPT
            {
                HipcAllocator::Initialize(nn::lmem::CreationOption_NoOption);
            }

        } g_HipcAllocatorInitializer;

        nn::sf::HipcSimpleClientSessionManager& GetClientManager() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(nn::sf::HipcSimpleClientSessionManager, g_ClientManager);
            return g_ClientManager;
        }

#if defined(NN_BUILD_CONFIG_OS_WIN)
        void* StdAllocate(size_t size) NN_NOEXCEPT
        {
            return std::malloc(size);
        }

        void StdDeallocate(void* p, size_t size) NN_NOEXCEPT
        {
            NN_UNUSED(size);
            return std::free(p);
        }
#endif

        nn::sf::SharedPointer<nn::htc::tenv::IServiceManager> GetTenvServiceManagerImpl() NN_NOEXCEPT
        {
            nn::sf::SharedPointer<nn::htc::tenv::IServiceManager> serviceManager;
#if defined(NN_BUILD_CONFIG_OS_WIN)
            static nn::sf::UnmanagedServiceObject<nn::htc::tenv::IServiceManager, nn::htc::tenv::ServiceManager> g_ServiceManager;
            serviceManager = g_ServiceManager.GetShared();
#else
            auto result = GetClientManager().InitializeByName<nn::htc::tenv::IServiceManager, HipcAllocator::Policy>(&serviceManager, nn::htc::tenv::ServiceName);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            const int SessionCount = 1;
            GetClientManager().SetSessionCount(SessionCount);
#endif
            return serviceManager;
        }

        nn::sf::SharedPointer<nn::htc::tenv::IService> GetTenvService() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<nn::htc::tenv::IServiceManager>, g_ServiceManager, = GetTenvServiceManagerImpl());
            nn::sf::SharedPointer<nn::htc::tenv::IService> service;
            NN_RESULT_DO(g_ServiceManager->GetServiceInterface(&service, 0));
            return service;
        }

        nn::htc::tenv::VariableName GetVariableName(const char* variableName)
        {
            nn::htc::tenv::VariableName variableNameObject;
            NN_SDK_REQUIRES_LESS(static_cast<size_t>(util::Strnlen(variableName, sizeof(variableNameObject.str))), sizeof(variableNameObject.str));
            NN_SDK_REQUIRES_GREATER(util::Strnlen(variableName, sizeof(variableNameObject.str)), 0);
            util::Strlcpy(variableNameObject.str, variableName, sizeof(variableNameObject.str));
            return variableNameObject;
        }
    }

    Result GetTargetEnvironmentVariableLength(size_t* pOutValue, const char* variableName) NN_NOEXCEPT
    {
        int64_t length = 0;
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(variableName);

        NN_RESULT_DO(GetTenvService()->GetVariableLength(nn::sf::Out<int64_t>(&length), GetVariableName(variableName)));
        *pOutValue = static_cast<size_t>(length);
        NN_RESULT_SUCCESS;
    }

    Result GetTargetEnvironmentVariable(size_t* pOutSize, char* outBuffer, size_t bufferSize, const char* variableName) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);
        NN_SDK_REQUIRES_NOT_NULL(variableName);

        int64_t outSize = 0;
        NN_RESULT_DO(GetTenvService()->GetVariable(nn::sf::Out<int64_t>(&outSize), nn::sf::OutBuffer(outBuffer, bufferSize), GetVariableName(variableName)));
        *pOutSize = static_cast<size_t>(outSize);
        NN_RESULT_SUCCESS;
    }

    Result WaitUntilTargetEnvironmentVariableAvailable(nn::TimeSpan timeOut) NN_NOEXCEPT
    {
        return GetTenvService()->WaitUntilVariableAvailable(timeOut.GetMilliSeconds());
    }

}}
