﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/fs_Base.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/htc/htc_Result.h>
#include <nn/htc/htc_ResultPrivate.h>
#include <nn/os/os_Thread.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "htc_TenvAllocator.h"
#include "htc_TenvDefinitionFileInfo.h"
#include "htc_TenvDefinitionFileParser.h"
#include "htc_TenvImpl.h"
#include "htc_TenvService.h"
#include <nn/os.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#endif

namespace nn { namespace htc { namespace tenv {

    namespace {

        nn::os::EventType g_MountHostEvent;

        class Buffer
        {
        public:
            Buffer() NN_NOEXCEPT
            {
            }

            Result Alloc(size_t size) NN_NOEXCEPT
            {
                m_Buffer.reset(reinterpret_cast<char*>(detail::Allocate(size)));
                NN_RESULT_THROW_UNLESS(m_Buffer, ResultAllocationMemoryFailed());
                m_Size = size;
                NN_RESULT_SUCCESS;
            }

            ~Buffer() NN_NOEXCEPT
            {
                if (m_Buffer)
                {
                    detail::Deallocate(m_Buffer.release(), m_Size);
                }
            }

            char* Get() NN_NOEXCEPT
            {
                if (!m_Buffer)
                {
                    return nullptr;
                }
                return m_Buffer.get();
            }

        private:
            std::unique_ptr<char> m_Buffer;
            size_t m_Size;
        };

        // "%" で囲まれた文字列を取得する
        bool GetIncludedVariable(char* outValue, size_t* nextBegin, size_t* nextEnd, const StdString& str, size_t pos) NN_NOEXCEPT
        {
            size_t begin = str.find("%", pos);
            if (begin == StdString::npos)
            {
                return false;
            }

            if (begin + 1 >= str.length())
            {
                return false;
            }

            size_t end = str.find("%", begin + 1);
            if (end == StdString::npos)
            {
                return false;
            }

            util::Strlcpy(outValue, str.c_str() + begin + 1, static_cast<int>(end - begin));
            *nextBegin = begin;
            *nextEnd = end;

            return true;
        }

        Result ReadDefinitionFile(int64_t* outValue, Buffer& buffer, const char* path) NN_NOEXCEPT
        {
            const int64_t FileSizeMax = 6 * 1024;

#if defined(NN_BUILD_CONFIG_OS_WIN32)
            wchar_t fullPath[MAX_PATH];
            if (!::MultiByteToWideChar(CP_UTF8, NULL, path, -1, fullPath, MAX_PATH))
            {
                return nn::htc::ResultConnectionFailure();
            }

            auto handle = ::CreateFileW(fullPath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
            if (handle == INVALID_HANDLE_VALUE)
            {
                return nn::htc::ResultConnectionFailure();
            }

            NN_UTIL_SCOPE_EXIT
            {
                ::CloseHandle(handle);
            };

            LARGE_INTEGER size;
            auto result = ::GetFileSizeEx(handle, &size);
            if (!result || size.QuadPart > FileSizeMax)
            {
                return nn::htc::ResultConnectionFailure();
            }

            NN_RESULT_DO(buffer.Alloc(static_cast<size_t>(size.QuadPart)));

            DWORD readSize = 0;
            if (!::ReadFile(handle, buffer.Get(), static_cast<DWORD>(size.QuadPart), &readSize, NULL))
            {
                return nn::htc::ResultConnectionFailure();
            }

            *outValue = size.QuadPart;
#else
            // MountHostRoot がまだであれば Failure を返す
            if(!TryWaitEvent(&g_MountHostEvent))
            {
                return nn::htc::ResultConnectionFailure();
            }

            Result result;

            fs::FileHandle file;
            result = fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
            if (result.IsFailure())
            {
                return nn::htc::ResultConnectionFailure();
            }

            NN_UTIL_SCOPE_EXIT
            {
                fs::CloseFile(file);
            };

            int64_t fileSize = 0;
            result = fs::GetFileSize(&fileSize, file);
            if (result.IsFailure() || fileSize > FileSizeMax)
            {
                return ResultConnectionFailure();
            }

            NN_RESULT_DO(buffer.Alloc(static_cast<size_t>(fileSize)));

            result = fs::ReadFile(file, 0, buffer.Get(), static_cast<size_t>(fileSize));
            if (result.IsFailure())
            {
                return ResultConnectionFailure();
            }

            *outValue = fileSize;
#endif
            NN_RESULT_SUCCESS;
        }
    }

    void InitializeMountHostSuccessEvent() NN_NOEXCEPT
    {
        nn::os::InitializeEvent(&g_MountHostEvent, false, nn::os::EventClearMode_ManualClear);
    }

    void SignalMountHostSuccessEvent() NN_NOEXCEPT
    {
        // 一旦 Signal されたらクリアしない
        nn::os::SignalEvent(&g_MountHostEvent);
    }

    Service::Service(nn::Bit64 processId) NN_NOEXCEPT : m_ProcessId(processId)
    {
    }

    Result Service::GetVariableImpl(StdString* outStr, const nn::htc::tenv::VariableName& variableName) NN_NOEXCEPT
    {
        auto filePath = GetDefinitionFilePath();
        if (filePath == nullptr)
        {
            return nn::htc::ResultConnectionFailure();
        }

        Buffer buffer;
        int64_t fileSize = 0;
        NN_RESULT_DO(ReadDefinitionFile(&fileSize, buffer, filePath));

        size_t valueLength = 0;
        if (!detail::SearchKey(nullptr, &valueLength, variableName.str, buffer.Get(), static_cast<size_t>(fileSize)))
        {
            return ResultNotFound();
        }

        StdString str(valueLength, '\0');

        if (!detail::SearchKey(const_cast<char*>(str.c_str()), &valueLength, variableName.str, buffer.Get(), static_cast<size_t>(fileSize)))
        {
            return ResultNotFound();
        }

        char nextKey[detail::SearchKeyLengthMax];
        size_t pos = 0;
        size_t nextBegin = 0;
        size_t nextEnd = 0;
        while (GetIncludedVariable(nextKey, &nextBegin, &nextEnd, str.c_str(), pos))
        {
            size_t innerValueLength = 0;
            if (detail::SearchKey(nullptr, &innerValueLength, nextKey, buffer.Get(), static_cast<size_t>(fileSize)))
            {
                StdString innerStr(innerValueLength, '\0');
                if (detail::SearchKey(const_cast<char*>(innerStr.c_str()), &innerValueLength, nextKey, buffer.Get(), static_cast<size_t>(fileSize)))
                {
                    str.replace(nextBegin, nextEnd - nextBegin + 1, StdString(""));
                    str.insert(nextBegin, innerStr);
                    valueLength += innerValueLength - (nextEnd - nextBegin + 1);
                }
            }
            pos = nextEnd;
        }

        outStr->append(str);

        NN_RESULT_SUCCESS;
    }

    Result Service::GetVariable(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outBuffer, const nn::htc::tenv::VariableName& variableName) NN_NOEXCEPT
    {
        StdString str;
        NN_RESULT_DO(GetVariableImpl(&str, variableName));
        NN_RESULT_THROW_UNLESS(outBuffer.GetSize() >= str.length() + 1, nn::htc::ResultNotEnoughBuffer());
        util::Strlcpy(outBuffer.GetPointerUnsafe(), str.c_str(), static_cast<int>(str.length() + 1));
        outValue.Set(str.length() + 1);
        NN_RESULT_SUCCESS;
    }

    Result Service::GetVariableLength(nn::sf::Out<std::int64_t> outValue, const nn::htc::tenv::VariableName& variableName) NN_NOEXCEPT
    {
        StdString str;
        NN_RESULT_DO(GetVariableImpl(&str, variableName));
        outValue.Set(str.length() + 1);
        NN_RESULT_SUCCESS;
    }

    Result Service::WaitUntilVariableAvailable(std::int64_t timeOutMilliSeconds) NN_NOEXCEPT
    {
        const int64_t IntervalMilliSeconds = 1;
        int64_t rest = timeOutMilliSeconds;

        while (rest > 0)
        {
            if (GetDefinitionFilePath() != nullptr)
            {
                NN_RESULT_SUCCESS;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(IntervalMilliSeconds));
            rest -= IntervalMilliSeconds;
        }

        return nn::htc::ResultConnectionFailure();
    }

    const char* Service::GetDefinitionFilePath() NN_NOEXCEPT
    {
        detail::DefinitionFileInfo* info;
        if (!detail::GetDefinitionFileInfo(&info, m_ProcessId))
        {
            return nullptr;
        }
        return info->path.string;
    }

}}}
