﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/htc/htc_ResultPrivate.h>
#include <nn/htc/tenv/htc_TenvPath.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_StringUtil.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#include <shlwapi.h>
#endif

#include "htc_TenvDefinitionFileInfo.h"
#include "htc_TenvImpl.h"

namespace nn { namespace htc { namespace tenv { namespace detail {

    namespace
    {
        bool SearchDefinitionFilePath(char* path, size_t size) NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            auto DefinitionFilePathExtention = L"\\*.tdf";
            // 実行ファイルパスを検索
            {
                wchar_t definitionFilePathPattern[MAX_PATH + 1];
                NN_ABORT_UNLESS(::GetModuleFileName(NULL, definitionFilePathPattern, MAX_PATH + 1) != 0);
                NN_ABORT_UNLESS(::PathRemoveFileSpec(definitionFilePathPattern) != 0);
                NN_ABORT_UNLESS(::PathAppend(definitionFilePathPattern, DefinitionFilePathExtention) != 0);

                ::WIN32_FIND_DATA data;
                ::HANDLE handle;
                handle = ::FindFirstFile(definitionFilePathPattern, &data);
                if (handle != INVALID_HANDLE_VALUE)
                {
                    NN_ABORT_UNLESS(::PathRemoveFileSpec(definitionFilePathPattern) != 0);
                    NN_ABORT_UNLESS(::PathAppend(definitionFilePathPattern, data.cFileName) != 0);
                    NN_ABORT_UNLESS(::WideCharToMultiByte(CP_UTF8, NULL, definitionFilePathPattern, MAX_PATH, path, static_cast<int>(size), NULL, NULL) != 0);
                    ::FindClose(handle);
                    return true;
                }
            }

            // 環境変数 "NINTENDO_SDK_TARGETENV_DEF_FILE_PATH" から取得
            {
                wchar_t env[MAX_PATH];
                if (::GetEnvironmentVariable(L"NINTENDO_SDK_TARGETENV_DEF_FILE_PATH", env, MAX_PATH))
                {
                    NN_ABORT_UNLESS(::WideCharToMultiByte(CP_UTF8, NULL, env, MAX_PATH, path, static_cast<int>(size), NULL, NULL) != 0);
                    return true;
                }
            }

            return false;
#else
            return false;
#endif
        }

        class DefinitionFileInfoManager
        {
        public:
            DefinitionFileInfoManager() NN_NOEXCEPT : m_Mutex(false)
            {
            }

            ~DefinitionFileInfoManager() NN_NOEXCEPT
            {
                while (!m_List.empty())
                {
                    auto p = &(*m_List.rbegin());
                    m_List.erase(m_List.iterator_to(*p));
                    delete p;
                }
            }

            void Add(DefinitionFileInfo* info) NN_NOEXCEPT
            {
                std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
                for (auto& existInfo : m_List)
                {
                    // 既に登録されているプロセス ID だった場合はファイルパスを上書き // TODO: 複数定義ファイル対応
                    if (existInfo.processId == info->processId)
                    {
                        util::Strlcpy(existInfo.path.string, info->path.string, sizeof(existInfo.path.string));
                    }
                }
                m_List.push_back(*info);
            }

            void Remove(DefinitionFileInfo* info) NN_NOEXCEPT
            {
                std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
                m_List.erase(m_List.iterator_to(*info));
                delete info;
            }

            DefinitionFileInfo* GetInfoForWindows(const char* path) NN_NOEXCEPT
            {
                static Path tmpPath;
                util::Strlcpy(tmpPath.string, path, sizeof(tmpPath.string));
                static DefinitionFileInfo info(0, &tmpPath);
                return &info;
            }

            // TODO: 複数定義ファイル対応
            DefinitionFileInfo* GetInfo(Bit64 processId) NN_NOEXCEPT
            {
                if (processId == 0) // For Windows
                {
                    char path[PathLengthMax];
                    if (SearchDefinitionFilePath(path, PathLengthMax))
                    {
                        return GetInfoForWindows(path);
                    }
                }

                std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
                for (auto& info : m_List)
                {
                    if (info.processId == processId)
                    {
                        return &info;
                    }
                }
                return nullptr;
            }

        private:
            typedef util::IntrusiveList<DefinitionFileInfo, util::IntrusiveListBaseNodeTraits<DefinitionFileInfo>> DefinitionFileInfoList;

            DefinitionFileInfoList m_List;
            os::Mutex m_Mutex;
        };

        DefinitionFileInfoManager* GetDefinitionFileInfoManager() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(DefinitionFileInfoManager, g_Manager, );
            return &g_Manager;
        }
    }

    Result RegisterDefinitionFilePath(Bit64 processId, Path* path) NN_NOEXCEPT
    {
        auto info = new DefinitionFileInfo(processId, path);
        NN_RESULT_THROW_UNLESS(info, ResultAllocationMemoryFailed());
        GetDefinitionFileInfoManager()->Add(info);
        NN_RESULT_SUCCESS;
    }

    void UnregisterDefinitionFilePath(Bit64 processId) NN_NOEXCEPT
    {
        if (processId == 0) // For Windows
        {
            return;
        }

        // TODO: 複数定義ファイル対応
        auto info = GetDefinitionFileInfoManager()->GetInfo(processId);
        if (info != nullptr)
        {
            GetDefinitionFileInfoManager()->Remove(info);
        }
    }

    bool GetDefinitionFileInfo(DefinitionFileInfo** outValue, Bit64 processId) NN_NOEXCEPT
    {
        auto info = GetDefinitionFileInfoManager()->GetInfo(processId);
        if (info != nullptr)
        {
            *outValue = info;
            return true;
        }
        return false;
    }

}}}}
