﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/gc.h>

#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_DeviceDetector.h>

namespace nn { namespace gc {
namespace detail {

StateMachine::StateMachine() NN_NOEXCEPT
{
    static StateMachineForAsicHandler s_Instance(this);
    stateThread = &s_Instance;

    m_AsicFirmwareType = AsicFirmwareType_Read;

    stateThread->ResetState();
}

StateMachine::~StateMachine() NN_NOEXCEPT
{
    delete stateThread;
}

StateMachine& StateMachine::GetInstance() NN_NOEXCEPT
{
    static StateMachine s_Instance;
    return s_Instance;
}

GameCardMode StateMachine::GetGameCardMode() NN_NOEXCEPT
{
    return m_GameCardMode;
}

AsicState StateMachine::GetAsicState() NN_NOEXCEPT
{
    return m_AsicState;
}

bool StateMachine::IsCardInserted() NN_NOEXCEPT
{
    // NOTE: AsicHandler(Core) が認識している状態ではなく、実際の状態に近い DeviceDetector のピン情報を返す
    return DeviceDetector::GetInstance().GetGpioDetectPin();
}

StateMachine::StateMachineForAsicHandler& StateMachine::GetInstanceForAsicHandler() NN_NOEXCEPT
{
    return *(StateMachine::GetInstance().stateThread);
}

// *** StateMachineForAsicHandler

void StateMachine::StateMachineForAsicHandler::SelectAsicFirmware(AsicFirmwareType asicFirmwareType) NN_NOEXCEPT
{
    m_pBase->m_AsicFirmwareType = asicFirmwareType;
}

nn::Result StateMachine::StateMachineForAsicHandler::ProcessTransitState(GameCardMode fromState, GameCardMode toState) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(m_pBase->m_GameCardMode == fromState, "Invalid ASIC state transition: %d -> %d\n", m_pBase->m_GameCardMode, toState);
    m_pBase->m_GameCardMode = toState;

    NN_RESULT_SUCCESS;
}

nn::Result StateMachine::StateMachineForAsicHandler::TransitStateWithActivateGameCard() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result =  ProcessTransitState(GameCardMode_Initial, GameCardMode_Normal);
    if(result.IsFailure())
    {
        return nn::fs::ResultGameCardShouldTransitFromInitialToNormal();
    }
    NN_RESULT_SUCCESS;
}

nn::Result StateMachine::StateMachineForAsicHandler::TransitStateWithSecureGameCard() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result = ProcessTransitState(GameCardMode_Normal, GameCardMode_Secure);
    if(result.IsFailure())
    {
        return nn::fs::ResultGameCardShouldTransitFromNormalModeToSecure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result StateMachine::StateMachineForAsicHandler::TransitStateWithDebugGameCard() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result =  ProcessTransitState(GameCardMode_Normal, GameCardMode_Debug);
    if(result.IsFailure())
    {
        return nn::fs::ResultGameCardShouldTransitFromNormalModeToDebug();
    }
    NN_RESULT_SUCCESS;
}

nn::Result StateMachine::StateMachineForAsicHandler::TransitStateWithInvalidateGameCard() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    m_pBase->m_GameCardMode = GameCardMode_Initial;

    NN_RESULT_SUCCESS;
}

StateMachine::StateMachineForAsicHandler::StateMachineForAsicHandler(StateMachine* base) NN_NOEXCEPT
{
    m_pBase = base;
}

nn::Result StateMachine::StateMachineForAsicHandler::TransitStateWithSecureAsic() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    if(m_pBase->m_AsicState != AsicState_Initial)
    {
        NN_SDK_REQUIRES(m_pBase->m_AsicState == AsicState_Initial, "Invalid game card state transition: %d -> %d\n", m_pBase->m_AsicState, AsicState_Secure);
        return nn::fs::ResultGameCardShouldTransitFromAsicInitialToSecure();
    }
    m_pBase->m_AsicState = AsicState_Secure;

    NN_RESULT_SUCCESS;
}

nn::Result StateMachine::StateMachineForAsicHandler::TransitStateWithResetAsic() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    m_pBase->m_AsicState = AsicState_Initial;

    NN_RESULT_SUCCESS;
}



} } }

