﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_AsicHandlerCore.h>
#include <nn/gc/detail/gc_DeviceDetector.h>

namespace nn { namespace gc {
namespace detail {

    AsicHandlerCore::AsicHandlerCore() NN_NOEXCEPT :
        m_pCardId1(&(m_SessionInfo.m_CardInitReceiveData.cardId1)),
        m_pCardUid(&(m_SessionInfo.m_CardSecurityInformation.uid)),
        m_AsicMutex(true)
{
    // このコンストラクタは AsicHandler のインスタンス生成時に呼ばれるので、
    // この段階で Reset 等を呼んではいけない（Reset 内部で Gpio 等をいじる仕様の為）

    std::memset(&m_TotalAsicInfo, 0, sizeof(m_TotalAsicInfo));
    std::memset(&m_CardAccessInternalInfo, 0, sizeof(m_CardAccessInternalInfo));
    std::memset(&m_SessionInfo, 0, sizeof(m_SessionInfo));

    // リセットしない値についてはコンストラクタで初期値を代入する
    m_InitializationResult = nn::ResultSuccess();
    m_InitializationFailureCount = 0;
    m_IsRemovedAfterActivate = false;
}

AsicHandlerCore& AsicHandlerCore::GetInstance() NN_NOEXCEPT
{
    static AsicHandlerCore s_Instance;
    return s_Instance;
}

void AsicHandlerCore::Reset() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 外部クラスのリセット
    StateMachine::GetInstanceForAsicHandler().ResetState();
    AsicRegister::GetInstance().ResetRegister();
    GcCrypto::GetInstance().ClearKeys();

    ClearSessionInfo();

    // ASIC をリセットしつつ無効化する
    nn::Result result = AsicHandlerProcedure::DeactivateAsicIo();
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("Failed to deactivate ASIC IO (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}

nn::Result AsicHandlerCore::ProcessInsert() NN_NOEXCEPT
{
    // カードを挿入するタイミングでエラーレポート向けの情報をリセット
    ClearSessionInfoForErrorReport();
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::ProcessRemove() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    m_IsRemovedAfterActivate = true;

    // deactivate を行う
    this->DeactivateCard();

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::InitializeAsicIfNeeded() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 強制リセットフラグが立っていたらリセットする
    if(StateMachine::GetInstance().IsForceReset())
    {
        DeactivateCard();
    }

    // 初期化されていなかったら初期化する
    if(StateMachine::GetInstance().IsAsicSecure() == false)
    {
        // カード抜去に関するクリティカルセクション
        {
            NN_DETAIL_GC_CANCEL_SDMMC_LOCK_GUARD;

            // カード抜去により sdmmc 通信を中断した可能性があるため、クリアしておく
            DataIo::GetInstance().ClearRemovedEvent();

            // Deactivate()->Reset() でリセットされているはずだが、明示的にリセットしておく：Activate するまでカード抜去による通信中断を禁止
            m_SessionInfo.m_IsCardBusEnabled = false;

            //（カード抜去による sdmmc 通信中断後は Deactivate（Reset, 電源OFF）するだけで最初の通信は
            //  InitializeAsic になるはずなので、ここで sdmmc の通信中断イベントを Clear すればよい）
        }

        // セッション構築
        if(nn::fs::ResultGameCardSplFailure::Includes(m_InitializationResult))
        {
            NN_DETAIL_GC_ERR_LOG("asic initialization spl failure: initialization skipped.\n");
        }
        else if(m_InitializationResult.IsFailure() && m_InitializationFailureCount >= InitializationFailureMaxCount)
        {
            NN_DETAIL_GC_ERR_LOG("asic initialization retry failure: initialization skipped.\n");
        }
        else
        {
            // ASIC はリセットされている状態のため InitializeAsicImpl を単品で発行
            m_InitializationResult = AsicHandlerProcedure::InitializeAsicImpl();
            m_TotalAsicInfo.initializeCount++;
            if(StateMachine::GetInstance().IsReaderFwSet()) // Writer FW のときはリトライしない
            {
                while(m_InitializationResult.IsFailure() && m_InitializationFailureCount < InitializationFailureMaxCount)
                {
                    NN_DETAIL_GC_ERR_LOG("asic initialization failed. (Module:%d, Description:%d)\n", m_InitializationResult.GetModule(), m_InitializationResult.GetDescription());

                    // 後始末をしておかないと ASIC が変な状態で Initialize しようとしてしまうため Reset してから初期化する
                    this->Reset();
                    m_InitializationResult = AsicHandlerProcedure::InitializeAsicImpl();
                    m_InitializationFailureCount++;
                    m_TotalAsicInfo.initializeCount++;
                }
            }
            // 初期化失敗の result は nn::fs::ResultGameCardInitializeAsicFailure でハンドリングできるようにしておく
            if( m_InitializationResult.IsFailure() && (!nn::fs::ResultGameCardInitializeAsicFailure::Includes(m_InitializationResult)) )
            {
                m_InitializationResult = nn::fs::ResultGameCardInitializeAsicFailure();
            }
        }
        return m_InitializationResult;
    }

    NN_RESULT_SUCCESS;
}

void AsicHandlerCore::SetSplFatalErrorOccurredFlag(nn::Result result) NN_NOEXCEPT
{
    m_InitializationResult = result;
    if(nn::fs::ResultGameCardSplFailure::Includes(result) == false)
    {
        NN_DETAIL_GC_ERR_LOG("Set spl request with unknown result (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        m_InitializationResult = nn::fs::ResultGameCardSplFailure();
    }
}


// *** internal API
// AsicHandler の仕事として呼ばれる API

nn::Result AsicHandlerCore::GetCardStatus(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 引数チェック
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo != nullptr);
    ReadInfo& rinfo = *(pReadInfo);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outDataBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.dataBufferSize >= sizeof(GameCardStatus));

    GameCardStatus status;
    memcpy(status.packageId, m_SessionInfo.m_CardHeader.packageId, GcPackageIdSize);
    status.partitionFsHeaderAddress = m_SessionInfo.m_CardHeader.partitionFsHeaderAddress;
    status.partitionFsHeaderSize = m_SessionInfo.m_CardHeader.partitionFsHeaderSize;
    memcpy(status.partitionFsHeaderHash, m_SessionInfo.m_CardHeader.partitionFsHeaderHash, GcPartitionFsHeaderHashSize);

    status.cardSize        = GetCardSize();
    status.cupVersion      = m_SessionInfo.m_CardInitReceiveData.cupVersion;
    status.cupId           = m_SessionInfo.m_CardHeader.cupId;
    status.normalAreaSize  = GetNormalAreaSize();
    status.secureAreaSize  = status.cardSize - status.normalAreaSize;
    status.flags           = m_SessionInfo.m_CardHeader.flags;
    NN_DETAIL_GC_LOG("size: %" PRIu64 "\n", status.cardSize);

    // コピーして戻す
    memcpy(rinfo.outDataBuffer, &status, sizeof(GameCardStatus));

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::GetCardDeviceId(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 引数チェック
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo != nullptr);
    ReadInfo& rinfo = *(pReadInfo);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outDataBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.dataBufferSize >= GcCardDeviceIdSize);

    return GetCardDeviceId(rinfo.outDataBuffer, rinfo.dataBufferSize);
}

nn::Result AsicHandlerCore::GetCardDeviceCertificate(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    // 引数チェック
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo != nullptr);
    ReadInfo& rinfo = *(pReadInfo);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outDataBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.dataBufferSize >= GcDeviceCertificateSize);

    // Secure モードでなければ証明書は読めない
    if(!StateMachine::GetInstance().IsCardSecure())
    {
        NN_DETAIL_GC_ERR_LOG("GetCardDeviceCertificate Card ModeError\n");
        return nn::fs::ResultGameCardInvalidGetCardDeviceCertificate();
    }

    memcpy(rinfo.outDataBuffer, m_SessionInfo.m_CardSecurityInformation.cardCerificate, GcDeviceCertificateSize);
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::GetCardImageHash(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    // 引数チェック
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo != nullptr);
    ReadInfo& rinfo = *(pReadInfo);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outDataBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.dataBufferSize == GcCardImageHashSize);

    memcpy(rinfo.outDataBuffer, m_SessionInfo.m_CardImageHash, GcCardImageHashSize);
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::GetCardIdSet(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    // 引数チェック
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo != nullptr);
    ReadInfo& rinfo = *(pReadInfo);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outDataBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.dataBufferSize >= sizeof(GameCardIdSet));

    size_t offset = 0;
    memcpy(rinfo.outDataBuffer + offset, m_pCardId1, sizeof(detail::CardId1));
    offset += sizeof(detail::CardId1);
    memcpy(rinfo.outDataBuffer + offset, &(m_SessionInfo.m_CardId2), sizeof(detail::CardId2));
    offset += sizeof(detail::CardId2);
    memcpy(rinfo.outDataBuffer + offset, &(m_SessionInfo.m_CardId3), sizeof(detail::CardId3));
    NN_RESULT_SUCCESS;
}


// *** internal API


bool AsicHandlerCore::IsRemoved() NN_NOEXCEPT
{
    // m_IsRemovedAfterActivate が false の時は GPIO イベントが来ていたら抜けた判定をする
    //（m_IsRemovedAfterActivate は AsicHandlerCore がメインループを回らないと更新されないので）
    if(m_IsRemovedAfterActivate == false)
    {
        if(nn::os::TryWaitEvent(DeviceDetector::GetInstance().GetDetectEvent()))
        {
            return true;
        }
    }
    return m_IsRemovedAfterActivate;
}

uint64_t AsicHandlerCore::GetCardSize() NN_NOEXCEPT
{
    switch(m_pCardId1->memoryCapacity)
    {
    case MemoryCapacity_1GB  : return  1 * AvailableSizeBase;
    case MemoryCapacity_2GB  : return  2 * AvailableSizeBase;
    case MemoryCapacity_4GB  : return  4 * AvailableSizeBase;
    case MemoryCapacity_8GB  : return  8 * AvailableSizeBase;
    case MemoryCapacity_16GB : return 16 * AvailableSizeBase;
    case MemoryCapacity_32GB : return 32 * AvailableSizeBase;
    default: return 0;
    }
}

uint64_t AsicHandlerCore::GetNormalAreaSize() NN_NOEXCEPT
{
    uint64_t normalAreaPageSize = m_SessionInfo.m_CardHeader.limArea;
    return normalAreaPageSize * GcPageSize;
}

nn::Result AsicHandlerCore::GetCardDeviceId(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(bufferLength >= GcCardDeviceIdSize);

    if(StateMachine::GetInstance().GetGameCardMode() != GameCardMode_Secure)
    {
        return nn::fs::ResultGameCardStateCardSecureModeRequired();
    }

    if(m_SessionInfo.m_CardHeader.selSec == SelectSecurityMode_T1)
    {
        memcpy(outBuffer, m_SessionInfo.m_CardSecurityInformation.t1CardCertificate.t1CardDeviceId, GcCardDeviceIdSize);
    }
    else if(m_SessionInfo.m_CardHeader.selSec == SelectSecurityMode_T2)
    {
        // TODO: TEMP:
        NN_DETAIL_GC_ERR_LOG("T2 is not implemented\n");
        return nn::fs::ResultGameCardNotImplemented();
    }
    else
    {
        return nn::fs::ResultGameCardUnregisteredCardSecureMethod();
    }

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::InvalidateAsicIfResultFailure(nn::Result result) NN_NOEXCEPT
{
    if(result.IsFailure())
    {
        StateMachine::GetInstanceForAsicHandler().SetForceResetFlag();
        NN_DETAIL_GC_RESULT_THROW_IF_REMOVED;
    }
    return result;
}


// ** DeviceDetector スレッドから呼ばれる

// カード抜去時に呼んでもらうコールバック
void AsicHandlerCore::ProcessCardRemovalForCallback(void *pParameter) NN_NOEXCEPT
{
    AsicHandlerCore* handler = reinterpret_cast<AsicHandlerCore*>(pParameter);
    handler->AbortCommunication();
}

// カード抜去時に sdmmc の転送中断を行う
void AsicHandlerCore::AbortCommunication() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // カード抜去に関するクリティカルセクション
    {
        NN_DETAIL_GC_CANCEL_SDMMC_LOCK_GUARD;

        // カード抜去時、カードに関する通信は直ちに中止する必要がある（さもなくばタイムアウトになりうる）
        // しかし Asic のセッション構築時、レジスタリード時などで不必要にセッションを破壊してしまうのは避けたい
        // そこで、カードバスを有効化したかどうかを境目にする
        if(IsCardBusEnabled())
        {
            // sdmmc ドライバに転送中断を依頼する（メインスレッドでは ResultDeviceRemoved が帰る）
            DataIo::GetInstance().SignalRemovedEvent();
        }
    }
}

} } }
