﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedBinaryMatch.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>

namespace nn { namespace fssystem { namespace utilTool {

namespace {

typedef RelocationTable::MatchResult MatchResult;

}

RelocationTable::RelocationTable() NN_NOEXCEPT
    : m_RegionInfo()
    , m_Entries()
    , m_IsCommitted(false)
{
}

void RelocationTable::Add(const Entry& entry) NN_NOEXCEPT
{
    m_Entries.push_back(entry);
    m_IsCommitted = false;
}

Result RelocationTable::Build(std::shared_ptr<Progress>* pOutProgress, const BuildInfo& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutProgress);

    if( info.IsOverlappedStorage() )
    {
        const auto range = info.GetOverlapStorageRange();

        // oldStorage はオーバーラップした領域以降を比較対象とする
        fs::SubStorage storage(&info.GetOldStorage(), range.first, range.second);

        std::vector<MatchResult> results;
        {
            Match match(info.GetBlockSize(), info.GetRegionSize(), info.GetMatchSize());
            match.SetRegion(m_RegionInfo.first);

            *pOutProgress = match.GetProgress();

            NN_RESULT_DO(match.Run(storage, info.GetNewStorage(), info.GetShiftSize()));

            results = match.GetResult();
            m_RegionInfo = match.GetRegionInfo();
        }

        OptimizeBinaryMatchResult(&results); // oldOffset の重複領域を取り除く
        MatchResult::Merge(&results);        // 連続する領域をマージ

        NN_RESULT_DO(MatchResult::Verify(results));

        BuildImpl(std::move(results), info, pOutProgress->get());

        NN_SDK_ASSERT_LESS_EQUAL(
            m_Entries.back().GetSrcOffset() + m_Entries.back().GetSize(),
            info.GetNewStorageSize() + info.GetExpandableSize()
        );
    }
    // 領域がまったくオーバーラップしていなかった
    else
    {
        const auto size = info.GetNewStorageSize();
        if( 0 < size )
        {
            RelocationTable::Entry data(0, 0, size, BinaryMatchDetail_RightJustify, 0);
            m_Entries.push_back(data);
        }
    }

    return Commit();
}

void RelocationTable::BuildImpl(std::vector<MatchResult>&& results, const BuildInfo& info, BinaryMatchProgress* pProgress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(results.size(), static_cast<size_t>(0));
    NN_SDK_REQUIRES_NOT_NULL(pProgress);

    pProgress->SetPhase(BinaryMatchPhase::Optimize, int(BinaryMatchOptimizePhase::Count));

    const auto adjustOldOffset = info.GetAdjustOldOffset();
    const auto limitOldOffset = info.GetLimitOldOffset();
    const auto limitNewOffset = info.GetLimitNewOffset();
    const auto expandableSize = info.GetExpandableSize();

    // 出力するデータストレージがずれていないか、前にずれている場合のみ対応
    NN_SDK_ASSERT_GREATER_EQUAL(adjustOldOffset, 0);
    NN_SDK_ASSERT_GREATER(limitOldOffset, 0);
    NN_SDK_ASSERT_GREATER(limitNewOffset, 0);
    NN_SDK_ASSERT_GREATER(expandableSize, 0);

    // 移動が必要なデータを移動する
    ShiftBinaryMatchResult(&results, limitOldOffset, limitNewOffset, expandableSize, pProgress);

    // newStorage が前にずれている場合に対応
    if( 0 < adjustOldOffset )
    {
        // BinaryMatch のリザルトを new 側から見たアドレスにずらす
        for( auto& result : results )
        {
            if( result.oldOffset != MatchResult::InvalidOffset )
            {
                result.oldOffset += adjustOldOffset;
            }
        }
    }

    // oldOffset が有効(oldOffset 昇順)、 無効(newOffset 昇順) の順番でソート
    std::sort(results.begin(), results.end(), PartitionOrderFunctor());

    // 新ストレージにしかないデータの開始位置を取得
    const auto point =
        std::partition_point(results.begin(), results.end(), PartitionPointFunctor());

    m_Entries.clear();

    InsertEntry(results, static_cast<size_t>(std::distance(results.begin(), point)));
    pProgress->SetValue(int(BinaryMatchOptimizePhase::Complete));
}

// 使用しない古い空き領域に新しいデータを分割して挿入していきます。
void RelocationTable::InsertEntry(
                          const std::vector<MatchResult>& results,
                          size_t partitionIndex
                      ) NN_NOEXCEPT
{
    int64_t currentOffset = 0;
    int64_t insertedSize = 0;
    auto insertIndex = partitionIndex;
    int detailIndex = 0;
    const auto resultCount = results.size();

    for( size_t i = 0; i < partitionIndex; ++i )
    {
        const auto result = results[i];
        NN_SDK_ASSERT_NOT_EQUAL(result.oldOffset, MatchResult::InvalidOffset);
        NN_SDK_ASSERT_LESS_EQUAL(currentOffset, result.oldOffset);

        // 新しいデータを空き領域に挿入する
        if( insertIndex < resultCount )
        {
            // 単純に前詰め
            while( currentOffset < result.oldOffset )
            {
                const auto insert = results[insertIndex];
                NN_SDK_ASSERT_EQUAL(insert.oldOffset, MatchResult::InvalidOffset);

                const auto insertSize =
                    std::min(result.oldOffset - currentOffset, insert.size - insertedSize);

                RelocationEntry data(
                    insert.newOffset + insertedSize,
                    currentOffset,
                    insertSize,
                    BinaryMatchDetail_ForceSplit,
                    detailIndex
                );
                m_Entries.push_back(data);

                ++detailIndex;
                insertedSize += insertSize;

                if( insertedSize == insert.size )
                {
                    detailIndex = 0;
                    insertedSize = 0;

                    ++insertIndex;
                    if( insertIndex == resultCount )
                    {
                        break; // 挿入データ終了
                    }
                }

                currentOffset += insertSize;
            }
        }
        NN_SDK_ASSERT_LESS_EQUAL(currentOffset, result.oldOffset);

        NN_SDK_ASSERT(result.detail == BinaryMatchDetail_Match
                        ? result.storageIndex == 0 : result.storageIndex != 0);

        // エントリーを追加
        RelocationEntry data(
            result.newOffset,
            result.oldOffset,
            result.size,
            static_cast<BinaryMatchDetail>(result.detail),
            result.detailIndex
        );
        m_Entries.push_back(data);

        currentOffset = result.oldOffset + result.size;
    }

    // 挿入途中のデータを追加
    if( 0 < insertedSize )
    {
        const auto insert = results[insertIndex];
        NN_SDK_ASSERT_LESS(insertedSize, insert.size);

        RelocationEntry data(
            insert.newOffset + insertedSize,
            currentOffset,
            insert.size - insertedSize,
            BinaryMatchDetail_ForceSplit,
            detailIndex
        );
        m_Entries.push_back(data);

        ++insertIndex;

        currentOffset += insert.size - insertedSize;
    }

    // 残りのデータを追加
    for( size_t i = insertIndex; i < resultCount; ++i )
    {
        const auto insert = results[i];
        NN_SDK_ASSERT_EQUAL(insert.oldOffset, MatchResult::InvalidOffset);

        RelocationEntry data(
            insert.newOffset,
            currentOffset,
            insert.size,
            BinaryMatchDetail_RightJustify,
            0
        );
        m_Entries.push_back(data);

        currentOffset += insert.size;
    }
}

Result RelocationTable::Commit() NN_NOEXCEPT
{
    std::sort(
        m_Entries.begin(),
        m_Entries.end(),
        [](const Entry& lhs, const Entry& rhs) NN_NOEXCEPT
        {
            return lhs.GetSrcOffset() < rhs.GetSrcOffset();
        }
    );

    m_IsCommitted = true;

    return Verify();
}

Result RelocationTable::Verify() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsCommitted);

    const auto end = m_Entries.end();
    for( auto iter = m_Entries.begin(); iter != end; ++iter )
    {
        const auto detail = iter->GetDetail();

        // ステータスのチェック
        NN_RESULT_THROW_UNLESS(
            BinaryMatchDetail_Unknown < detail && detail < BinaryMatchDetail_Max,
            nn::fs::ResultDataCorrupted()
        );

        const auto offset = iter->GetSrcOffset();
        const auto size = iter->GetSize();

        NN_RESULT_THROW_UNLESS(0 < size, nn::fs::ResultDataCorrupted());

        auto next = iter;
        ++next;

        // 入力空間を隙間なくカバーできているかチェック
        NN_RESULT_THROW_UNLESS(
            next == m_Entries.end() || (offset + size) == next->GetSrcOffset(),
            nn::fs::ResultDataCorrupted()
        );
    }

    NN_RESULT_SUCCESS;
}

// 指定した範囲にテーブルを適用する
Result RelocationTable::ApplyTo(
                            std::vector<Range>* pOutRanges,
                            const Range& range
                        ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRanges);

    pOutRanges->clear();

    NN_RESULT_THROW_UNLESS(range.first <= range.second, nn::fs::ResultInvalidArgument());
    if( range.first == range.second )
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(IsValidRange(range), nn::fs::ResultInvalidArgument());

    auto iterBegin = Find(range.first);
    NN_RESULT_THROW_UNLESS(iterBegin != end(), fs::ResultInvalidOffset());

    auto iterEnd = Find(range.second - 1);
    NN_RESULT_THROW_UNLESS(iterEnd != end(), fs::ResultInvalidOffset());

    ++iterEnd;

    for( auto iter = iterBegin; iter != iterEnd; ++iter )
    {
        const auto begin = std::max(range.first, iter->GetSrcOffset());
        const auto end = std::min(range.second, iter->GetSrcOffset() + iter->GetSize());

        pOutRanges->push_back(iter->ToDstRange(std::make_pair(begin, end)));
    }

    NN_RESULT_SUCCESS;
}

// 指定した範囲にテーブルを適用する
Result RelocationTable::ApplyTo(
                            std::vector<PaddingRange>* pOutRanges,
                            const Range& range
                        ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRanges);

    pOutRanges->clear();

    NN_RESULT_THROW_UNLESS(range.first <= range.second, nn::fs::ResultInvalidArgument());
    if( range.first == range.second )
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(IsValidRange(range), nn::fs::ResultInvalidArgument());

    auto iterBegin = Find(range.first);
    NN_RESULT_THROW_UNLESS(iterBegin != end(), fs::ResultInvalidOffset());

    auto iterEnd = Find(range.second - 1);
    NN_RESULT_THROW_UNLESS(iterEnd != end(), fs::ResultInvalidOffset());

    ++iterEnd;

    for( auto iter = iterBegin; iter != iterEnd; ++iter )
    {
        const auto begin = std::max(range.first, iter->GetSrcOffset());
        const auto end = std::min(range.second, iter->GetSrcOffset() + iter->GetSize());

        pOutRanges->push_back(std::make_pair(
            iter->IsPadding(),
            iter->ToDstRange(std::make_pair(begin, end))
        ));
    }

    NN_RESULT_SUCCESS;
}

// 指定したデータにテーブルを適用する
Result RelocationTable::ApplyTo(
                            fs::IStorage* pDstStorage,
                            fs::IStorage* pSrcStorage,
                            void* buffer,
                            size_t bufferSize
                        ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDstStorage);
    NN_SDK_REQUIRES_NOT_NULL(pSrcStorage);
    NN_SDK_REQUIRES(pDstStorage != pSrcStorage);

    int64_t srcSize;
    NN_RESULT_DO(pSrcStorage->GetSize(&srcSize));
    int64_t dstSize;
    NN_RESULT_DO(pDstStorage->GetSize(&dstSize));

    NN_RESULT_THROW_UNLESS(dstSize == srcSize, fs::ResultInvalidSize());

    for( int64_t offset = 0; offset < srcSize; )
    {
        const auto readSize =
            static_cast<size_t>(std::min<int64_t>(bufferSize, srcSize - offset));
        NN_RESULT_DO(pSrcStorage->Read(offset, buffer, readSize));

        // 読み取った範囲に RelocationTable を適用し、書き込み範囲を得る
        std::vector<Range> relocatedRanges;
        NN_RESULT_DO(ApplyTo(&relocatedRanges, std::make_pair(offset, offset + readSize)));

        // 各書き込み範囲に書き込みを実行
        {
            int64_t writeOffset = 0;
            for( const auto& range : relocatedRanges )
            {
                NN_SDK_ASSERT_MINMAX(
                    range.second - range.first, 0, static_cast<int64_t>(readSize));

                const auto writeSize = static_cast<size_t>(range.second - range.first);
                NN_RESULT_DO(pDstStorage->Write(
                    range.first, reinterpret_cast<char*>(buffer) + writeOffset, writeSize));

                writeOffset += writeSize;
            }
        }

        offset += readSize;
    }

    NN_RESULT_SUCCESS;
}

// src と dst を入れ替えたテーブルを作成する
nn::Result RelocationTable::Invert(
                                RelocationTable* pOutTable,
                                int64_t* pOutSrcSize,
                                int64_t baseSrcOffset,
                                int64_t baseDstOffset
                            ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutTable);
    NN_SDK_REQUIRES(pOutTable != this);
    NN_SDK_REQUIRES_NOT_NULL(pOutSrcSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(baseSrcOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(baseDstOffset, 0);
    NN_SDK_REQUIRES(m_IsCommitted);

    const auto adjustOffset = baseSrcOffset - baseDstOffset;
    int64_t srcOffset = 0;

    auto entries = m_Entries;
    if( !entries.empty() )
    {
        srcOffset = entries.back().GetSrcOffset() + entries.back().GetSize();

        std::sort(
            entries.begin(),
            entries.end(),
            [](const Entry& lhs, const Entry& rhs) NN_NOEXCEPT
            {
                return lhs.GetDstOffset() < rhs.GetDstOffset();
            }
        );

        int64_t dstOffset = 0;
        const auto entryCount = entries.size();

        for( size_t i = 0; i < entryCount; ++i )
        {
            const auto entry = entries[i];

            auto size = entry.GetDstOffset() - dstOffset;

            // dst 側の穴あき部分を埋める
            if( 0 < size )
            {
                entries.push_back(
                    Entry(dstOffset, dstOffset + adjustOffset, size, BinaryMatchDetail_Padding, 0)
                );

                srcOffset += size;
            }

            entries[i].SwapOffset();

            dstOffset = entry.GetDstOffset() + entry.GetSize();
        }
    }

    *pOutSrcSize = srcOffset;

    pOutTable->m_Entries = std::move(entries);

    return pOutTable->Commit();
}

RelocationTable::const_iterator RelocationTable::Find(int64_t offset) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsCommitted);

    if( !IsValidOffset(offset) )
    {
        return m_Entries.end();
    }

    return std::lower_bound(
        m_Entries.begin(),
        m_Entries.end(),
        offset,
        [](const Entry& entry, int64_t srcOffset) NN_NOEXCEPT
        {
            return entry.GetSrcOffset() + entry.GetSize() <= srcOffset;
        }
    );
}

// 世代が同じエントリーをまとめてアライメントの調整をする
Result OptimizeAesCtrCounterExtendedEntry(
           std::vector<AesCtrCounterExtendedStorage::Entry>* pEntries,
           int64_t endOffset,
           int generation
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEntries);
    NN_SDK_REQUIRES_GREATER_EQUAL(endOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(generation, 0);

    auto& entries = *pEntries;
    if( !entries.empty() )
    {
        const auto srcCount = entries.size();
        size_t dstIndex = 1;

        for( size_t srcIndex = 1; srcIndex < srcCount; ++srcIndex )
        {
            NN_SDK_ASSERT_LESS(entries[srcIndex - 1].GetOffset(), entries[srcIndex].GetOffset());

            // 同じ世代番号のエントリーはスキップ（マージ相当）
            if( entries[dstIndex - 1].generation == entries[srcIndex].generation )
            {
                continue;
            }

            auto entry = entries[srcIndex];
            auto entryOffset = entry.GetOffset();

            if( endOffset <= entryOffset )
            {
                break;
            }

            if( !util::is_aligned(entryOffset, AesCtrCounterExtendedStorage::BlockSize) )
            {
                // オフセットを切り下げる
                if( entry.generation == generation )
                {
                    NN_SDK_ASSERT(entries[dstIndex - 1].generation != generation);

                    entryOffset = util::align_down(entryOffset, AesCtrCounterExtendedStorage::BlockSize);

                    // 1 つ前のエントリーを無かったことにする
                    if( entries[dstIndex - 1].GetOffset() == entryOffset )
                    {
                        --dstIndex;
                    }
                }
                // オフセットを切り上げる
                else if( entries[dstIndex - 1].generation == generation )
                {
                    entryOffset = util::align_up(entryOffset, AesCtrCounterExtendedStorage::BlockSize);

                    if( endOffset <= entryOffset )
                    {
                        break;
                    }

                    // このエントリーを無かったことにする
                    if( (srcIndex + 1 < srcCount) && (entries[srcIndex + 1].GetOffset() == entryOffset) )
                    {
                        continue;
                    }
                }
                else
                {
                    // 過去の処理で BlockSize アライメントされているはずなのでエラーにする
                    NN_RESULT_THROW(fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
                }

                entry.SetOffset(entryOffset);
            }

            entries[dstIndex] = entry;
            ++dstIndex;
        }

        entries.resize(dstIndex);
    }

    NN_RESULT_SUCCESS;
}

namespace
{

class AesCtrCounterExtendedEntry
{
public:
    AesCtrCounterExtendedEntry(int64_t offset, int64_t size, uint32_t generation) NN_NOEXCEPT
        : m_Offset(offset)
        , m_Size(size)
        , m_Generation(generation)
    {
    }
    AesCtrCounterExtendedEntry() NN_NOEXCEPT
        : m_Offset(0)
        , m_Size(0)
        , m_Generation(0)
    {
    }
    int64_t GetOffset() const NN_NOEXCEPT
    {
        return m_Offset;
    }
    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }
    uint32_t GetGeneration() const NN_NOEXCEPT
    {
        return m_Generation;
    }

private:
    int64_t m_Offset;
    int64_t m_Size;
    uint32_t m_Generation;
};

Result MakeAesCtrCounterExtendedEntries(
           std::vector<AesCtrCounterExtendedEntry>* pOutValue,
           const BucketTree& bucketTree
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BucketTree::Visitor visitor;
    NN_RESULT_DO(bucketTree.Find(&visitor, 0));

    std::vector<AesCtrCounterExtendedEntry> entries;
    while( NN_STATIC_CONDITION(true) )
    {
        const auto entry = *visitor.Get<AesCtrCounterExtendedStorage::Entry>();
        const auto offset = entry.GetOffset();
        bool isToBreak = false;

        int64_t nextOffset;
        if( visitor.CanMoveNext() )
        {
            NN_RESULT_DO(visitor.MoveNext());
            nextOffset = visitor.Get<AesCtrCounterExtendedStorage::Entry>()->GetOffset();
        }
        else
        {
            nextOffset = bucketTree.GetEnd();
            isToBreak = true;
        }

        entries.emplace_back(offset, nextOffset - offset, entry.generation);

        if( isToBreak )
        {
            break;
        }
    }

    *pOutValue = std::move(entries);

    NN_RESULT_SUCCESS;
}

}

Result VerifyAesCtrCounterExtendedTable(
           MemoryResource* pAllocator,
           int64_t currentOffset,
           const void* pCurrentHeader,
           size_t currentHeaderSize,
           fs::IStorage* pCurrentTableStorage,
           int64_t currentIndirectTableOffset,
           int64_t currentIndirectTableSize,
           int64_t previousOffset,
           const void* pPreviousHeader,
           size_t previousHeaderSize,
           fs::IStorage* pPreviousTableStorage,
           uint32_t generationForUpdatedRegion,
           const RelocationTable& invertedRelocatinTable
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_GREATER_EQUAL(currentOffset, 0);
    NN_SDK_REQUIRES_NOT_NULL(pCurrentHeader);
    NN_SDK_REQUIRES(currentHeaderSize == sizeof(BucketTree::Header));
    NN_SDK_REQUIRES_NOT_NULL(pCurrentTableStorage);
    NN_SDK_REQUIRES_GREATER_EQUAL(currentIndirectTableOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(currentIndirectTableSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(previousOffset, 0);
    NN_SDK_REQUIRES(pPreviousHeader == nullptr ||
                    previousHeaderSize == sizeof(BucketTree::Header));
    NN_SDK_REQUIRES(pPreviousHeader == nullptr ||
                    pPreviousTableStorage != nullptr);
    NN_UNUSED(currentHeaderSize);
    NN_UNUSED(previousHeaderSize);

    std::vector<AesCtrCounterExtendedEntry> previousEntries;
    // Previous の {offset, size, generation} のテーブルを作る
    if( pPreviousHeader != nullptr )
    {
        const auto& bucketTreeHeader =
            *reinterpret_cast<const BucketTree::Header*>(pPreviousHeader);
        NN_RESULT_DO(bucketTreeHeader.Verify());

        const auto entryCount = bucketTreeHeader.entryCount;
        const int64_t nodeOffset = 0;
        const auto nodeSize = AesCtrCounterExtendedStorage::QueryNodeStorageSize(entryCount);
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = AesCtrCounterExtendedStorage::QueryEntryStorageSize(entryCount);

        BucketTree bucketTree;
        NN_RESULT_DO(bucketTree.Initialize(
            pAllocator,
            fs::SubStorage(pPreviousTableStorage, nodeOffset, nodeSize),
            fs::SubStorage(pPreviousTableStorage, entryOffset, entrySize),
            AesCtrCounterExtendedStorage::NodeSize,
            sizeof(AesCtrCounterExtendedStorage::Entry),
            entryCount
        ));

        NN_RESULT_DO(MakeAesCtrCounterExtendedEntries(&previousEntries, bucketTree));
    }

    std::vector<AesCtrCounterExtendedEntry> currentEntries;
    // Current の {offset, size, generation} のテーブルを作る
    {
        const auto& bucketTreeHeader =
            *reinterpret_cast<const BucketTree::Header*>(pCurrentHeader);
        NN_RESULT_DO(bucketTreeHeader.Verify());

        const auto entryCount = bucketTreeHeader.entryCount;
        const int64_t nodeOffset = 0;
        const auto nodeSize = AesCtrCounterExtendedStorage::QueryNodeStorageSize(entryCount);
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = AesCtrCounterExtendedStorage::QueryEntryStorageSize(entryCount);

        BucketTree bucketTree;
        NN_RESULT_DO(bucketTree.Initialize(
            pAllocator,
            fs::SubStorage(pCurrentTableStorage, nodeOffset, nodeSize),
            fs::SubStorage(pCurrentTableStorage, entryOffset, entrySize),
            AesCtrCounterExtendedStorage::NodeSize,
            sizeof(AesCtrCounterExtendedStorage::Entry),
            entryCount
        ));

        NN_RESULT_DO(MakeAesCtrCounterExtendedEntries(&currentEntries, bucketTree));
    }

    // マッチングの結果を格納したテーブル
    auto matchingTable = invertedRelocatinTable;
    if( !matchingTable.empty() )
    {
        const auto& last = matchingTable.back();
        const auto expectedIndirectTableOffset = last.GetSrcOffset() + last.GetSize();

        // アライメントによるパディング分を追加
        if( expectedIndirectTableOffset < currentIndirectTableOffset )
        {
            matchingTable.Add(
                RelocationTable::Entry(
                    expectedIndirectTableOffset,
                    expectedIndirectTableOffset,
                    currentIndirectTableOffset - expectedIndirectTableOffset,
                    BinaryMatchDetail_Padding,
                    0
                )
            );
        }
    }

    // IndirectStorage のテーブル分の領域を追加
    matchingTable.Add(
        RelocationTable::Entry(
            currentIndirectTableOffset,
            currentIndirectTableOffset,
            currentIndirectTableSize,
            BinaryMatchDetail_RightJustify,
            0
        )
    );
    NN_RESULT_DO(matchingTable.Commit());

    // Current の 各エントリーについて
    // 対応する RelocationTable のエントリーを引っ張ってくる
    // 引っ張ってきたエントリーは全て IsMatched か、全てそうでないか、のみ
    // IsMatched でなければ Current の generation は generation
    // IsMatched の場合、対応する Previous の各エントリーを引っ張ることができ、generation は一致する
    for( const auto& currentEntry : currentEntries )
    {
        NN_RESULT_THROW_UNLESS(
            util::is_aligned(currentEntry.GetOffset(), AesCtrCounterExtendedStorage::BlockSize),
            fs::ResultInvalidAesCtrCounterExtendedEntryOffset()
        );

        const auto currentBegin = currentEntry.GetOffset() + currentOffset;
        const auto currentEnd = currentBegin + currentEntry.GetSize();

        bool isMatched = false;
        bool isPadding = false;
        {
            int64_t tableBegin = std::numeric_limits<int64_t>::max();
            int64_t tableEnd = std::numeric_limits<int64_t>::min();
            bool isSetMatched = false;

            // 対応する RelocationTable のエントリーを引っ張ってくる
            for( auto tableIter = matchingTable.Find(currentEntry.GetOffset());
                 tableIter != matchingTable.end(); ++tableIter )
            {
                const auto& tableEntry = *tableIter;

                const auto begin = tableEntry.GetSrcOffset() + currentOffset;
                const auto end = begin + tableEntry.GetSize();

                // オーバーラップを求める
                const auto overlapBegin = std::max(currentBegin, begin);
                const auto overlapEnd = std::min(currentEnd, end);
                const auto overlapSize = overlapEnd - overlapBegin;

                if( overlapSize <= 0 )
                {
                    break;
                }

                tableBegin = std::min(overlapBegin, tableBegin);
                tableEnd = std::max(overlapEnd, tableEnd);

                if( tableEntry.IsPadding() )
                {
                    if( !isSetMatched )
                    {
                        isPadding = true;
                    }
                }
                else
                {
                    if( !isSetMatched )
                    {
                        isSetMatched = true;
                        isMatched = tableEntry.IsMatched();
                        isPadding = false;
                    }
                    else
                    {
                        // 全て同じ matched のはず
                        NN_RESULT_THROW_UNLESS(
                            isMatched == tableEntry.IsMatched(),
                            fs::ResultInvalidAesCtrCounterExtendedEntryOffset()
                        );
                    }
                }
            }

            NN_RESULT_THROW_UNLESS(currentBegin == tableBegin,
                                   fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
            NN_RESULT_THROW_UNLESS(currentEnd == tableEnd,
                                   fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
            NN_RESULT_THROW_UNLESS(isPadding || isSetMatched,
                                   fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
        }

        if( !isMatched )
        {
            if( !isPadding )
            {
                // IsMatched でないなら generation は generation と一致
                NN_RESULT_THROW_UNLESS(currentEntry.GetGeneration() == generationForUpdatedRegion,
                                       fs::ResultInvalidAesCtrCounterExtendedGeneration());
            }
            continue;
        }

        // IsMatched の場合、対応する Previous の各エントリーを引っ張ることができ、generation は一致する
        uint32_t previousGeneration = 0;
        {
            int64_t previousBegin = std::numeric_limits<int64_t>::max();
            int64_t previousEnd = std::numeric_limits<int64_t>::min();
            bool isSetPreviousGeneration = false;

            auto previousIter = std::lower_bound(
                previousEntries.begin(),
                previousEntries.end(),
                currentBegin,
                [=](const AesCtrCounterExtendedEntry& entry, int64_t begin) NN_NOEXCEPT
                {
                    return (entry.GetOffset() + entry.GetSize()) + previousOffset <= begin;
                }
            );
            for( ; previousIter != previousEntries.end(); ++previousIter )
            {
                const auto& previousEntry = *previousIter;

                const auto begin = previousEntry.GetOffset() + previousOffset;
                const auto end = begin + previousEntry.GetSize();

                // オーバーラップを求める
                const auto overlapBegin = std::max(currentBegin, begin);
                const auto overlapEnd = std::min(currentEnd, end);
                const auto overlapSize = overlapEnd - overlapBegin;

                if( overlapSize <= 0 )
                {
                    break;
                }

                previousBegin = std::min(overlapBegin, previousBegin);
                previousEnd = std::max(overlapEnd, previousEnd);

                if( !isSetPreviousGeneration )
                {
                    isSetPreviousGeneration = true;
                    previousGeneration = previousEntry.GetGeneration();
                }
                else
                {
                    // 全て同じ generation のはず
                    NN_RESULT_THROW_UNLESS(previousGeneration == previousEntry.GetGeneration(),
                                           fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
                }
            }

            NN_RESULT_THROW_UNLESS(previousBegin == currentBegin,
                                   fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
            NN_RESULT_THROW_UNLESS(previousEnd == currentEnd,
                                   fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
            NN_RESULT_THROW_UNLESS(isSetPreviousGeneration,
                                   fs::ResultInvalidAesCtrCounterExtendedEntryOffset());
        }

        // previous の generation と一致
        NN_RESULT_THROW_UNLESS(currentEntry.GetGeneration() == previousGeneration,
                               fs::ResultInvalidAesCtrCounterExtendedGeneration());
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}}}
