﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_BucketTree.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/utilTool/fs_BucketTreeAnalyzer.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilderImpl.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace fssystem { namespace utilTool {

namespace {

typedef BinaryMatch Match;
typedef Match::Result MatchResult;

}

NN_DEFINE_STATIC_CONSTANT(const int IndirectStorageBuilder::HashBitCount);

/**
 * @brief   コンストラクタです。
 */
IndirectStorageBuilder::IndirectStorageBuilder() NN_NOEXCEPT
    : m_pOldStorage(nullptr)
    , m_pNewStorage(nullptr)
    , m_OldStorageSize(0)
    , m_NewStorageSize(0)
    , m_DataStorageSize(0)
    , m_ExcludeRangeOldArray(nullptr)
    , m_ExcludeRangeOldCount(0)
    , m_ExcludeRangeNewArray(nullptr)
    , m_ExcludeRangeNewCount(0)
    , m_BinaryMatchHint()
    , m_Match()
    , m_MatchCount(0)
    , m_pProgress()
    , m_SortedMatches()
{
}

/**
 * @brief   デストラクタです。
 */
IndirectStorageBuilder::~IndirectStorageBuilder() NN_NOEXCEPT
{
    Finalize();
}

/**
 * @brief   初期化をします。
 *
 * @param[in]   pOldStorage     古いデータを指すストレージのポインタ
 * @param[in]   pNewStorage     新しいデータを指すストレージのポインタ
 * @param[in]   blockSize       ハッシュ化するデータのサイズ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pOldStorage != nullptr
 *      - pNewStorage != nullptr
 *      - pOldStorage != pNewStorage
 *      - 8 <= blockSize
 *      - ispow2(blockSize) != false
 *      - IsInitialize() == false
 */
Result IndirectStorageBuilder::Initialize(
         fs::IStorage* pOldStorage,
         fs::IStorage* pNewStorage,
         size_t blockSize
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOldStorage);
    NN_SDK_REQUIRES_NOT_NULL(pNewStorage);
    NN_SDK_REQUIRES_NOT_EQUAL(pOldStorage, pNewStorage);
    NN_SDK_REQUIRES(sizeof(int64_t) <= blockSize && util::ispow2(blockSize));
    NN_SDK_REQUIRES(!IsInitialized());

    NN_RESULT_DO(pOldStorage->GetSize(&m_OldStorageSize));
    NN_RESULT_DO(pNewStorage->GetSize(&m_NewStorageSize));

    if( !util::is_aligned(m_OldStorageSize, blockSize) ||
        !util::is_aligned(m_NewStorageSize, blockSize) )
    {
        return fs::ResultInvalidSize();
    }

    m_pOldStorage = pOldStorage;
    m_pNewStorage = pNewStorage;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   終了処理をします。
 */
void IndirectStorageBuilder::Finalize() NN_NOEXCEPT
{
    if( m_pOldStorage != nullptr )
    {
        m_pOldStorage = nullptr;
        m_pNewStorage = nullptr;
        m_DataStorageSize = 0;
        m_Match.reset();
        m_MatchCount = 0;
        m_SortedMatches.clear();
    }
    m_ExcludeRangeOldArray = nullptr;
    m_ExcludeRangeOldCount = 0;
    m_ExcludeRangeNewArray = nullptr;
    m_ExcludeRangeNewCount = 0;
    m_BinaryMatchHint = decltype(m_BinaryMatchHint)();
    m_BinaryRegion = decltype(m_BinaryRegion)();
    m_pProgress.reset();
}

/**
 * @brief   ストレージを解析してテーブルを作成します。
 *
 * @param[in]   shiftSize   ハッシュ化するデータの移動量
 * @param[in]   blockSize   ハッシュ化するデータのサイズ
 * @param[in]   regionSize  比較するデータの最小単位
 * @param[in]   matchSize   マッチするデータの最小単位
 * @param[in]   windowSize  比較するデータ領域の最大幅
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - 0 < shiftSize
 *      - shiftSize <= blockSize
 *      - ispow2(shiftSize) != false
 *      - 8 <= blockSize
 *      - ispow2(blockSize) != false
 *      - blockSize < regionSize
 *      - ispow2(regionSize) != false
 *      - regionSize < windowSize
 *      - IsInitialized() != false
 *      - IsBuilt() == false
 */
Result IndirectStorageBuilder::Build(
                                   size_t shiftSize,
                                   size_t blockSize,
                                   size_t regionSize,
                                   int64_t matchSize,
                                   int64_t windowSize
                               ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sizeof(int64_t) <= blockSize && util::ispow2(blockSize));
    NN_SDK_REQUIRES(0 < shiftSize && shiftSize <= blockSize && util::ispow2(shiftSize));
    NN_SDK_REQUIRES(blockSize < regionSize && util::ispow2(regionSize));
    NN_SDK_REQUIRES(static_cast<int64_t>(regionSize) < windowSize);
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());

    // マッチング
    std::vector<MatchResult> results;
    {
        Match match(blockSize, regionSize, matchSize);
        match.SetWindowSize(windowSize);
        match.SetHint(std::move(m_BinaryMatchHint));
        match.SetRegion(m_BinaryRegion.first);

        // 比較しない範囲の追加
        for( int i = 0; i < m_ExcludeRangeOldCount; ++i )
        {
            NN_RESULT_DO(match.AddExcludeRangeForOldStorage(m_ExcludeRangeOldArray[i]));
        }
        for( int i = 0; i < m_ExcludeRangeNewCount; ++i )
        {
            NN_RESULT_DO(match.AddExcludeRangeForNewStorage(m_ExcludeRangeNewArray[i]));
        }

        m_pProgress = match.GetProgress();

        NN_RESULT_DO(
            match.Run(
                fs::SubStorage(m_pOldStorage, 0, m_OldStorageSize),
                fs::SubStorage(m_pNewStorage, 0, m_NewStorageSize),
                shiftSize
            )
        );

        results = match.GetResult();

        NN_RESULT_THROW_UNLESS(!results.empty(), fs::ResultNotFound());

        m_BinaryRegion = match.GetRegionInfo();
    }

    m_Match.reset(new MatchEntry[results.size()]);
    NN_RESULT_THROW_UNLESS(
        m_Match != nullptr, fs::ResultAllocationMemoryFailedInIndirectStorageBuilderB());

    auto match1 = results[0];
    auto entry = match1;
    int entryCount = 0;
    int64_t entrySize = match1.size;

    for( size_t i = 1, count = results.size(); i < count; ++i )
    {
        const auto& match2 = results[i];

        NN_SDK_ASSERT_LESS(match1.newOffset, match2.newOffset);
        NN_SDK_ASSERT_EQUAL(match1.newOffset + match1.size, match2.newOffset);

        // 同じストレージで連続している
        if( match1.CanMerge(match2) )
        {
            // 何もしない
        }
        else
        {
            if( entrySize < matchSize && entry.storageIndex == 0 )
            {
                entry.oldOffset = MatchResult::InvalidOffset;
                entry.storageIndex = 1;
            }
            entryCount += SetEntry(entryCount, entry, entrySize);

            entry = match2;
            entrySize = 0;
        }
        entrySize += match2.size;

        match1 = match2;
    }

    NN_SDK_ASSERT_EQUAL(m_NewStorageSize - entry.newOffset, entrySize);

    if( entrySize < matchSize )
    {
        entry.oldOffset = MatchResult::InvalidOffset;
        entry.storageIndex = 1;
    }
    entryCount += SetEntry(entryCount, entry, entrySize);

    m_MatchCount = entryCount;

    NN_RESULT_DO(SetSortedMatches(false));

    NN_RESULT_SUCCESS;
}

/**
 * @brief   生成済みのテーブルを注入します。// オーサリングツール向け
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   headerStorage   テーブルヘッダのストレージ
 * @param[in]   nodeStorage     テーブルノードのストレージ
 * @param[in]   entryStorage    テーブルエントリのストレージ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - headerStorage, nodeStorage, entryStorage に正しい情報が書き込まれている
 *      - IsInitialized() != false
 *      - IsBuilt() == false
 */
Result IndirectStorageBuilder::Build(
                                   IAllocator* pAllocator,
                                   fs::SubStorage headerStorage,
                                   fs::SubStorage nodeStorage,
                                   fs::SubStorage entryStorage
                               ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());

    BucketTree::Header header;
    NN_RESULT_DO(headerStorage.Read(0, &header, sizeof(header)));
    NN_RESULT_DO(header.Verify());

    BucketTree table;
    NN_RESULT_DO(
        table.Initialize(
            pAllocator,
            nodeStorage,
            entryStorage,
            IndirectStorage::NodeSize,
            sizeof(IndirectStorage::Entry),
            header.entryCount
        )
    );

    std::unique_ptr<MatchEntry[]> matches(new MatchEntry[header.entryCount + 1]);

    NN_RESULT_THROW_UNLESS(
        matches != nullptr, fs::ResultAllocationMemoryFailedInIndirectStorageBuilderD());

    BucketTree::Visitor visitor;
    NN_RESULT_DO(table.Find(&visitor, 0));

    int index = 0;
    do
    {
        auto& data = matches[index];
        {
            auto entry = visitor.Get<IndirectStorage::Entry>();
            data.virtualOffset = entry->GetVirtualOffset();
            data.physicalOffset = entry->GetPhysicalOffset();
            data.storageIndex = entry->storageIndex;
        }
        if( !visitor.CanMoveNext() )
        {
            break;
        }
        NN_RESULT_DO(visitor.MoveNext());
        index++;
    } while( NN_STATIC_CONDITION(true) );

    m_MatchCount = header.entryCount;
    m_Match = std::move(matches);

    NN_RESULT_DO(SetSortedMatches(true));

    NN_RESULT_SUCCESS;
}

/**
 * @brief   エントリを設定します。
 *
 * @param[in]   entryIndex  エントリのインデックス
 * @param[in]   entry       設定するエントリ情報
 * @param[in]   entrySize   エントリのサイズ
 *
 * @return      追加したエントリの個数
 */
int IndirectStorageBuilder::SetEntry(
                                int entryIndex,
                                const MatchResult& entry,
                                int64_t entrySize
                            ) NN_NOEXCEPT
{
    auto& data = m_Match[entryIndex];

    if( entry.storageIndex == 0 )
    {
        // 古いデータを使う
        data.virtualOffset = entry.newOffset;
        data.physicalOffset = entry.oldOffset;
        data.storageIndex = entry.storageIndex;
        return 1;
    }
    else
    {
        // 新しいデータを使う

        // 直前と連続する新しいデータ以外の場合にエントリを追加する
        auto entryCount = 0;
        if( entry.oldOffset != MatchResult::InvalidOffset
            || entryIndex == 0
            || m_Match[entryIndex - 1].storageIndex == 0
            || m_DataStorageSize - m_Match[entryIndex - 1].physicalOffset
                != entry.newOffset - m_Match[entryIndex - 1].virtualOffset )
        {
            data.virtualOffset = entry.newOffset;
            data.storageIndex = entry.storageIndex;
            if( entry.oldOffset == MatchResult::InvalidOffset )
            {
                // データは新しく追加する
                data.physicalOffset = m_DataStorageSize;
            }
            else if( entry.oldOffset < entry.newOffset )
            {
                // 既に追加されたデータから探索する
                data.physicalOffset = MatchResult::InvalidOffset;
            }
            else
            {
                // データはまだ追加せず、追加したときにこのオフセットを書き直す
                // そのための目印として参照先の仮想オフセットの正負を反転したものを仮に格納しておく
                data.physicalOffset = -entry.oldOffset;
            }
            entryCount = 1;
        }

        // データを追加する
        if( entry.oldOffset == MatchResult::InvalidOffset )
        {
            m_DataStorageSize += entrySize;
        }

        // セルフマッチの物理オフセットを解決する
        if( entry.oldOffset == MatchResult::InvalidOffset )
        {
            // 他のセルフマッチした領域がこの領域を指している場合、既に追加されたエントリのオフセットを書き直す
            const auto& matchData = m_Match[entryIndex + entryCount - 1];
            const auto matchSize = m_DataStorageSize - matchData.physicalOffset;
            for( auto previousIndex = 0; previousIndex < entryIndex + entryCount - 1; ++previousIndex )
            {
                if( m_Match[previousIndex].physicalOffset < MatchResult::InvalidOffset
                    && matchData.virtualOffset <= -m_Match[previousIndex].physicalOffset
                    && -m_Match[previousIndex].physicalOffset < matchData.virtualOffset + matchSize )
                {
                    m_Match[previousIndex].physicalOffset = matchData.physicalOffset
                        + (-m_Match[previousIndex].physicalOffset - matchData.virtualOffset);
                }
            }
        }
        else if( entry.oldOffset < entry.newOffset )
        {
            // セルフマッチの参照先が既に追加されているなら、既に追加されている領域のオフセットを探索する
            for( auto previousIndex = 0; previousIndex < entryIndex + entryCount - 1; ++previousIndex )
            {
                if( m_Match[previousIndex].virtualOffset <= entry.oldOffset
                    && entry.oldOffset < m_Match[previousIndex + 1].virtualOffset )
                {
                    NN_SDK_ASSERT_EQUAL(m_Match[previousIndex].storageIndex, 1);
                    data.physicalOffset
                        = m_Match[previousIndex].physicalOffset
                        + entry.oldOffset - m_Match[previousIndex].virtualOffset;
                    break;
                }
            }
            NN_SDK_ASSERT_NOT_EQUAL(data.physicalOffset, MatchResult::InvalidOffset);
        }

        return entryCount;
    }
}

Result IndirectStorageBuilder::SetSortedMatches(bool isStorageSizeUpdateRequired) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsBuilt());
    NN_SDK_REQUIRES(!m_SortedMatches.IsValid());

    // m_Match のうち storageIndex == 1 のエントリをサイズ付きでコピーし physicalOffset でソート
    m_SortedMatches = BucketTreeRangeArray(m_MatchCount);
    NN_RESULT_THROW_UNLESS(m_SortedMatches.IsValid(), fs::ResultAllocationMemoryFailedNew());

    if( isStorageSizeUpdateRequired )
    {
        m_DataStorageSize = 0;
    }

    for( int i = 0; i < m_MatchCount; i++ )
    {
        auto entry = m_Match[i];
        if( entry.storageIndex == 1 )
        {
            const auto endOffset = (i == m_MatchCount - 1) ? m_NewStorageSize : m_Match[i + 1].virtualOffset;

            m_SortedMatches.emplace_back(entry.virtualOffset, entry.physicalOffset, endOffset - entry.virtualOffset);
            NN_SDK_ASSERT_LESS(0, m_SortedMatches.back().size);

            if( isStorageSizeUpdateRequired )
            {
                m_DataStorageSize = std::max(m_DataStorageSize, entry.physicalOffset + m_SortedMatches.back().size);
            }
        }
    }

    std::sort(
        m_SortedMatches.begin(),
        m_SortedMatches.end(),
        [](const MatchEntryWithExtentSize& x, const MatchEntryWithExtentSize& y) NN_NOEXCEPT
        {
            return x.physicalOffset < y.physicalOffset;
        }
    );

    NN_RESULT_SUCCESS;
}

/**
 * @brief   テーブルを書き出します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   headerStorage   テーブルヘッダのストレージ
 * @param[in]   nodeStorage     テーブルノードのストレージ
 * @param[in]   entryStorage    テーブルエントリのストレージ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - IsBuilt() != false
 */
Result IndirectStorageBuilder::WriteTable(
                                   IAllocator* pAllocator,
                                   fs::SubStorage headerStorage,
                                   fs::SubStorage nodeStorage,
                                   fs::SubStorage entryStorage
                               ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsBuilt());

    if( m_MatchCount == 0 )
    {
        return fs::ResultNotFound();
    }

    BucketTreeBuilder builder;
    NN_RESULT_DO(
        builder.Initialize(
            pAllocator,
            headerStorage,
            nodeStorage,
            entryStorage,
            IndirectStorage::NodeSize,
            sizeof(IndirectStorage::Entry),
            m_MatchCount
        )
    );

    for( int i = 0; i < m_MatchCount; ++i )
    {
        const auto& data = m_Match[i];

        IndirectStorage::Entry entry;
        entry.SetVirtualOffset(data.virtualOffset);
        entry.SetPhysicalOffset(data.physicalOffset);
        entry.storageIndex = data.storageIndex;

        NN_RESULT_DO(builder.Write(entry));
    }

    return builder.Commit(m_NewStorageSize);
}

/**
 * @brief   テーブルを元に必要なデータを書き出します。
 *
 * @param[in]   pDataStorage    データを書き出すストレージのポインタ
 * @param[in]   buffer          データの読み書きに使用するバッファの先頭アドレス
 * @param[in]   bufferSize      データの読み書きに使用するバッファのサイズ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pDataStorage != nullptr
 *      - buffer != nullptr
 *      - 0 < bufferSize
 *      - IsBuilt() != false
 */
Result IndirectStorageBuilder::WriteData(
                                   fs::IStorage* pDataStorage,
                                   void* buffer,
                                   size_t bufferSize
                               ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDataStorage);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_LESS(static_cast<size_t>(0), bufferSize);
    NN_SDK_REQUIRES(IsBuilt());

    if( m_MatchCount == 0 )
    {
        return fs::ResultNotFound();
    }

    class CopyStorage
    {
    public:
        CopyStorage(
            fs::IStorage* pSrcStorage,
            fs::IStorage* pDstStorage,
            void* buffer,
            size_t bufferSize
        ) NN_NOEXCEPT
            : m_pSrcStorage(pSrcStorage)
            , m_pDstStorage(pDstStorage)
            , m_Buffer(buffer)
            , m_BufferSize(bufferSize)
        {
        }

        Result Copy(int64_t srcOffset, int64_t dstOffset, int64_t size) NN_NOEXCEPT
        {
            auto writeOffset = dstOffset;
            auto readOffset = srcOffset;
            auto restSize = size;

            while( 0 < restSize )
            {
                // NOTE: m_BufferSize <= size_t::max() なのでキャストok
                auto readSize = static_cast<size_t>(std::min(restSize, m_BufferSize));

                NN_RESULT_DO(m_pSrcStorage->Read(readOffset, m_Buffer, readSize));
                NN_RESULT_DO(m_pDstStorage->Write(writeOffset, m_Buffer, readSize));

                writeOffset += readSize;
                readOffset += readSize;
                restSize -= readSize;
            }

            NN_RESULT_SUCCESS;
        }

    private:
        fs::IStorage* const m_pSrcStorage;
        fs::IStorage* const m_pDstStorage;
        void* const m_Buffer;
        const int64_t m_BufferSize;
    }
    storage(m_pNewStorage, pDataStorage, buffer, bufferSize);

    auto entry1 = m_Match[0];

    // m_Match[0, m_MatchCount - 1) までのデータを出力
    for( int i = 1; i < m_MatchCount; ++i )
    {
        const auto& entry2 = m_Match[i];

        if( entry1.storageIndex == 1 )
        {
            const auto size = entry2.virtualOffset - entry1.virtualOffset;
            NN_SDK_ASSERT_LESS(0, size);

            NN_RESULT_DO(storage.Copy(entry1.virtualOffset, entry1.physicalOffset, size));
        }

        entry1 = entry2;
    }

    // m_Match[m_MatchCount - 1] のデータを出力
    if( entry1.storageIndex == 1 )
    {
        const auto size = m_NewStorageSize - entry1.virtualOffset;
        NN_SDK_ASSERT_LESS(0, size);

        NN_RESULT_DO(storage.Copy(entry1.virtualOffset, entry1.physicalOffset, size));
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   IndirectStorage のビルド情報を出力します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
 * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
 * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
 * @param[in]   pDataStorage    データ情報を書き出すストレージのポインタ
 * @param[in]   startOffset     IndirectStorage の開始オフセット
 *
 * @pre
 *      - pAllocator != nullptr
 *      - pTableStorage != nullptr
 *      - pNodeStorage != nullptr
 *      - pEntryStorage != nullptr
 *      - pDataStorage != nullptr
 *      - 0 <= startOffset
 */
Result IndirectStorageBuilder::OutputBuildLog(
                                   IAllocator* pAllocator,
                                   fs::IStorage* pTableFileStorage,
                                   fs::IStorage* pNodeFileStorage,
                                   fs::IStorage* pEntryFileStorage,
                                   fs::IStorage* pDataFileStorage,
                                   int64_t startOffset
                               ) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pAllocator != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pTableFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pNodeFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pEntryFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pDataFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(0 <= startOffset, fs::ResultInvalidOffset());

    return OutputBucketTreeBuildLog(
        pTableFileStorage,
        pNodeFileStorage,
        pEntryFileStorage,
        pDataFileStorage,
        m_MatchCount,
        startOffset,
        startOffset + m_NewStorageSize,
        // BucketTreeBuilder からの書き出しをハックする
        [=](const BucketTreeAnalyzer& analyzer) NN_NOEXCEPT -> Result
        {
            char headerBuffer[sizeof(BucketTree::Header)];
            NN_SDK_ASSERT_EQUAL(
                QueryTableHeaderStorageSize(), static_cast<int64_t>(sizeof(headerBuffer)));
            fs::MemoryStorage headerStorage(headerBuffer, sizeof(headerBuffer));

            BucketTreeBuilder builder;
            NN_RESULT_DO(
                builder.Initialize(
                    pAllocator,
                    fs::SubStorage(&headerStorage, 0, sizeof(headerBuffer)),
                    fs::SubStorage(analyzer.GetNodeStorage(), 0, QueryTableNodeStorageSize()),
                    fs::SubStorage(analyzer.GetEntryStorage(), 0, QueryTableEntryStorageSize()),
                    IndirectStorage::NodeSize,
                    sizeof(IndirectStorage::Entry),
                    m_MatchCount
                )
            );

            for( int i = 0; i < m_MatchCount; ++i )
            {
                const auto& data = m_Match[i];

                IndirectStorage::Entry entry;
                entry.SetVirtualOffset(data.virtualOffset);
                entry.SetPhysicalOffset(data.physicalOffset);
                entry.storageIndex = data.storageIndex;

                NN_RESULT_DO(builder.Write(entry));
            }

            return builder.Commit(m_NewStorageSize);
        }
    );
}

/**
 * @brief   IndirectStorage のビルド情報を出力します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
 * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
 * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
 * @param[in]   pDataStorage    データ情報を書き出すストレージのポインタ
 * @param[in]   indirectStorage 書き出し済みの IndirectStorage のテーブル情報
 * @param[in]   entryCount      IndirectStorage のエントリ数
 * @param[in]   startOffset     IndirectStorage の開始オフセット
 * @param[in]   endOffset       IndirectStorage の終了オフセット
 *
 * @pre
 *      - pAllocator != nullptr
 *      - pTableStorage != nullptr
 *      - pNodeStorage != nullptr
 *      - pEntryStorage != nullptr
 *      - pDataStorage != nullptr
 *      - 0 < entryCount
 *      - 0 <= startOffset
 *      - startOffset < endOffset
 */
Result IndirectStorageBuilder::OutputBuildLog(
                                   IAllocator* pAllocator,
                                   fs::IStorage* pTableFileStorage,
                                   fs::IStorage* pNodeFileStorage,
                                   fs::IStorage* pEntryFileStorage,
                                   fs::IStorage* pDataFileStorage,
                                   fs::SubStorage indirectStorage,
                                   int entryCount,
                                   int64_t startOffset,
                                   int64_t endOffset
                               ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pAllocator != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pTableFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pNodeFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pEntryFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pDataFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(0 < entryCount, fs::ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(0 <= startOffset, fs::ResultInvalidOffset());
    NN_RESULT_THROW_UNLESS(startOffset < endOffset, fs::ResultInvalidOffset());

    return OutputBucketTreeBuildLog(
        pTableFileStorage,
        pNodeFileStorage,
        pEntryFileStorage,
        pDataFileStorage,
        entryCount,
        startOffset,
        endOffset,
        // 書き出し済みのストレージから Read/Write する
        [=](const BucketTreeAnalyzer& analyzer) NN_NOEXCEPT -> Result
        {
            static const size_t BufferSize = IndirectStorage::NodeSize;

            void* buffer = pAllocator->allocate(BufferSize);
            NN_RESULT_THROW_UNLESS(
                buffer != nullptr, fs::ResultAllocationMemoryFailedInIndirectStorageBuilderC());
            NN_UTIL_SCOPE_EXIT
            {
                pAllocator->deallocate(buffer, BufferSize);
            };

            fs::SubStorage storage(indirectStorage);

            const auto nodeSize = IndirectStorage::QueryNodeStorageSize(entryCount);
            {
                fs::IStorage* pNodeStorage = analyzer.GetNodeStorage();

                for( int64_t offset = 0; offset < nodeSize; offset += BufferSize )
                {
                    NN_RESULT_DO(storage.Read(offset, buffer, BufferSize));
                    NN_RESULT_DO(pNodeStorage->Write(offset, buffer, BufferSize));
                }
            }

            const auto entrySize = IndirectStorage::QueryEntryStorageSize(entryCount);
            {
                fs::IStorage* pEntryStorage = analyzer.GetEntryStorage();

                for( int64_t offset = 0; offset < entrySize; offset += BufferSize )
                {
                    NN_RESULT_DO(storage.Read(nodeSize + offset, buffer, BufferSize));
                    NN_RESULT_DO(pEntryStorage->Write(offset, buffer, BufferSize));
                }
            }

            NN_RESULT_SUCCESS;
        }
    );
}

}}}
