﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <memory>
#include <algorithm>
#include <random>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_BinTypes.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fssystem/save/fs_MappingTable.h>
#include <nn/fssystem/save/fs_JournalStorage.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    //!< 初期化済みコード 'JNGL'
    static const uint32_t MagicCode = NN_UTIL_CREATE_SIGNATURE_4('J','N','G','L');

    //!< バージョン
    static const uint32_t VersionCode = 0x00010000;
}

/**
* @brief        マッピングテーブルのメタデータサイズを取得します。
*
* @param[out]   outTableSize                    マッピングテーブルに必要なサイズ
* @param[out]   outBitmapSizeUpdatedPhysical    更新済みビットマップ(物理インデックス)に必要なサイズ
* @param[out]   outBitmapSizeUpdatedVirtual     更新済みビットマップ(論理インデックス)に必要なサイズ
* @param[out]   outBitmapSizeUnassigned         未割当ビットマップ(物理インデックス)に必要なサイズ
* @param[in]    sizeArea                        データ領域の総サイズ
* @param[in]    sizeReservedArea                予約領域のサイズ
* @param[in]    sizeBlock                       ブロックサイズ
*
* @return       マッピングテーブルのメタデータサイズを返します。
*/
void JournalStorage::QueryMappingMetaSize(
         int64_t* outTableSize,
         int64_t* outBitmapSizeUpdatedPhysical,
         int64_t* outBitmapSizeUpdatedVirtual,
         int64_t* outBitmapSizeUnassigned,
         int64_t sizeArea,
         int64_t sizeReservedArea,
         int64_t sizeBlock
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTableSize);
    NN_SDK_REQUIRES_NOT_NULL(outBitmapSizeUpdatedPhysical);
    NN_SDK_REQUIRES_NOT_NULL(outBitmapSizeUpdatedVirtual);
    NN_SDK_REQUIRES_NOT_NULL(outBitmapSizeUnassigned);
    NN_SDK_REQUIRES((sizeArea % sizeBlock) == 0);

    const MappingTable::Index countMapEntry = MappingTable::MakeIndex((sizeArea - sizeReservedArea) / sizeBlock);
    const MappingTable::Index countReserved = MappingTable::MakeIndex(sizeReservedArea / sizeBlock);
    MappingTable::QueryMappingMetaSize(
        outTableSize,
        outBitmapSizeUpdatedPhysical,
        outBitmapSizeUpdatedVirtual,
        outBitmapSizeUnassigned,
        countMapEntry,
        countReserved
    );
}

/**
* @brief        マッピングテーブルメタデータ領域をフォーマットします。
*
* @param[in]    storageControlArea              管理領域を配置するレイヤー
* @param[in]    storageTable                    マッピングテーブルのメタデータを配置するレイヤー
* @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    sizeArea                        データ領域の総サイズ
* @param[in]    sizeReservedArea                予約領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::Format(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageTable,
           fs::SubStorage storageBitmapUpdatedPhysical,
           fs::SubStorage storageBitmapUpdatedVirtual,
           fs::SubStorage storageBitmapUnassigned,
           int64_t sizeArea,
           int64_t sizeReservedArea,
           int64_t sizeBlock
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sizeBlock > 0);
    NN_SDK_REQUIRES((sizeArea % sizeBlock) == 0);
    NN_SDK_REQUIRES((sizeReservedArea % sizeBlock) == 0);

    const MappingTable::Index countMapEntry = MappingTable::MakeIndex((sizeArea - sizeReservedArea) / sizeBlock);
    const MappingTable::Index countReserved = MappingTable::MakeIndex(sizeReservedArea / sizeBlock);

    // 管理領域を初期化します。
    ControlArea header;
    std::memset(&header, 0, sizeof(header));
    header.signature = MagicCode;
    header.version = VersionCode;
    header.areaSize = sizeArea;
    header.reservedAreaSize = sizeReservedArea;
    header.sizeBlock = sizeBlock;
    NN_RESULT_DO(storageControlArea.Write(0, &header, sizeof(header)));

    fs::SubStorage storageMappingControlArea(
                   &storageControlArea,
                   offsetof(ControlArea, mappingInfo),
                   sizeof(MappingTable::ControlArea)
               );
    return MappingTable::Format(
               storageMappingControlArea,
               storageTable,
               storageBitmapUpdatedPhysical,
               storageBitmapUpdatedVirtual,
               storageBitmapUnassigned,
               countMapEntry,
               countReserved
           );
}

/**
* @brief        マッピングテーブルメタデータ領域を拡張します。
*
* @param[in]    storageMeta                     マッピングテーブルのメタデータを配置するレイヤー
* @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    sizeAreaNew                     拡張後のデータ領域の総サイズ
* @param[in]    sizeReservedAreaNew             拡張後の予約領域のサイズ
*
* @return       関数の処理結果を返します。
*
* @details      ストレージが管理する領域を拡張します。
*               マウントしていない状態で実行してください。
*/
Result JournalStorage::Expand(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageTable,
           fs::SubStorage storageBitmapUpdatedPhysical,
           fs::SubStorage storageBitmapUpdatedVirtual,
           fs::SubStorage storageBitmapUnassigned,
           int64_t sizeAreaNew,
           int64_t sizeReservedAreaNew
       ) NN_NOEXCEPT
{
    // 管理領域を読み込みします。
    ControlArea header;
    NN_RESULT_DO(storageControlArea.Read(0, &header, sizeof(header)));

    // 管理領域の内容を検証します。
    if( header.version > VersionCode )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    const MappingTable::Index countMapEntryNew = MappingTable::MakeIndex((sizeAreaNew - sizeReservedAreaNew) / header.sizeBlock);
    const MappingTable::Index countReservedNew = MappingTable::MakeIndex(sizeReservedAreaNew / header.sizeBlock);

    NN_SDK_REQUIRES((sizeAreaNew % header.sizeBlock) == 0);
    NN_SDK_REQUIRES((sizeReservedAreaNew % header.sizeBlock) == 0);

    // 管理領域を更新します。
    header.areaSize = sizeAreaNew;
    header.reservedAreaSize = sizeReservedAreaNew;
    NN_RESULT_DO(storageControlArea.Write(0, &header, sizeof(header)));

    fs::SubStorage storageMappingControlArea(
                   &storageControlArea,
                   offsetof(ControlArea, mappingInfo),
                   sizeof(MappingTable::ControlArea)
               );
    return MappingTable::Expand(
               storageMappingControlArea,
               storageTable,
               storageBitmapUpdatedPhysical,
               storageBitmapUpdatedVirtual,
               storageBitmapUnassigned,
               countMapEntryNew,
               countReservedNew
           );
}

//! コンストラクタ
JournalStorage::JournalStorage() NN_NOEXCEPT
    : m_IsMappingTableFrozon(false)
{
}

//! デストラクタ
JournalStorage::~JournalStorage() NN_NOEXCEPT
{
}

/**
* @brief        マッピングテーブルレイヤーとしてマウントします。
*
* @param[in]    storageControlArea              管理領域を配置するレイヤー
* @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
* @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageData                     実データ領域を配置するレイヤー
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::Initialize(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageTable,
           fs::SubStorage storageBitmapUpdatedPhysical,
           fs::SubStorage storageBitmapUpdatedVirtual,
           fs::SubStorage storageBitmapUnassigned,
           fs::SubStorage storageData
       ) NN_NOEXCEPT
{
    m_StorageData = storageData;

    // 管理領域を読み込みます。
    NN_RESULT_DO(storageControlArea.Read(0, &m_ControlArea, sizeof(m_ControlArea)));

    // 管理領域の内容を検証します。
    if( m_ControlArea.version > VersionCode )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    // 論物変換テーブルをマウントします。
    fs::SubStorage storageMappingControlArea(
                   &storageControlArea,
                   offsetof(ControlArea, mappingInfo),
                   sizeof(MappingTable::ControlArea)
               );
    NN_RESULT_DO(
        m_MappingTable.Initialize(
            storageMappingControlArea,
            storageTable,
            storageBitmapUpdatedPhysical,
            storageBitmapUpdatedVirtual,
            storageBitmapUnassigned
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        コミット操作を実行します。
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::Commit() NN_NOEXCEPT
{
    return m_MappingTable.Commit();
}

/**
* @brief        マッピングテーブル上のインデックスを物理アドレスに変換します。
*
* @param[out]   outValue    物理アドレス
* @param[in]    index       マッピングテーブル上の論理/物理インデックス
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::GetPhysicalAddress(
           int64_t* outValue,
           MappingTable::Index index
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    MappingTable::Index indexPhysical;
    NN_RESULT_DO(m_MappingTable.GetPhysicalIndex(&indexPhysical, index));
    *outValue = indexPhysical * GetBlockSize();

    NN_RESULT_SUCCESS;
}

/**
* @brief        データ書き込み前にマッピングテーブルを更新します。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::MarkUpdate(int64_t offset, int64_t size) NN_NOEXCEPT
{
    const int64_t addressAligned = AlignAddress(offset);
    const int64_t sizeAligned = AlignUpAddress(offset + size) - addressAligned;

    // マッピングテーブルに書き込みを通知します。
    bool needCopyHead;
    MappingTable::Index indexHeadSrc;
    MappingTable::Index indexHeadDst;
    bool needCopyTail;
    MappingTable::Index indexTailSrc;
    MappingTable::Index indexTailDst;
    NN_RESULT_DO(
        m_MappingTable.MarkUpdate(
            &needCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &needCopyTail,
            &indexTailSrc,
            &indexTailDst,
            MappingTable::MakeIndex(addressAligned / GetBlockSize()),
            MappingTable::MakeIndex(sizeAligned / GetBlockSize())
        )
    );

    // 書き込みが発生する領域の先頭、末尾データかつ
    // 非アライメントな書き込みが発生する場合のみ
    // 変更前の領域から書き換え先領域にコピーします。

    const bool alignHead = (offset == addressAligned) && (size >= GetBlockSize());
    const bool alignTail = ((offset + size) == (addressAligned + sizeAligned));

    // ブロックサイズは 32 ビットに収まります。
    NN_SDK_ASSERT(nn::util::IsIntValueRepresentable<uint32_t>(GetBlockSize()));
    const size_t blockSize = static_cast<size_t>(GetBlockSize());

    if( needCopyHead && !alignHead )
    {
        std::unique_ptr<char[], nn::fs::detail::Deleter> buf
            = nn::fs::detail::MakeUnique<char[]>(blockSize);
        NN_RESULT_THROW_UNLESS(buf != nullptr, nn::fs::ResultAllocationMemoryFailedInJournalStorageA());
        int64_t addressSrc;
        NN_RESULT_DO(GetPhysicalAddress(&addressSrc, indexHeadSrc));
        NN_RESULT_DO(m_StorageData.Read(addressSrc, buf.get(), blockSize));
        int64_t addressDst;
        NN_RESULT_DO(GetPhysicalAddress(&addressDst, indexHeadDst));
        NN_RESULT_DO(m_StorageData.Write(addressDst, buf.get(), blockSize));
    }
    if( needCopyTail && !alignTail )
    {
        std::unique_ptr<char[], nn::fs::detail::Deleter> buf
            = nn::fs::detail::MakeUnique<char[]>(blockSize);
        NN_RESULT_THROW_UNLESS(buf != nullptr, nn::fs::ResultAllocationMemoryFailedInJournalStorageB());
        int64_t addressSrc;
        NN_RESULT_DO(GetPhysicalAddress(&addressSrc, indexTailSrc));
        NN_RESULT_DO(m_StorageData.Read(addressSrc, buf.get(), blockSize));
        int64_t addressDst;
        NN_RESULT_DO(GetPhysicalAddress(&addressDst, indexTailDst));
        NN_RESULT_DO(m_StorageData.Write(addressDst, buf.get(), blockSize));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブル経由でファイルにデータを書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
#if defined(USE_RANGE_LOCK)
    // ライターロックを取得します
    ScopedWriterLock locker(&m_Lock, offset, size);
#endif // defined(USE_RANGE_LOCK)

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(nn::fs::IStorage::CheckOffsetAndSize(offset, size), nn::fs::ResultInvalidOffset());

    if( !m_IsMappingTableFrozon )
    {
        // 書き込み前にマッピングテーブルを更新します。
        NN_RESULT_DO(MarkUpdate(offset, size));
    }

    const char* p = static_cast<const char*>(buffer);
    NN_RESULT_DO(IterateMappingTable(
        offset,
        static_cast<int64_t>(size),
        [&](int64_t address, int64_t accessSize) NN_NOEXCEPT -> nn::Result
        {
            NN_SDK_ASSERT_LESS_EQUAL(accessSize, static_cast<int64_t>(size));
            const size_t sizeWrite = static_cast<size_t>(accessSize);

            NN_RESULT_DO(m_StorageData.Write(address, p, sizeWrite));
            p += sizeWrite;
            NN_RESULT_SUCCESS;
        }));

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブル経由でファイルからデータを読み込みます。
*
* @param[out]   pRead   実際に読み込んだデータサイズ
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
#if defined(USE_RANGE_LOCK)
    // リーダーロックを取得します。
    ScopedReaderLock locker(&m_Lock, offset, size);
#endif // defined(USE_RANGE_LOCK)

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    char* p = static_cast<char*>(buffer);
    NN_RESULT_DO(IterateMappingTable(
        offset,
        static_cast<int64_t>(size),
        [&](int64_t address, int64_t accessSize) NN_NOEXCEPT -> nn::Result
        {
            NN_SDK_ASSERT_LESS_EQUAL(accessSize, static_cast<int64_t>(size));
            const size_t sizeRead = static_cast<size_t>(accessSize);

            NN_RESULT_DO(m_StorageData.Read(address, p, sizeRead));
            p += sizeRead;
            NN_RESULT_SUCCESS;
        }));

    NN_RESULT_SUCCESS;
}

/**
* @brief        書き込みのフラッシュを行ないます。
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::Flush() NN_NOEXCEPT
{
    return m_StorageData.Flush();
}

/**
* @brief        ファイルサイズを変更します。
*
* @param[in]    size    変更後のファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    return fs::ResultNotImplemented();
}

/**
* @brief        ファイルサイズを取得します。
*
* @param[out]   outValue    ファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_UNUSED(outValue);
    return fs::ResultNotImplemented();
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result JournalStorage::OperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::QueryRange:
        {
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(fs::QueryRangeInfo), fs::ResultInvalidSize());

            fs::QueryRangeInfo outInfo;
            outInfo.Clear();
            NN_RESULT_DO(IterateMappingTable(
                offset,
                size,
                [&](int64_t address, int64_t accessSize) NN_NOEXCEPT -> nn::Result
                {
                    fs::QueryRangeInfo info;
                    NN_RESULT_DO(m_StorageData.OperateRange(
                        &info,
                        sizeof(info),
                        operationId,
                        address,
                        accessSize,
                        inBuffer,
                        inBufferSize));
                    outInfo.Merge(info);
                    NN_RESULT_SUCCESS;
                }));

            *reinterpret_cast<fs::QueryRangeInfo*>(outBuffer) = outInfo;
        }
        break;
    default:
        NN_RESULT_THROW(fs::ResultUnsupportedOperation());
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブルエントリーのイテレーションを行います。
*
* @param[in]    offset  イテレーション開始位置
* @param[in]    size    データサイズ
* @param[in]    func    エントリーごとに行う処理
*
* @return       関数の処理結果を返します。
*/
template<typename TFunc>
Result JournalStorage::IterateMappingTable(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(nn::fs::IStorage::CheckOffsetAndSize(offset, size), nn::fs::ResultInvalidOffset());

    // 開始論理インデックスを指定して、マッピングテーブルエントリーの
    // イテレーションを開始します。
    MappingTable::Iterator iterator;
    const int64_t addressAligned = AlignAddress(offset);
    const int64_t sizeAligned = AlignUpAddress(offset + size) - addressAligned;
    NN_RESULT_DO(
        m_MappingTable.MakeIterator(
            &iterator,
            MappingTable::MakeIndex(addressAligned / GetBlockSize()),
            MappingTable::MakeIndex(sizeAligned / GetBlockSize())
        )
    );

    // 初回のみオフセットを考慮します。
    int64_t offsetBlock = offset % GetBlockSize();

    // 読み込み開始セクタからサイズ分だけ処理を行ないます。
    int64_t remainSize = size;
    while( remainSize > 0 )
    {
        const int64_t accessSize = std::min<int64_t>(remainSize, iterator.GetBlockCount() * GetBlockSize() - offsetBlock);
        int64_t address;
        NN_RESULT_DO(GetPhysicalAddress(&address, iterator.GetPhysicalIndex()));
        address += offsetBlock;

        // 処理関数を実行します。
        NN_RESULT_DO(func(address, accessSize));

        // 読み込んだサイズ分、位置とサイズを更新します。
        offsetBlock = 0;
        remainSize -= accessSize;

        // イテレーターの位置を更新します。
        if( remainSize > 0 )
        {
            NN_RESULT_DO(m_MappingTable.UpdateIterator(&iterator));
        }
    }

    NN_RESULT_SUCCESS;
}


}}} // namespace nn::fssystem::save

