﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <vector>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_SaveStorageFile.h>
#include <nn/fssystem/save/fs_IInternalStorageFileSystemVisitor.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>
#include <nn/fssystem/save/fs_UnionStorage.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    const int BufferCount = 2;
}

JournalIntegritySaveDataFileSystemDriver::JournalIntegritySaveDataFileSystemDriver() NN_NOEXCEPT
    : m_Locker(true)
{
}

JournalIntegritySaveDataFileSystemDriver::~JournalIntegritySaveDataFileSystemDriver() NN_NOEXCEPT
{
    Finalize();
}

// セーブデータをフォーマットするのに必要なデータサイズを計算します。
Result JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
           int64_t* outSizeTotal,
           size_t sizeBlock,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
           uint32_t countDataBlock,
           uint32_t countReservedBlock
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSizeTotal);
    return JournalIntegritySaveDataFileSystem::QuerySize(
               outSizeTotal,
               countExpandMax,
               paramDuplex,
               paramIntegrity,
               static_cast<uint32_t>(sizeBlock),
               countDataBlock,
               countReservedBlock
           );
}

// セーブデータをフォーマットするのに必要なデータサイズを計算します。
Result JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
           int64_t* outSizeTotal,
           size_t sizeBlock,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
           uint32_t countDataBlock,
           uint32_t countReservedBlock,
           uint32_t version
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSizeTotal);
    NN_RESULT_DO(
        JournalIntegritySaveDataFileSystem::QuerySize(
            outSizeTotal,
            countExpandMax,
            paramDuplex,
            paramIntegrity,
            static_cast<uint32_t>(sizeBlock),
            countDataBlock,
            countReservedBlock,
            version
        )
    );
    NN_RESULT_SUCCESS;
}

// セーブデータ領域のサイズから実データブロック数を求めます。
Result JournalIntegritySaveDataFileSystemDriver::QueryDataBlockCount(
           uint32_t* outCountDataBlock,
           int64_t sizeTotal,
           int64_t sizeReservedArea,
           size_t sizeBlock,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCountDataBlock);

    // ブロック数を変えながら、sizeCapacityに近い値を探し出します。
    uint32_t countDataBlockMin = 0;
    uint32_t countDataBlockMax = static_cast<uint32_t>(sizeTotal / sizeBlock);
    uint32_t countReservedBlock = static_cast<uint32_t>(sizeReservedArea / sizeBlock);
    uint32_t tries = 32;

    while( countDataBlockMax - countDataBlockMin >= 1 && (tries-- > 0) )
    {
        uint32_t countDataBlock = (countDataBlockMin + countDataBlockMax) / 2;
        int64_t sizeQuery = 0;
        NN_RESULT_DO(
            QueryTotalSize(
                &sizeQuery,
                sizeBlock,
                countExpandMax,
                paramDuplex,
                paramIntegrity,
                countDataBlock,
                countReservedBlock
            )
        );
        if( sizeQuery > sizeTotal )
        {
            countDataBlockMax = countDataBlock;
        }
        else
        {
            countDataBlockMin = countDataBlock;
        }

        // 途中抜け判定(これ以上継続しても変化なし)
        if( (countDataBlockMax - countDataBlockMin) <= 1
         && (countDataBlock == countDataBlockMin) )
        {
            break;
        }
    }

    {
        int64_t sizeQuery = 0;
        NN_RESULT_DO(
            QueryTotalSize(
                &sizeQuery,
                sizeBlock,
                countExpandMax,
                paramDuplex,
                paramIntegrity,
                countDataBlockMin,
                countReservedBlock
            )
        );
        if( sizeQuery > sizeTotal )
        {
            // メモリ不足です。
            *outCountDataBlock = 0;
            return nn::fs::ResultUsableSpaceNotEnough();
        }
    }

    *outCountDataBlock = countDataBlockMin;
    NN_RESULT_SUCCESS;
}

// セーブデータ領域をフォーマットします。
Result JournalIntegritySaveDataFileSystemDriver::Format(
           fs::SubStorage baseStorage,
           size_t sizeBlock,
           int countExpandMax,
           const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
           uint32_t countDataBlock,
           uint32_t countReservedBlock,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator,
           const nn::fs::SaveDataHashSalt& hashSalt
       ) NN_NOEXCEPT
{
    // ファイルアクセスはこの関数内で完結するので、
    // ローカルの排他制御オブジェクトで構いません
    nn::os::Mutex locker(true);

    FilesystemBufferManagerSet bufferManagerSet;
    for( auto& pBuffer : bufferManagerSet.pBuffer )
    {
        pBuffer = pBufferManager;
    }

    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [&]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            NN_RESULT_DO(JournalIntegritySaveDataFileSystem::Format(
                baseStorage,
                countExpandMax,
                paramDuplex,
                paramIntegrity,
                static_cast<uint32_t>(sizeBlock),
                countDataBlock,
                countReservedBlock,
                &bufferManagerSet,
                pBufferManager,
                &locker,
                pMacGenerator,
                hashSalt
            ));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));

    NN_RESULT_SUCCESS;
}

// セーブデータ ファイルシステムを拡張するのに必要なログ領域のサイズを計算します。
int64_t JournalIntegritySaveDataFileSystemDriver::QueryExpandLogSize(
            size_t sizeBlock,
            uint32_t countDataBlock,
            uint32_t countReservedBlock
        ) NN_NOEXCEPT
{
    // 大雑把に
    return (1024 * 1024) +
           (static_cast<int64_t>(countDataBlock) + static_cast<int64_t>(countReservedBlock)) *
           static_cast<int64_t>(sizeBlock) / (32 * 1024 * 1024) * (32 * 1024);
}

// セーブデータ領域を拡張します。
Result JournalIntegritySaveDataFileSystemDriver::OperateExpand(
           fs::SubStorage baseStorage,
           fs::SubStorage logStorage,
           size_t sizeBlock,
           uint32_t countDataBlock,
           uint32_t countReservedBlock,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    // ファイルアクセスはこの関数内で完結するので、
    // ローカルの排他制御オブジェクトで構いません
    nn::os::Mutex locker(true);

    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [&]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;

            UnionStorage unionStorage;
            int64_t sizeUnionStorage = 0;

            int64_t logStorageSize = 0;
            NN_RESULT_DO(logStorage.GetSize(&logStorageSize));
            BufferedStorage storageBufferingLog;
            NN_RESULT_DO(
                storageBufferingLog.Initialize(
                    logStorage,
                    pBufferManager,
                    sizeBlock,
                    BufferCount
                )
            );
            nn::fs::SubStorage bufferdLogStorage(&storageBufferingLog, 0, logStorageSize);
            NN_RESULT_DO(UnionStorage::Format(bufferdLogStorage, sizeBlock));
            NN_RESULT_DO(unionStorage.Initialize(baseStorage, bufferdLogStorage, sizeBlock));
            NN_RESULT_DO(unionStorage.GetSize(&sizeUnionStorage));

            NN_RESULT_DO(
                JournalIntegritySaveDataFileSystem::Expand(
                    fs::SubStorage(&unionStorage, 0, sizeUnionStorage),
                    static_cast<uint32_t>(sizeBlock),
                    countDataBlock,
                    countReservedBlock,
                    pBufferManager,
                    &locker,
                    pMacGenerator,
                    minimumVersion
                )
            );

            NN_RESULT_DO(unionStorage.Freeze());
            NN_RESULT_DO(unionStorage.Flush());

            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));

    NN_RESULT_SUCCESS;
}

//! セーブデータ領域の拡張を反映します。
Result JournalIntegritySaveDataFileSystemDriver::CommitExpand(
           fs::SubStorage baseStorage,
           fs::SubStorage logStorage,
           size_t sizeBlock,
           IBufferManager* pBufferManager
       ) NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [&]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;

            UnionStorage unionStorage;

            int64_t baseStorageSize = 0;
            NN_RESULT_DO(baseStorage.GetSize(&baseStorageSize));
            BufferedStorage storageBufferingBase;
            NN_RESULT_DO(
                storageBufferingBase.Initialize(
                    baseStorage,
                    pBufferManager,
                    sizeBlock,
                    16
                )
            );
            int64_t logStorageSize = 0;
            NN_RESULT_DO(logStorage.GetSize(&logStorageSize));
            BufferedStorage storageBufferingLog;
            NN_RESULT_DO(
                storageBufferingLog.Initialize(
                    logStorage,
                    pBufferManager,
                    sizeBlock,
                    BufferCount
                )
            );
            nn::fs::SubStorage bufferdBaseStorage(&storageBufferingBase, 0, baseStorageSize);
            nn::fs::SubStorage bufferdLogStorage(&storageBufferingLog, 0, logStorageSize);

            NN_RESULT_DO(unionStorage.Initialize(bufferdBaseStorage, bufferdLogStorage, sizeBlock));

            NN_RESULT_DO(unionStorage.Commit());
            NN_RESULT_DO(unionStorage.Flush());

            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! 拡張データを読み込みます
Result JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
           JournalIntegritySaveDataFileSystem::ExtraData* outData,
           fs::SubStorage baseStorage,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            NN_RESULT_DO(JournalIntegritySaveDataFileSystem::ReadExtraData(outData, baseStorage, pBufferManager, pMacGenerator, minimumVersion));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! セーブデータをマウントします。
Result JournalIntegritySaveDataFileSystemDriver::Initialize(
           fs::SubStorage baseStorage,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBufferManager);

    m_BaseStorage = baseStorage;

    m_pBufferManager = pBufferManager;
    for( auto& pBuffer : m_BufferManagerSet.pBuffer )
    {
        pBuffer = pBufferManager;
    }

    // m_SaveDataFileSystem.Initialize はリトライできないつくりなので、
    // 常にブロッキングでバッファを確保する
    buffers::ScopedBufferManagerContextRegistration scopedRegistration;
    buffers::EnableBlockingBufferManagerAllocation();

    // セーブデータをマウントします。
    NN_RESULT_DO(
        m_SaveDataFileSystem.Initialize(
            m_BaseStorage,
            &m_BufferManagerSet,
            m_pBufferManager,
            &m_Locker,
            pMacGenerator,
            minimumVersion
        )
    );

    ProxyFileSystemWithRetryingBufferAllocation::Initialize(&m_SaveDataFileSystem);

    NN_RESULT_SUCCESS;
}

//! セーブデータをアンマウントします。
void JournalIntegritySaveDataFileSystemDriver::Finalize() NN_NOEXCEPT
{
    // Result 返ってこないので、常に末端でリトライ
    buffers::ScopedBufferManagerContextRegistration scopedRegistration;
    buffers::EnableBlockingBufferManagerAllocation();

    m_SaveDataFileSystem.Finalize();
    ProxyFileSystemWithRetryingBufferAllocation::Finalize();
}

//! 拡張データを更新します
Result JournalIntegritySaveDataFileSystemDriver::WriteExtraData(
           const JournalIntegritySaveDataFileSystem::ExtraData& extraData
       ) NN_NOEXCEPT
{
    return m_SaveDataFileSystem.WriteExtraData(extraData);
}

//! 拡張データを取得します
void JournalIntegritySaveDataFileSystemDriver::ReadExtraData(
         JournalIntegritySaveDataFileSystem::ExtraData* outData
     ) NN_NOEXCEPT
{
    return m_SaveDataFileSystem.ReadExtraData(outData);
}

//! セーブデータへの変更をコミットします。
Result JournalIntegritySaveDataFileSystemDriver::Commit() NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            buffers::EnableBlockingBufferManagerAllocation();
            NN_RESULT_DO(m_SaveDataFileSystem.CommitFileSystem());
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! セーブデータへの変更を仮コミットします。
Result JournalIntegritySaveDataFileSystemDriver::CommitProvisionally(int64_t counter) NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            buffers::EnableBlockingBufferManagerAllocation();
            NN_RESULT_DO(m_SaveDataFileSystem.CommitFileSystemProvisionally(counter));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! セーブデータへの変更を巻き戻します。
Result JournalIntegritySaveDataFileSystemDriver::Rollback() NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            buffers::EnableBlockingBufferManagerAllocation();
            NN_RESULT_DO(m_SaveDataFileSystem.RollbackFileSystem());
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! MAC を更新します。
Result JournalIntegritySaveDataFileSystemDriver::UpdateMac(
           fs::SubStorage baseStorage,
           IMacGenerator* pMacGenerator,
           uint32_t minimumVersion
       ) NN_NOEXCEPT
{
    return JournalIntegritySaveDataFileSystem::UpdateMac(baseStorage, pMacGenerator, minimumVersion);
}

// 複数セーブデータの一括コミット用カウンタ値を取得します。
int64_t JournalIntegritySaveDataFileSystemDriver::GetCounterForBundledCommit() const NN_NOEXCEPT
{
    return m_SaveDataFileSystem.GetCounterForBundledCommit();
}

//! セーブデータ管理領域のパラメータを抽出します。
void JournalIntegritySaveDataFileSystemDriver::ExtractParameters(JournalIntegritySaveDataParameters* outParameters) NN_NOEXCEPT
{
    m_SaveDataFileSystem.ExtractParameters(
        &outParameters->paramDuplex,
        &outParameters->paramIntegrity,
        &outParameters->countDataBlock,
        &outParameters->countJournalBlock,
        &outParameters->blockSize,
        &outParameters->countExpandMax,
        &outParameters->version
    );
}

JournalIntegritySaveDataParameters JournalIntegritySaveDataFileSystemDriver::SetUpSaveDataParameters(int32_t blockSize, int64_t size, int64_t journalSize) NN_NOEXCEPT
{
    JournalIntegritySaveDataParameters params = {};

    params.paramDuplex.sizeBlockLevel[0] = blockSize;
    params.paramDuplex.sizeBlockLevel[1] = blockSize;

    params.paramIntegrity.sizeBlockLevel[0] = blockSize;
    params.paramIntegrity.sizeBlockLevel[1] = blockSize;
    params.paramIntegrity.sizeBlockLevel[2] = blockSize;
    params.paramIntegrity.sizeBlockLevel[3] = blockSize;

    // size = 利用可能サイズとする
    params.countDataBlock = static_cast<uint32_t>((size + (blockSize - 1)) / blockSize);
    params.countJournalBlock = static_cast<uint32_t>((journalSize + (blockSize - 1)) / blockSize);
    params.blockSize = blockSize;

    params.countExpandMax = 1;

    return params;
}

Result JournalIntegritySaveDataFileSystemDriver::AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVisitor);
    ScopedFSLock scopedLock(&m_Locker);
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            buffers::EnableBlockingBufferManagerAllocation();
            NN_RESULT_DO(m_SaveDataFileSystem.AcceptVisitor(pVisitor));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

}}}
