﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystem.h>
#include <nn/util/util_BinTypes.h>

// SaveDataFileSystemCore::QueryTotalSize を呼び出すためにインクルードします。
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    //! 完全性セーブデータファイルシステムの初期化済みコード 'ISFS'
    const uint32_t MagicCode = NN_UTIL_CREATE_SIGNATURE_4('I', 'S', 'F', 'S');

    //! ファイルシステムのバージョン
    const uint32_t VersionCode = 0x00010000;
    const uint32_t VersionCodeMask = 0xFFFF0000;

    const int BufferCount = 16;
    const int MetaBufferCount = 32;

    const auto MasterHashSize = 32;

    const int64_t AlignmentSmall = 512;
    const int64_t AlignmentLarge = 16 * 1024;

    inline int64_t AdjustAlign64(int64_t offset, int64_t align)
    {
        return ((offset + align - 1) & ~(align - 1));
    }
}

NN_DEFINE_STATIC_CONSTANT( const size_t IntegritySaveDataFileSystem::ControlAreaHashSize );
IntegritySaveDataFileSystem::ControlAreaHolder::ControlAreaHolder() NN_NOEXCEPT
    : m_pHeader(),
    m_pStorage(),
    m_BufferSize(0)
{
}

Result IntegritySaveDataFileSystem::ControlAreaHolder::Reset(const MasterHeader* pAtomic) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAtomic);

    if( pAtomic->controlArea.header.sizeMasterHash != MasterHashSize )
    {
        NN_RESULT_THROW(nn::fs::ResultInvalidIntegritySaveDataHashSize());
    }

    const auto bufferSize = static_cast<size_t>(GetControlAreaSize() + MasterHashSize);

    if( m_BufferSize == 0 )
    {
        m_BufferSize = bufferSize;

        m_pHeader = nn::fs::detail::MakeUnique<char[]>(m_BufferSize);
        if( m_pHeader == nullptr )
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedMakeUnique());
        }

        m_pStorage.reset(new nn::fs::MemoryStorage(m_pHeader.get(), static_cast<int64_t>(bufferSize)));
        if( m_pStorage == nullptr )
        {
            NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedNew());
        }
    }
    else
    {
        NN_SDK_ASSERT_GREATER_EQUAL(m_BufferSize, bufferSize);
    }

    NN_SDK_ASSERT_NOT_NULL(m_pHeader);
    NN_SDK_ASSERT_NOT_NULL(m_pStorage);

    const auto pDst = reinterpret_cast<char*>(m_pHeader.get());
    const auto pSrc = reinterpret_cast<const char*>(pAtomic);
    const auto controlAreaSize = GetControlAreaSize();
    std::memcpy(
        pDst,
        pSrc + offsetof(MasterHeader, hierarchicalIntegrityControlArea),
        controlAreaSize
    );
    std::memcpy(
        pDst + controlAreaSize,
        pSrc + pAtomic->controlArea.header.offsetMasterHash,
        MasterHashSize
    );

    NN_RESULT_SUCCESS;
}

void IntegritySaveDataFileSystem::ControlAreaHolder::Clear() NN_NOEXCEPT
{
    m_BufferSize = 0;
    m_pHeader.reset();
    m_pStorage.reset();
}

void IntegritySaveDataFileSystem::ControlAreaHolder::Commit(MasterHeader* pAtomic) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHeader);
    const auto pDst = reinterpret_cast<char*>(pAtomic);
    const auto pSrc = reinterpret_cast<const char*>(m_pHeader.get());
    const auto controlAreaSize = GetControlAreaSize();
    std::memcpy(
        pDst + offsetof(MasterHeader, hierarchicalIntegrityControlArea),
        pSrc,
        controlAreaSize);
    std::memcpy(
        pDst + pAtomic->controlArea.header.offsetMasterHash,
        pSrc + controlAreaSize,
        MasterHashSize
    );
}

inline IntegritySaveDataFileSystem::ExtraData* IntegritySaveDataFileSystem::ControlAreaHolder::GetExtraData() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHeader);
    return reinterpret_cast<ExtraData*>(m_pHeader.get() + offsetof(MasterHeader, extraData) - sizeof(MasterHeader::ControlArea));
}

inline const nn::fs::SubStorage IntegritySaveDataFileSystem::ControlAreaHolder::GetIntegrityControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    return nn::fs::SubStorage(
               m_pStorage.get(),
               offsetof(MasterHeader, hierarchicalIntegrityControlArea) - sizeof(MasterHeader::ControlArea),
               sizeof(decltype(MasterHeader::hierarchicalIntegrityControlArea))
           );
}

inline const nn::fs::SubStorage IntegritySaveDataFileSystem::ControlAreaHolder::GetSaveDataControlAreaStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    return nn::fs::SubStorage(
               m_pStorage.get(),
               offsetof(MasterHeader, saveDataControlArea) - sizeof(MasterHeader::ControlArea),
               sizeof(decltype(MasterHeader::saveDataControlArea))
           );
}

inline const nn::fs::SubStorage IntegritySaveDataFileSystem::ControlAreaHolder::GetMasterHashStorage() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorage);
    return nn::fs::SubStorage(m_pStorage.get(), GetControlAreaSize(), MasterHashSize);
}

inline size_t IntegritySaveDataFileSystem::ControlAreaHolder::GetControlAreaSize() NN_NOEXCEPT
{
    return offsetof(MasterHeader, originalExtraData) - offsetof(MasterHeader, hierarchicalIntegrityControlArea);
}

/**
* @brief        ファイルシステムをコミットします。
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::CommitFileSystemCore() NN_NOEXCEPT
{
    // ファイルやディレクトリオープン中はコミットできません。
    if( m_FileSystem.HasOpenedFiles(nn::fs::OpenMode_Write) )
    {
        NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
    }

    // キャッシュマネージャーに対して排他制御します。
    ScopedFSLock lock(GetLocker());

    // 完全性検証レイヤーに対してコミット処理を行います。
    // コミットすると変更フラグが落ちるので、事前に変更があったかどうかを問い合わせます。
    auto needsCommitMaster = m_IntegrityStorage.IsModified();
    NN_RESULT_DO(m_IntegrityStorage.Commit());

    // データが更新されたなら、完全性保証メタデータのフラッシュを行います。
    NN_RESULT_DO(m_StorageBufferingMeta.Flush());

    // 以下のいずれかの場合、マスター領域を更新します。
    // ・ファイルシステムに変更が加えられた時
    // ・拡張データ更新後のコミット時
    if( m_IsExtraDataModified )
    {
        needsCommitMaster = true;
    }

    if( needsCommitMaster )
    {
        MasterHeaderBufferType pAtomic;
        NN_RESULT_DO(GetMasterHeader(&pAtomic));
        nn::fs::MemoryStorage memoryStorageControlArea(pAtomic.get(), pAtomic.get_deleter().GetSize());

        // マスター領域に不正が無いことを検証します。
        NN_RESULT_DO(
            Verify(
                m_pMacGenerator,
                &pAtomic->controlArea.header,
                sizeof(pAtomic->controlArea.header),
                pAtomic->controlArea.signature,
                sizeof(pAtomic->controlArea.signature)
            )
        );

        //管理領域の変更を反映します。
        m_ControlArea.Commit(pAtomic.get());

        // 管理領域のSHAを計算します。
        nn::crypto::GenerateSha256Hash(
            pAtomic->controlArea.header.hash,
            sizeof(pAtomic->controlArea.header.hash),
            &pAtomic->hierarchicalIntegrityControlArea,
            sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
        );

        // ベースファイルのSignを用いてマスター管理領域の先頭に署名します。
        NN_RESULT_DO(
            Sign(
                &memoryStorageControlArea,
                m_pMacGenerator,
                pAtomic->controlArea.signature,
                sizeof(pAtomic->controlArea.signature),
                &pAtomic->controlArea.header,
                sizeof(pAtomic->controlArea.header)
            )
        );

        // 管理領域以外のデータをフラッシュします。
        NN_RESULT_DO(m_FileSystemStorage.Flush());

        // 管理領域を書き込みます。
        NN_RESULT_DO(
            m_BaseStorageControlArea.Write(
                0,
                pAtomic.get(),
                sizeof(MasterHeader)
            )
        );
        NN_RESULT_DO(m_BaseStorageControlArea.Flush());

        // 拡張データ更新
        std::memcpy(&pAtomic->originalExtraData, &pAtomic->extraData, sizeof(ExtraData));

        StoreMasterHeader(pAtomic.release(), pAtomic.get_deleter().GetSize());
    }

    NN_RESULT_SUCCESS;
}

/**
* @breif        指定のパラメータのセーブデータにおける各データの配置を計算します。
*
* @param[out]   outFileSystemLayoutHeader           レイアウトヘッダ
* @param[in]    inputParamIntegrity                 完全性検証パラメータ
* @param[in]    sizeBlock                           ブロックサイズ
* @param[in]    countDataBlock                      データブロック数
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::FormatFileSystemLayoutHeader(
           FileSystemLayoutHeader* outFileSystemLayoutHeader,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outFileSystemLayoutHeader);

    // データ構造
    //
    // マスター管理領域
    // {
    //     ヘッダ
    //
    //     階層ハッシュ検証管理領域
    //     セーブデータ管理領域
    //     拡張データ × 2
    //
    //     マスターハッシュ
    // }
    //
    // 完全性保証メタデータ
    // {
    //     メタデータ
    //     {
    //         階層ハッシュ L1, L2, L3
    //         セーブデータメタ
    //     }
    // }
    //
    // セーブデータ
    // {
    //     セーブデータ用領域
    // }

    // セーブデータ領域のサイズを求めます。
    const auto sizeCoreData = SaveDataFileSystemCore::QueryDataSize(
                                  sizeBlock,
                                  countDataBlock
                              );
    const auto sizeCoreMeta = SaveDataFileSystemCore::QueryMetaSize(
                                  sizeBlock,
                                  countDataBlock
                              );

    // 検証機能付きストレージ管理領域のサイズを取得します。
    HierarchicalIntegrityVerificationSizeSet integritySizes = { 0 };
    NN_RESULT_DO(
        IntegritySaveDataStorage::QuerySize(
            &integritySizes,
            inputParamIntegrity,
            sizeCoreData
        )
    );

    static const int64_t SizeMasterHash = static_cast<int64_t>(nn::crypto::Sha256Generator::HashSize);
    const auto sizeLayeredHashL1 = inputParamIntegrity.sizeBlockLevel[0];
    if( integritySizes.masterHashSize != static_cast<int64_t>(SizeMasterHash)
        || integritySizes.layeredHashSizes[0] != static_cast<int64_t>(sizeLayeredHashL1) )
    {
        NN_RESULT_THROW(nn::fs::ResultInvalidSize());
    }

    // メタデータ内のサイズを計算します。
    int64_t sizeMeta = 0;
    {
        outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL1
            = nn::util::align_up(
                integritySizes.layeredHashSizes[0],
                AlignmentSmall
            );
        sizeMeta
            += outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL1;
        outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL2
            = nn::util::align_up(
                  integritySizes.layeredHashSizes[1],
                  AlignmentSmall
              );
        sizeMeta
            += outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL2;
        outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL3
            = nn::util::align_up(
                  integritySizes.layeredHashSizes[2],
                  AlignmentSmall
              );
        sizeMeta
            += outFileSystemLayoutHeader->sizeHierarchicalIntegrityLayeredHashL3;
        outFileSystemLayoutHeader->sizeSaveDataMeta
            = nn::util::align_up(
                  sizeCoreMeta,
                  AlignmentSmall
              );
        sizeMeta
            += outFileSystemLayoutHeader->sizeSaveDataMeta;
    }

    // マスター管理領域内のオフセットとサイズを計算します。
    {
        int64_t offset = 0;
        offset += offsetof(MasterHeader, reserved);
        offset = AdjustAlign64(offset, sizeof(int64_t));

        outFileSystemLayoutHeader->offsetMasterHash = offset;
        outFileSystemLayoutHeader->sizeMasterHash = integritySizes.masterHashSize;
        offset += outFileSystemLayoutHeader->sizeMasterHash;
        offset = AdjustAlign64(offset, sizeof(int64_t));
        NN_SDK_ASSERT_EQUAL(integritySizes.masterHashSize, SizeMasterHash);
    }

    // ブロックサイズの最大値を取得します。
    const auto sizeBlockMax =
        (static_cast<size_t>(sizeBlock - 1) |
        (inputParamIntegrity.sizeBlockLevel[0] - 1) |
        (inputParamIntegrity.sizeBlockLevel[1] - 1) |
        (inputParamIntegrity.sizeBlockLevel[2] - 1) |
        (inputParamIntegrity.sizeBlockLevel[3] - 1) |
        (AlignmentLarge - 1)) + 1;

    // ファイルシステム内の各領域のオフセットとサイズを計算します。
    {
        outFileSystemLayoutHeader->sizeMeta
            = nn::util::align_up(
                  sizeMeta,
                  AlignmentSmall
              );
        outFileSystemLayoutHeader->sizeSaveDataCoreOriginal = sizeCoreData;
        outFileSystemLayoutHeader->sizeSaveDataCore
            = nn::util::align_up(
                  sizeCoreData,
                  static_cast<size_t>(sizeBlockMax)
              );
        outFileSystemLayoutHeader->offsetFileSystem
            = sizeof(MasterHeader);
        outFileSystemLayoutHeader->sizeFileSystem
            = nn::util::align_up(
                      + outFileSystemLayoutHeader->sizeMeta
                      + outFileSystemLayoutHeader->sizeSaveDataCore,
                  AlignmentLarge
              );
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result IntegritySaveDataFileSystem::GetMasterHeader(
           MasterHeaderBufferType* outAtomic
       ) NN_NOEXCEPT
{
    auto buffer = m_pBufferManager->AcquireCache(m_MasterHandle);
    if( buffer.first != 0 )
    {
        *outAtomic = MasterHeaderBufferType(
            reinterpret_cast<MasterHeader*>(buffer.first),
            buffers::detail::BufferManagerBufferDeallocator(m_pBufferManager, buffer.second)
        );
        NN_RESULT_SUCCESS;
    }
    else
    {
        *outAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(m_pBufferManager, NN_CURRENT_FUNCTION_NAME);

        NN_RESULT_DO(
            ReadMasterControlArea(
                outAtomic->get(),
                outAtomic->get_deleter().GetSize(),
                m_BaseStorageControlArea,
                m_pMacGenerator
            )
        );
        NN_RESULT_SUCCESS;
    }
}

void IntegritySaveDataFileSystem::StoreMasterHeader(
         MasterHeader* pAtomic,
         size_t size
     ) NN_NOEXCEPT
{
    m_MasterHandle = m_pBufferManager->RegisterCache(
                         reinterpret_cast<uintptr_t>(pAtomic),
                         size,
                         IBufferManager::BufferAttribute()
                     );
}

/**
* @brief        完全性検証機能付きセーブデータ ファイルシステムとしてフォーマットします。
*
* @param[in]    storage                  ストレージ
* @param[in]    inputParamIntegrity      完全性検証パラメータ
* @param[in]    sizeBlock                ブロックサイズ
* @param[in]    countDataBlock           データブロック数
* @param[in]    pIntegrityCacheBufferSet キャッシュマネージャ
* @param[in]    pCacheBuffer             キャッシュマネージャ
* @param[in]    pLocker                  キャッシュマネージャのスレッド間同期オブジェクト
* @param[in]    pMacGenerator            MAC 生成クラス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::Format(
           fs::SubStorage storage,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pCacheBuffer,
           os::Mutex* pLocker,
           IMacGenerator* pMacGenerator
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pLocker);

    // フォーマット処理中はキャッシュマネージャーに対して排他制御します。
    ScopedFSLock lock(pLocker);

    // マスター管理領域を初期化します。
    const auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pCacheBuffer, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    std::memset(pAtomic.get(), 0, sizeof(MasterHeader));
    FileSystemLayoutHeader& layoutHeader = pAtomic->controlArea.header;
    layoutHeader.magic = MagicCode;
    layoutHeader.version = VersionCode;

    int64_t sizeStorage = 0;
    NN_RESULT_DO(storage.GetSize(&sizeStorage));
    if( sizeStorage < static_cast<int64_t>(sizeof(MasterHeader)) )
    {
        NN_RESULT_THROW(nn::fs::ResultInvalidSize());
    }

    NN_RESULT_DO(storage.Write(0, pAtomic.get(), sizeof(MasterHeader)));

    // 各種データサイズを取得します。
    NN_RESULT_DO(
        FormatFileSystemLayoutHeader(
            &layoutHeader,
            inputParamIntegrity,
            sizeBlock,
            countDataBlock
        )
    );
    if( layoutHeader.offsetSaveData + layoutHeader.sizeSaveDataCore > sizeStorage )
    {
        NN_RESULT_THROW(nn::fs::ResultInvalidSize());
    }

    // 管理領域ごとにストレージを分割します。
    fs::MemoryStorage storageControlArea(
                      pAtomic.get(),
                      sizeof(MasterHeader)
                  );
    fs::SubStorage storageIntegrityControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, hierarchicalIntegrityControlArea),
                   sizeof(decltype(MasterHeader::hierarchicalIntegrityControlArea))
               );
    fs::SubStorage storageSaveDataControlArea(
                   &storageControlArea,
                   offsetof(MasterHeader, saveDataControlArea),
                   sizeof(decltype(MasterHeader::saveDataControlArea))
               );
    fs::SubStorage storageMasterHash(
                   &storageControlArea,
                   layoutHeader.offsetMasterHash,
                   layoutHeader.sizeMasterHash
               );

    // サブストレージを敷きます。
    BufferedStorage storageBuffering;
    storageBuffering.Initialize(
        storage,
        pCacheBuffer,
        sizeBlock,
        BufferCount
    );

    {
        int64_t offset = 0;

        layoutHeader.offsetsMeta = offset;
        offset += layoutHeader.sizeMeta;

        layoutHeader.offsetSaveData = offset;
        offset += layoutHeader.sizeSaveDataCore;

        NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.sizeFileSystem, offset);
    }

    fs::SubStorage storageFileSystem(
                   &storageBuffering,
                   layoutHeader.offsetFileSystem,
                   layoutHeader.sizeFileSystem
               );

    NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.offsetsMeta, 0);
    fs::SubStorage storageMeta(
                   &storageFileSystem,
                   layoutHeader.offsetsMeta,
                   layoutHeader.sizeMeta
               );

    NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.offsetSaveData, layoutHeader.offsetsMeta + layoutHeader.sizeMeta);
    fs::SubStorage storageIntegritySaveData(
                   &storageFileSystem,
                   layoutHeader.offsetSaveData,
                   layoutHeader.sizeSaveDataCoreOriginal
               );

    // さらにメタデータ用サブストレージを敷きます。
    {
        int64_t offset = 0;

        layoutHeader.offsetHierarchicalIntegrityLayeredHashL1 = offset;
        offset += layoutHeader.sizeHierarchicalIntegrityLayeredHashL1;

        layoutHeader.offsetHierarchicalIntegrityLayeredHashL2 = offset;
        offset += layoutHeader.sizeHierarchicalIntegrityLayeredHashL2;

        layoutHeader.offsetHierarchicalIntegrityLayeredHashL3 = offset;
        offset += layoutHeader.sizeHierarchicalIntegrityLayeredHashL3;

        layoutHeader.offsetSaveDataMeta = offset;
        offset += layoutHeader.sizeSaveDataMeta;

        NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.sizeMeta, offset);
    }

    NN_RESULT_DO(storageMeta.Flush());

    // メタデータ内をさらに分割します。
    NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.offsetHierarchicalIntegrityLayeredHashL1, 0);
    fs::SubStorage storageIntegrityLayeredHashL1(
                   &storageMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL1,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL1
               );

    NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.offsetHierarchicalIntegrityLayeredHashL2, layoutHeader.offsetHierarchicalIntegrityLayeredHashL1 + layoutHeader.sizeHierarchicalIntegrityLayeredHashL1);
    fs::SubStorage storageIntegrityLayeredHashL2(
                   &storageMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL2,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL2
               );
    NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.offsetHierarchicalIntegrityLayeredHashL3, layoutHeader.offsetHierarchicalIntegrityLayeredHashL2 + layoutHeader.sizeHierarchicalIntegrityLayeredHashL2);

    fs::SubStorage storageIntegrityLayeredHashL3(
                   &storageMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL3,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL3
               );

    NN_SDK_ASSERT_GREATER_EQUAL(layoutHeader.offsetSaveDataMeta, layoutHeader.offsetHierarchicalIntegrityLayeredHashL3 + layoutHeader.sizeHierarchicalIntegrityLayeredHashL3);
    fs::SubStorage storageSaveDataMeta(
                   &storageMeta,
                   layoutHeader.offsetSaveDataMeta,
                   layoutHeader.sizeSaveDataMeta
               );

    // FIXME: ここのスコープは別関数化した方が良いかも
    {
        // 階層ハッシュ検証付きストレージをフォーマットし、マウントします。
        HierarchicalIntegrityVerificationMetaInformation metaIntegrity;
        metaIntegrity.Format();
        metaIntegrity.sizeMasterHash = static_cast<uint32_t>(layoutHeader.sizeMasterHash);

        metaIntegrity.infoLevelHash.info[0].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL1;
        metaIntegrity.infoLevelHash.info[0].size = layoutHeader.sizeHierarchicalIntegrityLayeredHashL1;
        metaIntegrity.infoLevelHash.info[0].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[0]));

        metaIntegrity.infoLevelHash.info[1].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL2;
        metaIntegrity.infoLevelHash.info[1].size = layoutHeader.sizeHierarchicalIntegrityLayeredHashL2;
        metaIntegrity.infoLevelHash.info[1].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[1]));

        metaIntegrity.infoLevelHash.info[2].offset = layoutHeader.offsetHierarchicalIntegrityLayeredHashL3;
        metaIntegrity.infoLevelHash.info[2].size = layoutHeader.sizeHierarchicalIntegrityLayeredHashL3;
        metaIntegrity.infoLevelHash.info[2].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[2]));

        metaIntegrity.infoLevelHash.info[3].offset = 0;
        metaIntegrity.infoLevelHash.info[3].size = layoutHeader.sizeSaveDataCoreOriginal;
        metaIntegrity.infoLevelHash.info[3].orderBlock
            = ILog2(static_cast<uint32_t>(inputParamIntegrity.sizeBlockLevel[3]));

        NN_RESULT_DO(
            IntegritySaveDataStorage::Format(
                storageIntegrityControlArea,
                storageMasterHash,
                metaIntegrity
            )
        );

        HierarchicalIntegrityVerificationInformation infoIntegrity;
        {
            ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
            NN_RESULT_DO(controlArea->Initialize(storageIntegrityControlArea));
            controlArea->GetLevelHashInfo(&infoIntegrity);
            infoIntegrity.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
        }

        ScopedFinalizeObject<IntegritySaveDataStorage> storageIntegrity;
        NN_RESULT_DO(
            storageIntegrity->Initialize(
                storageMasterHash,
                storageIntegrityLayeredHashL1,
                storageIntegrityLayeredHashL2,
                storageIntegrityLayeredHashL3,
                storageIntegritySaveData,
                infoIntegrity,
                pIntegrityCacheBufferSet,
                pLocker
            )
        );

        // セーブデータ領域をフォーマットします。
        NN_RESULT_DO(
            SaveDataFileSystemCore::Format(
                storageSaveDataControlArea,
                storageSaveDataMeta,
                fs::SubStorage(
                    storageIntegrity.Ptr(),
                    0,
                    SaveDataFileSystemCore::QueryDataSize(sizeBlock, countDataBlock)
                ),
                sizeBlock,
                pCacheBuffer
            )
        );

        NN_RESULT_DO(storageIntegrity->Commit());
    }
    NN_RESULT_DO(
        storageMeta.Flush()
    );

    // 管理領域のSHAを計算します。
    nn::crypto::GenerateSha256Hash(
        pAtomic->controlArea.header.hash,
        sizeof(pAtomic->controlArea.header.hash),
        &pAtomic->hierarchicalIntegrityControlArea,
        sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
    );

    // ベースファイルのSignを用いてマスター管理領域の先頭に署名します。
    NN_RESULT_DO(Sign(&storage, pMacGenerator, pAtomic->controlArea.signature, sizeof(pAtomic->controlArea.signature), &pAtomic->controlArea.header, sizeof(pAtomic->controlArea.header)));

    // マスター管理領域をベースファイルに書き込みます。
    NN_RESULT_DO(storage.Write(0, pAtomic.get(), sizeof(MasterHeader)));
    NN_RESULT_DO(storage.Flush());

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        拡張データを取得します。
*
* @param[out]   outData     拡張データ
* @param[in]    storage     ストレージ
*
* @return       関数の処理結果を返します。
*
* @details      最後にコミットした際の拡張データを読み込みます。
*               すなわち、WriteExtraData で書き込んだデータは
*               コミットするまでこの関数では読み込めません。
*               マウントしていなくても実行可能です。
*/
Result IntegritySaveDataFileSystem::ReadExtraData(
           ExtraData* outData,
           fs::SubStorage storage,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outData);

    // マスター管理領域を読み込みます。
    const auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pBufferManager, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    NN_RESULT_DO(
        ReadMasterControlArea(
            pAtomic.get(),
            sizeof(MasterHeader),
            storage,
            pMacGenerator
        )
    );

    std::memcpy(
        outData,
        &pAtomic->originalExtraData,
        sizeof(ExtraData)
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証機能付きセーブデータファイルシステムをフォーマットするために必要なサイズを取得します。
*
* @param[out]   outSize             完全性検証に必要なサイズを取得します。
* @param[in]    inputParamIntegrity 完全性検証パラメータ
* @param[in]    sizeBlock           ブロックサイズ
* @param[in]    countDataBlock      データブロック数
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::QuerySize(
           int64_t* outSize,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam&
           inputParamIntegrity,
           uint32_t sizeBlock,
           uint32_t countDataBlock
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);

    FileSystemLayoutHeader layoutHeader;

    // 完全性検証に必要なサイズを求めます。
    const auto result = FormatFileSystemLayoutHeader(
                            &layoutHeader,
                            inputParamIntegrity,
                            sizeBlock,
                            countDataBlock
                        );
    if( result.IsSuccess() )
    {
        *outSize = layoutHeader.offsetFileSystem + layoutHeader.sizeFileSystem;
    }
    NN_RESULT_THROW(result);
}

/**
* @brief        署名を付けます。
*
* @param[in]    pStorage    計算したハッシュを格納するストレージ
* @param[in]    pHash       計算したハッシュを格納するバッファ
* @param[in]    pData       計算対象データ
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::Sign(
           fs::IStorage* pStorage,
           IMacGenerator* pMacGenerator,
           void* pHash,
           size_t hashSize,
           const void* pData,
           size_t dataSize
       ) NN_NOEXCEPT
{
    NN_UNUSED(pStorage);
    NN_UNUSED(pData);
    NN_SDK_REQUIRES_EQUAL(hashSize, static_cast<size_t>(MasterHeader::SignatureSize));
    NN_SDK_REQUIRES_NOT_NULL(pMacGenerator);

    pMacGenerator->Generate(pHash, hashSize, pData, dataSize);

    NN_RESULT_DO(pStorage->Write(0, pHash, hashSize));
    NN_RESULT_DO(pStorage->Flush());
    NN_RESULT_SUCCESS;
}

/**
* @brief        署名の正当性を検証します。
*
* @param[in]    pData         計算対象データ
* @param[in]    pSignature    比較に使用する署名
* @param[in]    signatureSize pSignature のサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::Verify(
           IMacGenerator* pMacGenerator,
           const void* pData,
           size_t dataSize,
           const void* pSignature,
           size_t signatureSize
       ) NN_NOEXCEPT
{
    NN_UNUSED(signatureSize);
    NN_SDK_REQUIRES_EQUAL(signatureSize, static_cast<size_t>(MasterHeader::SignatureSize));
    NN_SDK_REQUIRES_NOT_NULL(pMacGenerator);

    char buf[MasterHeader::SignatureSize];
    pMacGenerator->Generate(buf, sizeof(buf), pData, dataSize);

    if( !nn::crypto::IsSameBytes(pSignature, buf, sizeof(buf)) )
    {
        NN_RESULT_THROW(nn::fs::ResultIntegritySaveDataMasterSignatureVerificationFailed());
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        マスターヘッダの正当性を検証します。
*
* @param[in]    pMasterHeader   マスターヘッダ
*
* @return       関数の処理結果を返します。
*/
bool IntegritySaveDataFileSystem::VerifyMasterHeaderContent(
         MasterHeader* pMasterHeader
     ) NN_NOEXCEPT
{
    if(
        (pMasterHeader->controlArea.header.magic != MagicCode)
        || ((pMasterHeader->controlArea.header.version & VersionCodeMask)
            != (VersionCode & VersionCodeMask))
        || (pMasterHeader->controlArea.header.sizeSaveDataCore < 0)
        || (static_cast<int64_t>(sizeof(MasterHeader))
            < (pMasterHeader->controlArea.header.offsetMasterHash
                + pMasterHeader->controlArea.header.sizeMasterHash))
        )
    {
        return false;
    }

    // 読み込んだ管理領域に対してSHA検証を行います。
    char hashControlArea[ControlAreaHashSize];
    nn::crypto::GenerateSha256Hash(
        hashControlArea,
        sizeof(hashControlArea),
        &pMasterHeader->hierarchicalIntegrityControlArea,
        sizeof(MasterHeader) - sizeof(MasterHeader::ControlArea)
    );

    return nn::crypto::IsSameBytes(
               pMasterHeader->controlArea.header.hash,
               hashControlArea,
               ControlAreaHashSize
           );
}


/**
* @brief        コンストラクタ
*/
IntegritySaveDataFileSystem::IntegritySaveDataFileSystem() NN_NOEXCEPT
    : m_pCommitLocker(nullptr)
    , m_IsInitialized(false)
{
}

/**
* @brief        デストラクタ
*/
IntegritySaveDataFileSystem::~IntegritySaveDataFileSystem() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        マスター管理領域を読み込みます。
*
* @param[out]   outHeader                    マスター管理領域の読み込み先
* @param[in]    headerBufferSize             読み込み先バッファのサイズ
* @param[in]    storage                      マスター管理領域を読み込むストレージ
* @param[in]    pMacGenerator                MAC 生成クラス
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outHeader != nullptr
*/
Result IntegritySaveDataFileSystem::ReadMasterControlArea(
           MasterHeader* outHeader,
           size_t headerBufferSize,
           fs::SubStorage storage,
           IMacGenerator* pMacGenerator
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHeader);
    NN_SDK_REQUIRES_GREATER_EQUAL(headerBufferSize, sizeof(MasterHeader));
    NN_UNUSED(headerBufferSize);

    {
        int64_t sizeStorage = 0;
        NN_RESULT_DO(storage.GetSize(&sizeStorage));
        if( sizeStorage < static_cast<int64_t>(sizeof(MasterHeader::ControlArea)) )
        {
            NN_RESULT_THROW(nn::fs::ResultInvalidIntegritySaveDataControlAreaSize());
        }
    }

    {
        NN_RESULT_DO(
            storage.Read(
                0,
                outHeader,
                sizeof(MasterHeader)
            )
        );

        // マスター領域に不正が無いことを検証します。
        fs::SubStorage storageControlArea(
                       &storage,
                       0,
                       sizeof(MasterHeader)
                   );
        NN_RESULT_DO(
            Verify(
                pMacGenerator,
                &outHeader->controlArea.header,
                sizeof(outHeader->controlArea.header),
                outHeader->controlArea.signature,
                sizeof(outHeader->controlArea.signature)
            )
        );

        if( !VerifyMasterHeaderContent(outHeader) )
        {
            // 管理領域が壊れているならマウントできません。
            NN_RESULT_THROW(nn::fs::ResultIntegritySaveDataControlAreaVerificationFailed());
        }
    }

    // 更新前のデータを更新しておきます。
    {
        // 更新前の拡張データを更新
        std::memcpy(&outHeader->originalExtraData, &outHeader->extraData, sizeof(ExtraData));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        セーブデータをマウントします。
*
* @param[in]    layoutHeader                レイアウトヘッダ
* @param[in]    storageFileSystem           セーブデータファイルシステムをマウントするストレージ
* @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
* @param[in]    pCacheBuffer                キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::MountSaveData(
           const FileSystemLayoutHeader& layoutHeader,
           fs::SubStorage storageFileSystem,
           size_t sizeBlock,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pCacheBuffer,
           os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    // 管理領域ごとにストレージを分割します。
    NN_RESULT_DO(
        m_StorageBuffering.Initialize(
            storageFileSystem,
            pCacheBuffer,
            sizeBlock,
            BufferCount
        )
    );

    // ファイルシステム用サブストレージを敷きます。
    m_FileSystemStorage = fs::SubStorage(
                   &m_StorageBuffering,
                   layoutHeader.offsetFileSystem,
                   layoutHeader.sizeFileSystem
               );

    // さらにメタデータ用サブストレージを敷きます。
    m_MetaStorage = fs::SubStorage(
                   &m_FileSystemStorage,
                   layoutHeader.offsetsMeta,
                   layoutHeader.sizeMeta
               );
    fs::SubStorage storageBufferingMeta(
        &m_MetaStorage,
        0,
        layoutHeader.sizeMeta
    );
    NN_RESULT_DO(
        m_StorageBufferingMeta.Initialize(
            storageBufferingMeta,
            pCacheBuffer,
            sizeBlock,
            MetaBufferCount
        )
    );

    // 完全性検証セーブデータをマウントします。
    NN_RESULT_THROW(MountIntegritySaveData(
               layoutHeader,
               pIntegrityCacheBufferSet,
               pCacheBuffer,
               pLocker
           ));
} // NOLINT(impl/function_size)

/**
* @brief        完全性検証セーブデータをマウントします。
*
* @param[in]    layoutHeader                レイアウトヘッダ
* @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
* @param[in]    pSaveDataCoreCacheBufferSet キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::MountIntegritySaveData(
           const FileSystemLayoutHeader& layoutHeader,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pSaveDataCoreCacheBuffer,
           os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    // 領域ごとにストレージを分割します。
    fs::SubStorage storageIntegrityControlArea = m_ControlArea.GetIntegrityControlAreaStorage();
    fs::SubStorage storageSaveDataControlArea = m_ControlArea.GetSaveDataControlAreaStorage();
    fs::SubStorage storageMasterHash = m_ControlArea.GetMasterHashStorage();

    fs::SubStorage storageIntegritySaveData(
                   &m_FileSystemStorage,
                   layoutHeader.offsetSaveData,
                   layoutHeader.sizeSaveDataCoreOriginal
               );

    fs::SubStorage storageIntegrityLayeredHashL1(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL1,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL1
               );
    fs::SubStorage storageIntegrityLayeredHashL2(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL2,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL2
               );
    fs::SubStorage storageIntegrityLayeredHashL3(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetHierarchicalIntegrityLayeredHashL3,
                   layoutHeader.sizeHierarchicalIntegrityLayeredHashL3
               );
    fs::SubStorage storageSaveDataMeta(
                   &m_StorageBufferingMeta,
                   layoutHeader.offsetSaveDataMeta,
                   layoutHeader.sizeSaveDataMeta
               );

    HierarchicalIntegrityVerificationInformation infoIntegrity;
    {
        ScopedFinalizeObject<HierarchicalIntegrityVerificationStorageControlArea> controlArea;
        NN_RESULT_DO(controlArea->Initialize(storageIntegrityControlArea));
        controlArea->GetLevelHashInfo(&infoIntegrity);
        infoIntegrity.maxLayers = nn::fssystem::save::IntegrityLayerCountSave;
    }

    ScopedFinalizer<IntegritySaveDataStorage> intStorageFinalizer(&m_IntegrityStorage);
    NN_RESULT_DO(
        m_IntegrityStorage.Initialize(
            storageMasterHash,
            storageIntegrityLayeredHashL1,
            storageIntegrityLayeredHashL2,
            storageIntegrityLayeredHashL3,
            storageIntegritySaveData,
            infoIntegrity,
            pIntegrityCacheBufferSet,
            pLocker
        )
    );

    // 完全性検証付きジャーナリングストレージ上にファイルシステムをマウントします。
    NN_RESULT_DO(
        m_FileSystem.Initialize(
            storageSaveDataControlArea,
            storageSaveDataMeta,
            fs::SubStorage(&m_IntegrityStorage, 0, layoutHeader.sizeSaveDataCoreOriginal),
            pSaveDataCoreCacheBuffer
        )
    );

    intStorageFinalizer.Release();

    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証付きセーブデータファイルシステムをマウントします。
*
* @param[in]    storage                     ストレージ
* @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
* @param[in]    pCacheBuffer                キャッシュマネージャ
* @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
* @param[in]    pMacGenerator               MAC 生成クラス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::Initialize(
           fs::SubStorage storage,
           FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
           IBufferManager* pCacheBuffer,
           os::Mutex* pLocker,
           IMacGenerator* pMacGenerator
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pIntegrityCacheBufferSet);
    NN_SDK_ASSERT_NOT_NULL(pCacheBuffer);
    NN_SDK_ASSERT_NOT_NULL(pLocker);
    NN_SDK_ASSERT_NOT_NULL(pMacGenerator);

    m_pCommitLocker = pLocker;
    m_pMacGenerator = pMacGenerator;

    // キャッシュマネージャーに対して排他制御
    ScopedFSLock lock(GetLocker());

    m_pBufferManager = pCacheBuffer;
    m_MasterHandle = 0;

    fs::SubStorage subStorage = storage;

    // マスター管理領域をRAMへ読み出します。
    auto pAtomic = buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(pCacheBuffer, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_ASSERT_NOT_NULL(pAtomic);

    NN_RESULT_DO(
        ReadMasterControlArea(
            pAtomic.get(),
            sizeof(MasterHeader),
            subStorage,
            m_pMacGenerator
        )
    );

    const auto& layoutHeader = pAtomic->controlArea.header;

    // 初期化チェックします。
    if( layoutHeader.magic != MagicCode )
    {
        // フォーマットされていません。
        NN_RESULT_THROW(nn::fs::ResultIncorrectIntegritySaveDataMagicCode());
    }
    if( (layoutHeader.version & VersionCodeMask) != (VersionCode & VersionCodeMask) )
    {
        // データのバージョン互換性エラーです。
        NN_RESULT_THROW(nn::fs::ResultUnsupportedVersion());
    }

    NN_RESULT_DO(m_ControlArea.Reset(pAtomic.get()));

    // セーブデータストレージをマウントします。
    NN_RESULT_DO(
        MountSaveData(
            pAtomic->controlArea.header,
            storage,
            static_cast<size_t>(pAtomic->saveDataControlArea.sizeBlocks),
            pIntegrityCacheBufferSet,
            pCacheBuffer,
            pLocker
        )
    );

    m_BaseStorageControlArea = fs::SubStorage(&storage, 0, sizeof(MasterHeader));

    m_IsExtraDataModified = false;

    StoreMasterHeader(pAtomic.release(), pAtomic.get_deleter().GetSize());

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        完全性検証付きセーブデータファイルシステムをアンマウントします。
*               アンマウント時に自動Commitは行いません。
*/
void IntegritySaveDataFileSystem::Finalize() NN_NOEXCEPT
{
    if( m_IsInitialized )
    {
        // キャッシュマネージャーに対して排他制御
        ScopedFSLock lock(GetLocker());

        const auto cache = m_pBufferManager->AcquireCache(m_MasterHandle);
        if( cache.first != 0 )
        {
            m_pBufferManager->DeallocateBuffer(cache.first, cache.second);
        }
        m_pBufferManager = nullptr;

        m_FileSystem.Finalize();

        m_IntegrityStorage.Finalize();

        m_StorageBufferingMeta.Finalize();

        m_MetaStorage = fs::SubStorage();
        m_FileSystemStorage = fs::SubStorage();

        m_StorageBuffering.Finalize();

        m_BaseStorageControlArea = fs::SubStorage();

        m_ControlArea.Clear();

        m_IsInitialized = false;
    }

    m_pCommitLocker = nullptr;
}

/**
* @brief        拡張データを更新します。
*
* @param[in]    extraData   拡張データ
*
* @return       関数の処理結果を返します。
*               データが書き込まれるのは本コミットをした後です。
*/
Result IntegritySaveDataFileSystem::WriteExtraData(
           const ExtraData& extraData
       ) NN_NOEXCEPT
{
    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    // 拡張データを更新します。
    std::memcpy(m_ControlArea.GetExtraData(), &extraData, sizeof(ExtraData));

    m_IsExtraDataModified = true;

    NN_RESULT_SUCCESS;
}

/**
* @brief        拡張データを取得します。
*
* @param[out]   outData     拡張データ
*
* @return       関数の処理結果を返します。
*
* @details      WriteExtraData で書き込んだ値が読み込まれます。
*/
void IntegritySaveDataFileSystem::ReadExtraData(
         ExtraData* outData
     ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outData);
    std::memcpy(outData, m_ControlArea.GetExtraData(), sizeof(ExtraData));
}

/**
* @brief        ファイルを作成します。
*
* @param[in]    path    ファイルパス
* @param[in]    size    ファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::CreateFile(
           const Path& path,
           int64_t size
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.CreateFile(path, size));
}

/**
* @brief        ディレクトリを作成します。
*
* @param[in]    path ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::CreateDirectory(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.CreateDirectory(path));
}

/**
* @brief        ファイルを開きます。
*
* @param[out]   outValue    ファイルオブジェクト
* @param[in]    path        ファイルパス
* @param[in]    mode        オープンモード
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::OpenFile(
           IFile** outValue,
           const Path& path,
           int mode
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IFile* pInnerFile = nullptr;
    NN_RESULT_DO(m_FileSystem.OpenFile(&pInnerFile, path, mode));

    IFile* pFileWrapper = new IntegritySaveDataFile(pInnerFile, this);
    if( pFileWrapper == nullptr )
    {
        m_FileSystem.CloseFile(pInnerFile);

        // メモリ不足です。
        NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedNew());
    }

    *outValue = pFileWrapper;

    NN_RESULT_SUCCESS;
}

/**
* @brief ディレクトリを開きます。
*
* @param[out] outValue ディレクトリオブジェクト
* @param[in] path パス
*
* @return 関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::OpenDirectory(
           IDirectory** outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IDirectory* pInnerDirectory = nullptr;
    NN_RESULT_DO(m_FileSystem.OpenDirectory(&pInnerDirectory, path));

    IDirectory* pDirectoryWrapper = new IntegritySaveDataDirectory(pInnerDirectory, this);
    if( pDirectoryWrapper == nullptr )
    {
        m_FileSystem.CloseDirectory(pInnerDirectory);

        // メモリ不足です。
        NN_RESULT_THROW(nn::fs::ResultAllocationMemoryFailedNew());
    }

    *outValue = pDirectoryWrapper;

    NN_RESULT_SUCCESS;
}

/**
* @brief ファイルを閉じます。
*
* @param[in] pFile ファイルオブジェクト
*/
void IntegritySaveDataFileSystem::CloseFile(IFile* pFile) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFile);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IntegritySaveDataFile* pFile2 = reinterpret_cast<IntegritySaveDataFile*>(pFile);
    m_FileSystem.CloseFile(pFile2->GetFile());
    delete pFile;
}

/**
* @brief ディレクトリを閉じます。
*
* @param[in] pDirectory ディレクトリオブジェクト
*/
void IntegritySaveDataFileSystem::CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDirectory);

    // 操作中はロックします。
    ScopedFSLock lock(GetLocker());

    IntegritySaveDataDirectory* pDirectory2 = reinterpret_cast<IntegritySaveDataDirectory*>(pDirectory);
    m_FileSystem.CloseDirectory(pDirectory2->GetDirectory());
    delete pDirectory;
}

/**
* @brief        ファイルを削除します。
*
* @param[in]    path    ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::DeleteFile(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.DeleteFile(path));
}

/**
* @brief        ディレクトリを削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::DeleteDirectory(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.DeleteDirectory(path));
}

/**
* @brief        ディレクトリを再帰的に削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::DeleteDirectoryRecursively(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.DeleteDirectoryRecursively(path));
}

/**
* @brief        全ての子エントリを削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::CleanDirectoryRecursively(
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.CleanDirectoryRecursively(path));
}

/**
* @brief        ファイルをリネームします。
*
* @param[in]    oldPath 変更対象のファイルパス
* @param[in]    newPath 変更後のファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::RenameFile(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.RenameFile(oldPath, newPath));
}

/**
* @brief        ディレクトリをリネームします。
*
* @param[in]    oldPath 変更対象のディレクトリパス
* @param[in]    newPath 変更後のディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::RenameDirectory(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.RenameDirectory(oldPath, newPath));
}

/**
* @brief        指定したファイルが存在しているかどうかを取得します。
*
* @param[out]   outValue    ファイルが存在しているかどうか
* @param[in]    path        ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::HasFile(
           bool* outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.HasFile(outValue, path));
}

/**
* @brief        指定したディレクトリが存在しているかどうかを取得します。
*
* @param[out]   outValue    ディレクトリが存在しているかどうか
* @param[in]    path        ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::HasDirectory(
           bool* outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.HasDirectory(outValue, path));
}

/**
* @brief        空き領域のサイズをバイト数単位で取得します。
*
* @param[out]   outValue    空き領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::GetFreeBytes(int64_t* outValue) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.GetFreeBytes(outValue));
}

/**
* @brief        実データ領域のサイズをバイト数単位で取得します。
*
* @param[out]   outValue    実データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFileSystem::GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT
{
    ScopedFSLock lock(GetLocker());
    NN_RESULT_THROW(m_FileSystem.GetDataAreaBytes(outValue));
}

/**
* @brief        乱数列生成関数をセットします。
*
* @return       なし
*/
void IntegritySaveDataFileSystem::SetGenerateRandomFunction(
            GenerateRandomFunction function
        ) NN_NOEXCEPT
{
    HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(function);
}


/**
* @brief        ファイルの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFile::ReadBytes(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);

    // 操作中はロックします。
    ScopedFSLock lock(m_pFileSystem->GetLocker());

    NN_RESULT_DO(FileInFile::ReadBytes(offset, buffer, size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタ
*/
IntegritySaveDataFile::IntegritySaveDataFile(
    IFile* pFile,
    IntegritySaveDataFileSystem* pFileSystem
) NN_NOEXCEPT
: m_pFileSystem(pFileSystem)
{
    NN_SDK_ASSERT_NOT_NULL(pFileSystem);
    FileInFile::Initialize(pFile);
}

/**
* @brief        デストラクタ
*/
IntegritySaveDataFile::~IntegritySaveDataFile() NN_NOEXCEPT
{
}

/**
* @brief        下位ファイルオブジェクトへの参照を解除します。
*/
void IntegritySaveDataFile::Finalize() NN_NOEXCEPT
{
    FileInFile::Finalize();
    m_pFileSystem = nullptr;
}
/**
* @brief        バッファの内容をファイルに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
* @param[in]    option  書き込みオプション
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataFile::WriteBytes(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    // 操作中はロックします。
    ScopedFSLock lock(m_pFileSystem->GetLocker());

    NN_RESULT_DO(FileInFile::WriteBytes(offset, buffer, size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルサイズを変更します。
*
* @param[in]    size    変更後のファイルサイズ
*
* @return       失敗の結果を返します。
*/
Result IntegritySaveDataFile::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);

    // 操作中はロックします。
    ScopedFSLock lock(m_pFileSystem->GetLocker());

    NN_RESULT_DO(FileInFile::SetSize(size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタ
*/
IntegritySaveDataDirectory::IntegritySaveDataDirectory(
    IDirectory* pDirectory,
    IntegritySaveDataFileSystem* pFileSystem
) NN_NOEXCEPT
    : m_pFileSystem(pFileSystem),
      m_pDirectory(pDirectory)
{
}

/**
* @brief        デストラクタ
*/
IntegritySaveDataDirectory::~IntegritySaveDataDirectory() NN_NOEXCEPT
{
}

/**
* @brief        下位ディレクトリオブジェクトへの参照を解除します。
*/
void IntegritySaveDataDirectory::Finalize() NN_NOEXCEPT
{
    m_pFileSystem = nullptr;
    m_pDirectory = nullptr;
}

/**
* @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
*
* @param[out]   outNumEntries   取得したエントリー数
* @param[out]   outEntries      取得したエントリー情報
*                               配列のサイズは numEntries 以上である必要があります。
* @param[in]    numEntries      取得するエントリー数
*
* @return       関数の処理結果を返します。
*/
Result IntegritySaveDataDirectory::Read(
           int32_t* outNumEntries,
           nn::fs::DirectoryEntry outEntries[],
           int32_t numEntries
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
    ScopedFSLock lock(m_pFileSystem->GetLocker());
    NN_RESULT_THROW(m_pDirectory->Read(outNumEntries, outEntries, numEntries));
}

}}}
