﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <atomic>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>

namespace nn { namespace fssystem { namespace save {

namespace {
    const uintptr_t InvalidAddress = 0; //!< 無効なアドレスです。
    const int64_t InvalidOffset = std::numeric_limits<int64_t>::max(); //!< 無効なオフセットです。
}

/**
* @brief        キャッシュを保持するためのクラスです。
*
* @details      BufferedStorage から本クラスへのアクセスは
*               UniqueCache または SharedCache を介して行います。
*/
class BufferedStorage::Cache : public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(Cache);

public:
    /**
    * @brief        コンストラクタです。
    */
    Cache() NN_NOEXCEPT
    : m_pBufferedStorage(nullptr),
      m_MemoryRange(InvalidAddress, 0),
      m_CacheHandle(),
      m_Offset(InvalidOffset),
      m_IsValid(false),
      m_IsDirty(false),
      m_ReferenceCount(1),
      m_pNext(nullptr),
      m_pPrevious(nullptr)
    {
    }

    /**
    * @brief        デストラクタです。
    */
    ~Cache() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
    * @brief        キャッシュを初期化します。
    *
    * @param[in]    pBufferedStorage    親ストレージ
    *
    * @pre
    *               - pBufferedStorage != nullptr
    *               - キャッシュが未初期化
    */
    void Initialize(BufferedStorage* pBufferedStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBufferedStorage);
        NN_SDK_REQUIRES(m_pBufferedStorage == nullptr);

        m_pBufferedStorage = pBufferedStorage;
        Link();
    }

    /**
    * @brief        キャッシュの終了処理を行います。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが参照されていない
    *
    * @details      内部で保持されているバッファを解放します。
    *               デストラクタで呼び出されるため、明示的に呼び出す必要はありません。
    */
    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES_EQUAL(0, m_ReferenceCount);

        if( IsValid() )
        {
            const auto pBufferManager = m_pBufferedStorage->m_pBufferManager;
            if( !m_IsDirty )
            {
                NN_SDK_ASSERT_EQUAL(m_MemoryRange.first, InvalidAddress);
                m_MemoryRange = pBufferManager->AcquireCache(m_CacheHandle);
            }
            if( m_MemoryRange.first != InvalidAddress )
            {
                pBufferManager->DeallocateBuffer(m_MemoryRange.first, m_MemoryRange.second);
                m_MemoryRange.first = InvalidAddress;
                m_MemoryRange.second = 0;
            }
        }

        m_pBufferedStorage = nullptr;
        m_Offset = InvalidOffset;
        m_IsValid = false;
        m_IsDirty = false;
        m_pNext = nullptr;
        m_pPrevious = nullptr;
    }

    /**
    * @brief        キャッシュをフェッチ可能リストに追加します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *
    * @details      不要なバッファのキャッシュ登録または解放を行うことがあります。
    *
    *               本関数はスレッドセーフではありません。
    *               ロックしてから呼び出す必要があります。
    */
    void Link() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES_LESS(0, m_ReferenceCount);
        // BufferedStorage::Initialize() および BufferedStorage::Flush() からロックせずに呼び出される
        //NN_SDK_REQUIRES(m_pBufferedStorage->m_Mutex.IsLockedByCurrentThread());

        --m_ReferenceCount;
        if( 0 == m_ReferenceCount )
        {
            NN_SDK_ASSERT(m_pNext == nullptr);
            NN_SDK_ASSERT(m_pPrevious == nullptr);

            // フェッチ可能リストに追加する
            if( m_pBufferedStorage->m_pNextFetchCache == nullptr )
            {
                // リストが空ならこのキャッシュを唯一の要素として追加する
                m_pBufferedStorage->m_pNextFetchCache = this;
                m_pNext = this;
                m_pPrevious = this;
            }
            else
            {
                // 重複するキャッシュが登録されているならこのキャッシュを無効化する
                {
                    auto pCache = m_pBufferedStorage->m_pNextFetchCache;
                    do
                    {
                        if( pCache->IsValid() && Hits(pCache->m_Offset, m_pBufferedStorage->m_BlockSize) )
                        {
                            m_IsValid = false;
                            break;
                        }
                        pCache = pCache->m_pNext;
                    } while( pCache != m_pBufferedStorage->m_pNextFetchCache );
                }

                // リストの末尾に追加する
                {
                    NN_SDK_ASSERT_NOT_NULL(m_pBufferedStorage->m_pNextFetchCache->m_pPrevious);
                    NN_SDK_ASSERT_EQUAL(m_pBufferedStorage->m_pNextFetchCache,
                        m_pBufferedStorage->m_pNextFetchCache->m_pPrevious->m_pNext);
                    m_pNext = m_pBufferedStorage->m_pNextFetchCache;
                    m_pPrevious = m_pBufferedStorage->m_pNextFetchCache->m_pPrevious;
                    m_pNext->m_pPrevious = this;
                    m_pPrevious->m_pNext = this;
                }

                // このキャッシュが無効ならこれを先頭にする
                if( !IsValid() )
                {
                    m_pBufferedStorage->m_pNextFetchCache = this;
                }
            }

            // 無効なキャッシュの情報をクリアする
            if( !IsValid() )
            {
                m_Offset = InvalidOffset;
                m_IsDirty = false;
            }

            // 保持しているバッファがクリーンならキャッシュ登録または解放する
            if( m_MemoryRange.first != InvalidAddress && !m_IsDirty )
            {
                if( IsValid() )
                {
                    m_CacheHandle = m_pBufferedStorage->m_pBufferManager->RegisterCache(
                                        m_MemoryRange.first,
                                        m_MemoryRange.second,
                                        IBufferManager::BufferAttribute()
                                    );
                }
                else
                {
                    m_pBufferedStorage->m_pBufferManager->DeallocateBuffer(
                        m_MemoryRange.first,
                        m_MemoryRange.second
                    );
                }
                m_MemoryRange.first = InvalidAddress;
                m_MemoryRange.second = 0;
            }
        }
    }

    /**
    * @brief        キャッシュをフェッチ可能リストから削除します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *
    * @details      本関数はスレッドセーフではありません。
    *               ロックしてから呼び出す必要があります。
    */
    void Unlink() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        // BufferedStorage::Flush() からロックせずに呼び出される
        //NN_SDK_REQUIRES(m_pBufferedStorage->m_Mutex.IsLockedByCurrentThread());

        NN_SDK_REQUIRES_LESS_EQUAL(0, m_ReferenceCount);
        ++m_ReferenceCount;

        if( 1 == m_ReferenceCount )
        {
            // フェッチ可能リストから削除する
            NN_SDK_ASSERT_NOT_NULL(m_pNext);
            NN_SDK_ASSERT_NOT_NULL(m_pPrevious);
            NN_SDK_ASSERT_EQUAL(this, m_pNext->m_pPrevious);
            NN_SDK_ASSERT_EQUAL(this, m_pPrevious->m_pNext);

            if( m_pBufferedStorage->m_pNextFetchCache == this )
            {
                if( m_pNext != this )
                {
                    m_pBufferedStorage->m_pNextFetchCache = m_pNext;
                }
                else
                {
                    m_pBufferedStorage->m_pNextFetchCache = nullptr;
                }
            }

            m_pBufferedStorage->m_pNextAcquireCache = this;

            m_pNext->m_pPrevious = m_pPrevious;
            m_pPrevious->m_pNext = m_pNext;
            m_pNext = nullptr;
            m_pPrevious = nullptr;
        }
        else
        {
            NN_SDK_ASSERT(m_pNext == nullptr);
            NN_SDK_ASSERT(m_pPrevious == nullptr);
        }
    }

    /**
    * @brief        キャッシュバッファから読み込みます。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    * @param[out]   buffer  読み込んだデータを格納するバッファ
    * @param[in]    size    読み込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが有効化されている
    */
    void Read(int64_t offset, void* buffer, size_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES(Hits(offset, 1));
        NN_SDK_REQUIRES_NOT_EQUAL(m_MemoryRange.first, InvalidAddress);

        const auto readOffset = offset - m_Offset;
        const auto readableOffsetMax = m_pBufferedStorage->m_BlockSize - size;
        const auto cacheBuffer =  reinterpret_cast<char*>(m_MemoryRange.first) + readOffset;
        NN_SDK_ASSERT_LESS_EQUAL(0, readOffset);
        NN_SDK_ASSERT_LESS_EQUAL(static_cast<size_t>(readOffset), readableOffsetMax);
        NN_UNUSED(readableOffsetMax);

        const auto pDst = reinterpret_cast<char*>(buffer);
        std::copy(cacheBuffer, cacheBuffer + size, pDst);
    }

    /**
    * @brief        キャッシュバッファへ書き込みます。
    *
    * @param[in]    offset  書き込む位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer  書き込むデータを格納したバッファ
    * @param[in]    size    書き込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが有効化されている
    */
    void Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES(Hits(offset, 1));
        NN_SDK_REQUIRES_NOT_EQUAL(m_MemoryRange.first, InvalidAddress);

        const auto writeOffset = offset - m_Offset;
        const auto writableOffsetMax = m_pBufferedStorage->m_BlockSize - size;
        const auto cacheBuffer = reinterpret_cast<char*>(m_MemoryRange.first) + writeOffset;
        NN_SDK_ASSERT_LESS_EQUAL(0, writeOffset);
        NN_SDK_ASSERT_LESS_EQUAL(static_cast<size_t>(writeOffset), writableOffsetMax);
        NN_UNUSED(writableOffsetMax);

        const auto pSrc = reinterpret_cast<const char*>(buffer);
        std::copy(pSrc, pSrc + size, cacheBuffer);
        m_IsDirty = true;
    }

    /**
    * @brief        キャッシュバッファをフラッシュします。
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュがフェッチ可能リストにない
    *               - キャッシュが有効である
    */
    Result Flush() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(IsValid());

        if( m_IsDirty )
        {
            NN_SDK_REQUIRES_NOT_EQUAL(m_MemoryRange.first, InvalidAddress);

            const auto baseSize = m_pBufferedStorage->m_BaseStorageSize;
            const auto leftSize = baseSize - m_Offset;
            const auto blockSize = static_cast<int64_t>(m_pBufferedStorage->m_BlockSize);
            const auto flushSize = static_cast<size_t>(std::min(blockSize, leftSize));
            auto& baseStorage = m_pBufferedStorage->m_BaseStorage;
            const auto cacheBuffer = reinterpret_cast<void*>(m_MemoryRange.first);

            NN_RESULT_DO(baseStorage.Write(m_Offset, cacheBuffer, flushSize));
            m_IsDirty = false;

            // 下位ストレージへの書き込みがあったら、それ以後はバッファ確保でリトライする
            buffers::EnableBlockingBufferManagerAllocation();
        }

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        フェッチの準備をします。
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュがフェッチ可能リストにない
    *               - キャッシュが有効
    *
    * @details      ダーティである場合、内部でフラッシュを行います。
    *
    *               戻り値の first は処理に成功したかどうかを示します。
    *               second はフェッチできる状態になったかどうかを示します。
    *               処理が成功してもフェッチ可能な状態にならない場合があることに注意してください。
    *
    *               本関数はスレッドセーフではありません。
    *               ロックしてから呼び出す必要があります。
    */
    const std::pair<Result, bool> PrepareFetch() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES(m_pBufferedStorage->m_Mutex.IsLockedByCurrentThread());

        std::pair<Result, bool> result(nn::ResultSuccess(), false);
        if( m_ReferenceCount == 1 )
        {
            result.first = Flush();
            if( result.first.IsSuccess() )
            {
                m_IsValid = false;
                m_ReferenceCount = 0;
                result.second = true;
            }
        }
        return result;
    }

    /**
    * @brief        フェッチを終了します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュがフェッチ可能リストにない
    *               - キャッシュが無効
    *               - キャッシュがクリーン
    *
    * @details      本関数はスレッドセーフではありません。
    *               ロックしてから呼び出す必要があります。
    */
    void UnprepareFetch() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(!IsValid());
        NN_SDK_REQUIRES(!m_IsDirty);
        NN_SDK_REQUIRES(m_pBufferedStorage->m_Mutex.IsLockedByCurrentThread());

        m_IsValid = true;
        m_ReferenceCount = 1;
    }

    /**
    * @brief        元ストレージからデータを読み込んで新たにキャッシュバッファに格納します。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュがフェッチ可能リストにない
    *               - キャッシュが無効化されている
    *               - キャッシュがクリーン
    */
    Result Fetch(int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(!IsValid());
        NN_SDK_REQUIRES(!m_IsDirty);

        // 直前までダーティなバッファを保持していたならそれを使い、さもなければバッファを確保する
        if( m_MemoryRange.first == InvalidAddress )
        {
            NN_RESULT_DO(AllocateFetchBuffer());
        }

        FetchParameter fetchParameter = {};
        CalcFetchParameter(&fetchParameter, offset);

        // データを読み込む
        auto& baseStorage = m_pBufferedStorage->m_BaseStorage;
        NN_RESULT_DO(baseStorage.Read(fetchParameter.offset, fetchParameter.buffer, fetchParameter.size));
        m_Offset = fetchParameter.offset;
        NN_SDK_ASSERT(Hits(offset, 1));

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        バッファからデータを読み込んで新たにキャッシュバッファに格納します。
    *
    * @param[in]    offset      読み込む位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer      読み込むバッファ
    * @param[in]    bufferSize  読み込むバッファのサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュがフェッチ可能リストにない
    *               - キャッシュが無効化されている
    *               - キャッシュがクリーン
    *               - offset がブロックサイズでアラインされている
    */
    Result FetchFromBuffer(int64_t offset, const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_UNUSED(bufferSize);

        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES(m_pNext == nullptr);
        NN_SDK_REQUIRES(m_pPrevious == nullptr);
        NN_SDK_REQUIRES(!IsValid());
        NN_SDK_REQUIRES(!m_IsDirty);
        NN_SDK_REQUIRES(util::is_aligned(offset, m_pBufferedStorage->m_BlockSize));

        // 直前までダーティなバッファを保持していたならそれを使い、さもなければバッファを確保する
        if( m_MemoryRange.first == InvalidAddress )
        {
            NN_RESULT_DO(AllocateFetchBuffer());
        }

        FetchParameter fetchParameter = {};
        CalcFetchParameter(&fetchParameter, offset);
        NN_SDK_ASSERT_EQUAL(fetchParameter.offset, offset);
        NN_SDK_ASSERT_LESS_EQUAL(fetchParameter.size, bufferSize);

        std::memcpy(fetchParameter.buffer, buffer, fetchParameter.size);
        m_Offset = fetchParameter.offset;
        NN_SDK_ASSERT(Hits(offset, 1));

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        キャッシュバッファの取得を試みます。
    *
    * @return       取得に成功したかどうかを返します。
    * @retval       true    キャッシュバッファの取得に成功しました。
    * @retval       false   有効なキャッシュバッファが取得できなかったため、再度フェッチする必要があります。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが有効
    *
    * @details      有効なキャッシュバッファが取得できなかった場合、このキャッシュは自動的に無効化されます。
    *
    *               本関数はスレッドセーフではありません。
    *               ロックしてから呼び出す必要があります。
    */
    bool TryAcquireCache() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage->m_pBufferManager);
        NN_SDK_REQUIRES(IsValid());

        if( m_MemoryRange.first != InvalidAddress )
        {
            return true;
        }
        else
        {
            // ロックされずに呼び出されることがある
            //NN_SDK_REQUIRES(m_pBufferedStorage->m_Mutex.IsLockedByCurrentThread());
            m_MemoryRange = m_pBufferedStorage->m_pBufferManager->AcquireCache(m_CacheHandle);
            m_IsValid = m_MemoryRange.first != 0;
            return m_IsValid;
        }
    }

    /**
    * @brief        キャッシュを無効化します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *
    * @details      キャッシュが実際に無効になるのはそれが参照されなくなったときであり、
    *               それまでは有効なキャッシュとして扱われます。
    */
    void Invalidate() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        m_IsValid = false;
    }

    /**
    * @brief        キャッシュが有効かどうか判定します。
    *
    * @return       判定の結果を返します。
    * @retval       true    キャッシュは有効です。
    * @retval       false   キャッシュは無効です。
    *
    * @pre
    *               - キャッシュが初期化済み
    *
    * @details      キャッシュが実際に無効になるのはそれが参照されなくなったときであり、
    *               それまでは有効なキャッシュとして扱われます。
    */
    bool IsValid() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);

        // 無効化されても参照されている間は有効扱い
        return m_IsValid || 0 < m_ReferenceCount;
    }

    /**
    * @brief        キャッシュが書き換え済みかどうか判定します。
    *
    * @return       判定結果を返します。
    */
    bool IsDirty() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        return m_IsDirty;
    }

    /**
    * @brief        指定の範囲と重なる範囲をキャッシュしているかどうか判定します。
    *
    * @param[in]    offset  判定する範囲の親ストレージの先頭からのオフセット
    * @param[in]    size    判定する範囲のサイズ
    *
    * @return       判定の結果を返します。
    * @retval       true    指定の範囲と重なる範囲をキャッシュしています。
    * @retval       false   指定の範囲と重なる範囲はキャッシュされていません。
    *
    * @pre
    *               - キャッシュが初期化済み
    */
    bool Hits(int64_t offset, int64_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);
        const auto blockSize = static_cast<int64_t>(m_pBufferedStorage->m_BlockSize);
        return (offset < m_Offset + blockSize) && (m_Offset < offset + size);
    }

private:
    struct FetchParameter
    {
        int64_t offset;
        void* buffer;
        size_t size;
    };
    NN_STATIC_ASSERT(std::is_pod<FetchParameter>::value);

private:
    /**
    * @brief        フェッチ用にバッファを確保します。
    *
    * @return       処理の結果を返します。
    */
    Result AllocateFetchBuffer() NN_NOEXCEPT
    {
        IBufferManager* pBufferManager = m_pBufferedStorage->m_pBufferManager;

        NN_SDK_ASSERT_EQUAL(pBufferManager->AcquireCache(m_CacheHandle).first, InvalidAddress);
        auto result = buffers::AllocateBufferUsingBufferManagerContext(
            &m_MemoryRange,
            pBufferManager,
            m_pBufferedStorage->m_BlockSize,
            IBufferManager::BufferAttribute(),
            [](const std::pair<uintptr_t, size_t>& buffer) NN_NOEXCEPT
            {
                return buffer.first != 0;
            },
            NN_CURRENT_FUNCTION_NAME
        );
        if( result.IsFailure() )
        {
            m_MemoryRange.first = InvalidAddress;
            NN_RESULT_THROW(result);
        }

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        フェッチで使用するパラメータを求めます。
    */
    void CalcFetchParameter(FetchParameter* pOutValue, int64_t offset) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        const auto blockSize = static_cast<int64_t>(m_pBufferedStorage->m_BlockSize);
        const auto cacheOffset = util::align_down(offset, m_pBufferedStorage->m_BlockSize);
        const auto baseSize = m_pBufferedStorage->m_BaseStorageSize;
        const auto leftSize = baseSize - cacheOffset;
        const auto cacheSize = static_cast<size_t>(std::min(blockSize, leftSize));
        const auto cacheBuffer = reinterpret_cast<void*>(m_MemoryRange.first);
        NN_SDK_ASSERT_LESS_EQUAL(0, offset);
        NN_SDK_ASSERT_LESS(offset, baseSize);

        pOutValue->offset = cacheOffset;
        pOutValue->buffer = cacheBuffer;
        pOutValue->size = cacheSize;
    }

private:
    BufferedStorage*             m_pBufferedStorage; //!< バッファ付きストレージ
    std::pair<uintptr_t, size_t> m_MemoryRange;      //!< バッファのメモリ範囲
    IBufferManager::CacheHandle  m_CacheHandle;      //!< キャッシュハンドル
    int64_t                      m_Offset;           //!< ストレージ内オフセット
    std::atomic<bool>            m_IsValid;          //!< 有効フラグ
    std::atomic<bool>            m_IsDirty;          //!< ダーティフラグ
    char                         reserved[2];
    int                          m_ReferenceCount;   //!< 参照カウント
    Cache*                       m_pNext;            //!< 次のフェッチ可能キャッシュ
    Cache*                       m_pPrevious;        //!< 前のフェッチ可能キャッシュ
};

/**
* @brief        同じキャッシュの所有権を共有するクラスです。
*
* @details      AcquireNextOverlappedCache(), AcquireNextCache() または AcquireFetchableCache()
*               によってキャッシュを獲得し、デストラクタでキャッシュを解放します。
*               他の SharedCache が所有しているキャッシュを所有することはできますが、
*               他の UniqueCache が所有しているキャッシュの所有はできません。
*
*               AcquireNextOverlappedCache() または AcquireFetchableCache() による
*               キャッシュの獲得はスレッドセーフですが、
*               AcquireNextCache() や獲得したキャッシュへのアクセスはスレッドアンセーフです。
*/
class BufferedStorage::SharedCache
{
    friend class UniqueCache;

    NN_DISALLOW_COPY(SharedCache);
    NN_DISALLOW_MOVE(SharedCache);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @param[in]    pBufferedStorage    親ストレージ
    *
    * @pre
    *               - pBufferedStorage != nullptr
    */
    explicit SharedCache(BufferedStorage* pBufferedStorage) NN_NOEXCEPT
        : m_pCache(nullptr),
          m_pStartCache(pBufferedStorage->m_pNextAcquireCache),
          m_pBufferedStorage(pBufferedStorage)
    {
        NN_SDK_REQUIRES_NOT_NULL(pBufferedStorage);
    }

    /**
    * @brief        デストラクタです。
    *
    * @details      所有しているキャッシュを解放します。
    */
    ~SharedCache() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);
        Release();
    }

    /**
    * @brief        指定の範囲と重なる範囲を持つ次のキャッシュの取得を試みます。
    *
    * @param[in]    offset  取得するキャッシュ範囲の親ストレージの先頭からのオフセット
    * @param[in]    size    取得するキャッシュ範囲のサイズ
    *
    * @return       処理の結果を返します。
    * @retval       true    キャッシュの取得に成功しました。
    * @retval       false   取得できるキャッシュはありませんでした。
    *
    * @pre
    *               - 初期化済みである
    */
    bool AcquireNextOverlappedCache(int64_t offset, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);

        auto isInitial = m_pCache == nullptr;
        const auto pBegin = isInitial ? m_pStartCache : m_pCache + 1;

        NN_SDK_ASSERT_GREATER_EQUAL(pBegin, m_pBufferedStorage->m_pCaches.get());
        NN_SDK_ASSERT_LESS_EQUAL(
            pBegin,
            m_pBufferedStorage->m_pCaches.get() + m_pBufferedStorage->m_CacheCount
        );

        std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);

        Release();
        NN_SDK_ASSERT(m_pCache == nullptr);

        for( auto pCache = pBegin; ; ++pCache )
        {
            if( m_pBufferedStorage->m_pCaches.get() + m_pBufferedStorage->m_CacheCount <= pCache )
            {
                pCache = m_pBufferedStorage->m_pCaches.get();
            }
            if( !isInitial && pCache == m_pStartCache )
            {
                break;
            }
            if( pCache->IsValid() && pCache->Hits(offset, size) && pCache->TryAcquireCache() )
            {
                pCache->Unlink();
                m_pCache = pCache;
                return true;
            }
            isInitial = false;
        }

        m_pCache = nullptr;
        return false;
    }

    /**
    * @brief        次のダーティなキャッシュの取得を試みます。
    *
    * @return       処理の結果を返します。
    * @retval       true    キャッシュの取得に成功しました。
    * @retval       false   取得できるキャッシュはありませんでした。
    *
    * @pre
    *               - 初期化済みである
    *
    * @details      本関数はスレッドアンセーフです。
    */
    bool AcquireNextDirtyCache() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);

        const auto pBegin
            = m_pCache != nullptr ? m_pCache + 1 : m_pBufferedStorage->m_pCaches.get();
        const auto pEnd = m_pBufferedStorage->m_pCaches.get() + m_pBufferedStorage->m_CacheCount;

        // Flush() からしか呼ばれないはずなのでロックしない
        // std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);

        NN_SDK_ASSERT_GREATER_EQUAL(pBegin, m_pBufferedStorage->m_pCaches.get());
        NN_SDK_ASSERT_LESS_EQUAL(pBegin, pEnd);

        Release();
        NN_SDK_ASSERT(m_pCache == nullptr);

        for( auto pCache = pBegin; pCache < pEnd; ++pCache )
        {
            if( pCache->IsValid() && pCache->IsDirty() && pCache->TryAcquireCache() )
            {
                pCache->Unlink();
                m_pCache = pCache;
                return true;
            }
        }

        m_pCache = nullptr;
        return false;
    }

    /**
    * @brief        次の有効なキャッシュの取得を試みます。
    *
    * @return       処理の結果を返します。
    * @retval       true    キャッシュの取得に成功しました。
    * @retval       false   取得できるキャッシュはありませんでした。
    *
    * @pre
    *               - 初期化済みである
    *
    * @details      本関数はスレッドアンセーフです。
    */
    bool AcquireNextValidCache() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBufferedStorage);

        const auto pBegin
            = m_pCache != nullptr ? m_pCache + 1 : m_pBufferedStorage->m_pCaches.get();
        const auto pEnd = m_pBufferedStorage->m_pCaches.get() + m_pBufferedStorage->m_CacheCount;

        // InvalidateCaches() からしか呼ばれないはずなのでロックしない
        // std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);

        NN_SDK_ASSERT_GREATER_EQUAL(pBegin, m_pBufferedStorage->m_pCaches.get());
        NN_SDK_ASSERT_LESS_EQUAL(pBegin, pEnd);

        Release();
        NN_SDK_ASSERT(m_pCache == nullptr);

        for( auto pCache = pBegin; pCache < pEnd; ++pCache )
        {
            if( pCache->IsValid() && pCache->TryAcquireCache() )
            {
                pCache->Unlink();
                m_pCache = pCache;
                return true;
            }
        }

        m_pCache = nullptr;
        return false;
    }

    /**
    * @brief        フェッチ可能なキャッシュの取得を試みます。
    *
    * @return       処理の結果を返します。
    * @retval       true    キャッシュの取得に成功しました。
    * @retval       false   取得できるキャッシュはありませんでした。
    *
    * @pre
    *               - 初期化済みである
    */
    bool AcquireFetchableCache() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pBufferedStorage);

        std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);

        Release();
        NN_SDK_ASSERT(m_pCache == nullptr);

        m_pCache = m_pBufferedStorage->m_pNextFetchCache;
        if( m_pCache != nullptr )
        {
            if( m_pCache->IsValid() )
            {
                m_pCache->TryAcquireCache();
            }
            m_pCache->Unlink();
        }

        return m_pCache != nullptr;
    }

    /**
    * @brief        所有しているキャッシュから読み込みます。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    * @param[out]   buffer  読み込んだデータを格納するバッファ
    * @param[in]    size    読み込むサイズ
    *
    * @pre
    *               - キャッシュを所有している
    */
    void Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        m_pCache->Read(offset, buffer, size);
    }

    /**
    * @brief        所有しているキャッシュに書き込みます。
    *
    * @param[in]    offset  書き込む位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer  書き込むデータを格納したバッファ
    * @param[in]    size    書き込むサイズ
    *
    * @pre
    *               - キャッシュを所有している
    */
    void Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        m_pCache->Write(offset, buffer, size);
    }

    /**
    * @brief        所有しているキャッシュをフラッシュします。
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュを所有している
    */
    Result Flush() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        return m_pCache->Flush();
    }

    /**
    * @brief        所有しているキャッシュを無効化します。
    *
    * @pre
    *               - キャッシュを所有している
    */
    void Invalidate() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        m_pCache->Invalidate();
    }

    /**
    * @brief        指定の範囲と重なる範囲をキャッシュしているかどうか判定します。
    *
    * @param[in]    offset  判定する範囲の親ストレージの先頭からのオフセット
    * @param[in]    size    判定する範囲のサイズ
    *
    * @return       判定の結果を返します。
    * @retval       true    指定の範囲と重なる範囲をキャッシュしています。
    * @retval       false   指定の範囲と重なる範囲はキャッシュされていません。
    *
    * @pre
    *               - キャッシュが初期化済み
    */
    bool Hits(int64_t offset, int64_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        return m_pCache->Hits(offset, size);
    }

private:
    /**
    * @brief        所有しているキャッシュを解放します。
    */
    void Release() NN_NOEXCEPT
    {
        if( m_pCache != nullptr )
        {
            NN_SDK_ASSERT_LESS_EQUAL(&m_pBufferedStorage->m_pCaches[0], m_pCache);
            NN_SDK_ASSERT_LESS_EQUAL(m_pCache,
                &m_pBufferedStorage->m_pCaches[m_pBufferedStorage->m_CacheCount]);

            m_pCache->Link();
            m_pCache = nullptr;
        }
    }

private:
    Cache*           m_pCache;           //!< 所有しているキャッシュ
    Cache*           m_pStartCache;      //!< キャッシュの探索開始位置
    BufferedStorage* m_pBufferedStorage; //!< 親ストレージ
};

/**
* @brief        キャッシュの所有権を占有するクラスです。
*
* @details      キャッシュをフェッチするために使用します。
*               Upgrade() によって SharedCache から昇格し、デストラクタで降格します。
*               本クラスが所有するキャッシュは他の SharedCache によって所有されません。
*
*               キャッシュの昇格および降格はスレッドセーフですが、
*               昇格したキャッシュへのアクセスはスレッドアンセーフです。
*/
class BufferedStorage::UniqueCache
{
    NN_DISALLOW_COPY(UniqueCache);
    NN_DISALLOW_MOVE(UniqueCache);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @param[in]    pBufferedStorage    親ストレージ
    *
    * @pre
    *               - pBufferedStorage != nullptr
    */
    explicit UniqueCache(BufferedStorage* pBufferedStorage) NN_NOEXCEPT
        : m_pCache(nullptr),
          m_pBufferedStorage(pBufferedStorage)
    {
        NN_SDK_REQUIRES_NOT_NULL(pBufferedStorage);
    }

    /**
    * @brief        デストラクタです。
    *
    * @details      キャッシュを共有キャッシュに降格します。
    */
    ~UniqueCache() NN_NOEXCEPT
    {
        if( m_pCache != nullptr )
        {
            std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);
            m_pCache->UnprepareFetch();
        }
    }

    /**
    * @brief        共有キャッシュを占有キャッシュに昇格します。
    *
    * @param[out]   uniqueCache 昇格する共有キャッシュ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - sharedCache が同じ親ストレージを持っている
    *               - sharedCache がキャッシュを所有している
    *
    * @details      戻り値の first は処理に成功したかどうかを示します。
    *               second は昇格できたかどうかを示します。
    *               処理が成功しても昇格できない場合があることに注意してください。
    */
    const std::pair<Result, bool> Upgrade(const SharedCache& sharedCache) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_pBufferedStorage, sharedCache.m_pBufferedStorage);
        NN_SDK_REQUIRES_NOT_NULL(sharedCache.m_pCache);

        std::lock_guard<nn::os::Mutex> lock(m_pBufferedStorage->m_Mutex);
        const auto result = sharedCache.m_pCache->PrepareFetch();
        if( result.first.IsSuccess() && result.second )
        {
            m_pCache = sharedCache.m_pCache;
        }
        return result;
    }

    /**
    * @brief        所有しているキャッシュをフェッチします。
    *
    * @param[in]    offset  フェッチする位置の親ストレージの先頭からのオフセット
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュを所有している
    */
    Result Fetch(int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        return m_pCache->Fetch(offset);
    }

    /**
    * @brief        所有しているキャッシュをバッファからフェッチします。
    *
    * @param[in]    offset      フェッチする位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer      読み込むバッファ
    * @param[in]    bufferSiz   読み込むバッファのサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュを所有している
    *               - offset がブロックサイズでアラインされている
    */
    Result FetchFromBuffer(int64_t offset, const void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pCache);
        NN_RESULT_DO(m_pCache->FetchFromBuffer(offset, buffer, bufferSize));
        NN_RESULT_SUCCESS;
    }

private:
    Cache*           m_pCache;           //!< 所有しているキャッシュ
    BufferedStorage* m_pBufferedStorage; //!< 親ストレージ
};

/**
* @brief        コンストラクタ
*/
BufferedStorage::BufferedStorage() NN_NOEXCEPT
    : m_BaseStorage(),
      m_pBufferManager(nullptr),
      m_BlockSize(0),
      m_BaseStorageSize(0),
      m_pCaches(),
      m_CacheCount(0),
      m_pNextAcquireCache(nullptr),
      m_pNextFetchCache(nullptr),
      m_Mutex(false),
      m_IsBulkReadEnabled(false)
{
}

/**
* @brief        デストラクタ
*/
BufferedStorage::~BufferedStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        キャッシュが有効なストレージとしてマウントします。
*
* @param[in]    baseStorage      キャッシュ対象ストレージ
* @param[in]    pBufferManager   バッファマネージャ
* @param[in]    blockSize        キャッシュするブロックのサイズ
* @param[in]    bufferCount      キャッシュバッファ数
*
* @return 関数の処理結果を返します。
*
* @pre
*               - baseStorage != nullptr
*               - pBufferManager != nullptr
*               - 0 < blockSize
*               - blockSize が 2 のべき乗
*               - 0 < bufferCount
*/
Result BufferedStorage::Initialize(
           fs::SubStorage baseStorage,
           IBufferManager* pBufferManager,
           size_t blockSize,
           int bufferCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBufferManager);
    NN_SDK_REQUIRES_LESS(static_cast<size_t>(0), blockSize);
    NN_SDK_REQUIRES(nn::util::ispow2(blockSize));
    NN_SDK_REQUIRES_LESS(0, bufferCount);

    NN_RESULT_DO(baseStorage.GetSize(&m_BaseStorageSize));

    m_BaseStorage = baseStorage;
    m_pBufferManager = pBufferManager;
    m_BlockSize = blockSize;
    m_CacheCount = bufferCount;

    m_pCaches.reset(new Cache[bufferCount]);
    NN_RESULT_THROW_UNLESS(m_pCaches != nullptr, fs::ResultAllocationMemoryFailedInBufferedStorageA());

    for( auto cacheIndex = 0; cacheIndex < bufferCount; ++cacheIndex )
    {
        m_pCaches[cacheIndex].Initialize(this);
    }
    m_pNextAcquireCache = m_pNextFetchCache = &m_pCaches[0];

    NN_RESULT_SUCCESS;
}

/**
* @brief    アンマウントします。
*/
void BufferedStorage::Finalize() NN_NOEXCEPT
{
    m_BaseStorage = fs::SubStorage();
    m_BaseStorageSize = 0;
    m_pCaches.reset();
    m_CacheCount = 0;
    m_pNextFetchCache = nullptr;
}

/**
* @brief        ストレージの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::Read(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());
    NN_RESULT_DO(ReadCore(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

/**
* @brief        バッファの内容をストレージに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());
    NN_RESULT_DO(WriteCore(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージサイズを取得します。
*
* @param[out]   outValue    ストレージサイズ
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES(IsInitialized());
    *outValue = m_BaseStorageSize;
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージサイズを変更します。
*
* @param[in]    size    変更後のストレージサイズ
*
* @return       失敗の結果を返します。
*/
Result BufferedStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    int64_t previousSize = m_BaseStorageSize;
    if( previousSize < size )
    {
        // 拡大する場合、末尾の半端なキャッシュをフラッシュし無効化する
        if( !nn::util::is_aligned(previousSize, m_BlockSize) )
        {
            SharedCache cache(this);
            const auto invalidateOffset = previousSize;
            const auto invalidateSize = size - previousSize;
            if( cache.AcquireNextOverlappedCache(invalidateOffset, invalidateSize) )
            {
                NN_RESULT_DO(cache.Flush());
                cache.Invalidate();
            }
            NN_SDK_ASSERT(!cache.AcquireNextOverlappedCache(invalidateOffset, invalidateSize));
        }
    }
    else if( size < previousSize )
    {
        // 縮小する場合、新たに末尾となる半端なキャッシュをフラッシュし、それと以降のキャッシュを無効化する
        SharedCache cache(this);
        const auto invalidateOffset = previousSize;
        const auto invalidateSize = previousSize - size;
        const auto isFragment = nn::util::is_aligned(size, m_BlockSize);
        while( cache.AcquireNextOverlappedCache(invalidateOffset, invalidateSize) )
        {
            if( isFragment && cache.Hits(invalidateOffset, 1) )
            {
                NN_RESULT_DO(cache.Flush());
            }
            cache.Invalidate();
        }
    }

    NN_RESULT_DO(m_BaseStorage.SetSize(size));

    int64_t currentSize = 0;
    NN_RESULT_DO(m_BaseStorage.GetSize(&currentSize));

    m_BaseStorageSize = currentSize;

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result BufferedStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    if( operationId == fs::OperationId::Invalidate )
    {
        SharedCache cache(this);
        while( cache.AcquireNextOverlappedCache(offset, size) )
        {
            cache.Invalidate();
        }
    }
    return m_BaseStorage.OperateRange(
        outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::Flush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    SharedCache cache(this);
    while( cache.AcquireNextDirtyCache() )
    {
        NN_RESULT_DO(cache.Flush());
    }
    NN_RESULT_DO(m_BaseStorage.Flush());
    NN_RESULT_SUCCESS;
}

/**
* @brief        キャッシュを無効化します。
*/
void BufferedStorage::InvalidateCaches() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    SharedCache cache(this);
    while( cache.AcquireNextValidCache() )
    {
        cache.Invalidate();
    }
}

/**
* @brief        バッファを新規確保する前の準備を行います。
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::PrepareAllocation() NN_NOEXCEPT
{
    // 空きが少なくなってきたらバッファを新規確保する前にフラッシュする
    const auto sizeThreshold = m_pBufferManager->GetTotalSize() / 8;
    if( m_pBufferManager->GetTotalAllocatableSize() < sizeThreshold )
    {
        NN_RESULT_DO(Flush());
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        ダーティキャッシュが多くならないように適宜フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::ControlDirtiness() NN_NOEXCEPT
{
    const auto sizeThreshold = m_pBufferManager->GetTotalSize() / 4;
    if( m_pBufferManager->GetTotalAllocatableSize() < sizeThreshold )
    {
        SharedCache flushCache(this);
        auto dirtyCount = 0;
        while( flushCache.AcquireNextDirtyCache() )
        {
            ++dirtyCount;
            if( 1 < dirtyCount )
            {
                NN_RESULT_DO(flushCache.Flush());
                flushCache.Invalidate();
            }
        }
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::ReadCore(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaches);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    const auto baseStorageSize = m_BaseStorageSize;

    // 一部でも書き込めるなら書き込む
    if( offset < 0 || baseStorageSize < offset )
    {
        return nn::fs::ResultInvalidOffset();
    }
    size_t leftSize = static_cast<size_t>(std::min<int64_t>(size, baseStorageSize - offset));
    int64_t readOffset = offset;
    int64_t bufferOffset = 0;

    if( m_IsBulkReadEnabled )
    {
        const auto isHeadCacheNeeded = ReadHeadCache(&readOffset, buffer, &leftSize, &bufferOffset);
        if( leftSize == 0 )
        {
            NN_RESULT_SUCCESS;
        }

        const auto isTailCacheNeeded = ReadTailCache(readOffset, buffer, &leftSize, bufferOffset);
        if( leftSize == 0 )
        {
            NN_RESULT_SUCCESS;
        }

        static const auto BulkReadSizeMax = 2 * 1024 * 1024;
        if( leftSize <= BulkReadSizeMax )
        {
            const auto result = BulkRead(
                readOffset,
                reinterpret_cast<char*>(buffer) + bufferOffset,
                leftSize,
                isHeadCacheNeeded,
                isTailCacheNeeded);

            // PooledBuffer が確保できない場合は以降の通常の読み込みを行う
            if( !nn::fs::ResultAllocationPooledBufferNotEnoughSize::Includes(result) )
            {
                NN_RESULT_DO(result);
                NN_RESULT_SUCCESS;
            }
        }
    }

    while( 0 < leftSize )
    {
        // 読み込む位置とサイズを決める
        const auto readBuffer = reinterpret_cast<char*>(buffer) + bufferOffset;
        size_t readSize = 0;
        if( (readOffset & (m_BlockSize - 1)) != 0 )
        {
            // 先頭のブロックサイズでアラインされていない領域
            size_t alignedSize = m_BlockSize - (readOffset & (m_BlockSize - 1));
            readSize = std::min(alignedSize, leftSize);
        }
        else if( leftSize < m_BlockSize )
        {
            // 末尾のブロックサイズでアラインされていない領域
            readSize = leftSize;
        }
        else
        {
            // 中央のブロックサイズでアラインされた領域
            readSize = leftSize & ~(m_BlockSize - 1);
        }

        //読み込む
        if( readSize <= m_BlockSize )
        {
            // ブロックサイズ以下の読み込みはキャッシュする
            SharedCache cache(this);
            if( !cache.AcquireNextOverlappedCache(readOffset, readSize) )
            {
                NN_RESULT_DO(PrepareAllocation());

                for( ; ; )
                {
                    if( cache.AcquireFetchableCache() )
                    {
                        UniqueCache fetchCache(this);
                        const auto upgradeResult = fetchCache.Upgrade(cache);
                        if( upgradeResult.first.IsSuccess() )
                        {
                            if( upgradeResult.second )
                            {
                                NN_RESULT_DO(fetchCache.Fetch(readOffset));
                                break;
                            }
                        }
                        else
                        {
                            return upgradeResult.first;
                        }
                    }
                    else
                    {
                        return nn::fs::ResultOutOfResource();
                    }
                }

                // ダーティキャッシュ数を制限する
                NN_RESULT_DO(ControlDirtiness());
            }
            cache.Read(readOffset, readBuffer, readSize);
        }
        else
        {
            // ブロックサイズを超える読み込みはフラッシュしてから一括で実行する
            {
                SharedCache cache(this);
                while( cache.AcquireNextOverlappedCache(readOffset, readSize) )
                {
                    NN_RESULT_DO(cache.Flush());
                    cache.Invalidate();
                }
            }
            NN_RESULT_DO(m_BaseStorage.Read(readOffset, readBuffer, readSize));
        }
        leftSize -= readSize;
        readOffset += readSize;
        bufferOffset += readSize;
    }
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        読み込み領域の先頭をキャッシュから読み込み、読み込み領域を狭めます。
*
* @param[in,out]    pOffset         読み込み開始位置
* @param[in]        buffer          読み込んだ内容をコピーするバッファ
* @param[in,out]    pSize           読み込むデータサイズ
* @param[in,out]    pBufferOffset   読み込んだ内容をコピーするバッファのオフセット
*
* @return       読み込み領域の先頭をキャッシュする必要があれば true を返します。
*/
bool BufferedStorage::ReadHeadCache(int64_t* pOffset, void* buffer, size_t* pSize, int64_t* pBufferOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOffset);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(pBufferOffset);
    NN_SDK_REQUIRES_NOT_NULL(pSize);

    bool isCacheNeeded = !util::is_aligned(*pOffset, m_BlockSize);

    while( *pSize > 0 )
    {
        size_t readSize = 0;
        if( !util::is_aligned(*pOffset, m_BlockSize) )
        {
            // 先頭のブロックサイズでアラインされていない領域
            const int64_t alignedSize = util::align_up(*pOffset, m_BlockSize) - *pOffset;
            readSize = static_cast<size_t>(std::min(alignedSize, static_cast<int64_t>(*pSize)));
        }
        else if( *pSize < m_BlockSize )
        {
            // 末尾のブロックサイズでアラインされていない領域
            readSize = *pSize;
        }
        else
        {
            // 中央のブロックサイズでアラインされた領域
            readSize = m_BlockSize;
        }

        SharedCache cache(this);
        if( cache.AcquireNextOverlappedCache(*pOffset, readSize) )
        {
            cache.Read(*pOffset, reinterpret_cast<char*>(buffer) + *pBufferOffset, readSize);
            *pOffset += readSize;
            *pBufferOffset += readSize;
            *pSize -= readSize;
            isCacheNeeded = false;
        }
        else
        {
            break;
        }
    }

    return isCacheNeeded;
}

/**
* @brief        読み込み領域の末尾をキャッシュから読み込み、読み込み領域を狭めます。
*
* @param[in]        offset          読み込み開始位置
* @param[in]        buffer          読み込んだ内容をコピーするバッファ
* @param[in]        bufferOffset    読み込んだ内容をコピーするバッファのオフセット
* @param[in,out]    pSize           読み込むデータサイズ
*
* @return       読み込み領域の末尾をキャッシュする必要があれば true を返します。
*/
bool BufferedStorage::ReadTailCache(int64_t offset, void* buffer, size_t* pSize, int64_t bufferOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(pSize);

    bool isCacheNeeded = !util::is_aligned(offset + static_cast<int64_t>(*pSize), m_BlockSize);

    while( *pSize > 0 )
    {
        const int64_t readOffsetEnd = offset + static_cast<int64_t>(*pSize);
        size_t readSize = 0;
        if( !util::is_aligned(readOffsetEnd, m_BlockSize) )
        {
            // 末尾のブロックサイズでアラインされていない領域
            const int64_t alignedSize = readOffsetEnd - util::align_down(readOffsetEnd, m_BlockSize);
            readSize = static_cast<size_t>(std::min(alignedSize, static_cast<int64_t>(*pSize)));
        }
        else if( *pSize < m_BlockSize )
        {
            // 先頭のブロックサイズでアラインされていない領域
            readSize = *pSize;
        }
        else
        {
            // 中央のブロックサイズでアラインされた領域
            readSize = m_BlockSize;
        }

        const int64_t readOffset = readOffsetEnd - static_cast<int64_t>(readSize);
        NN_SDK_ASSERT_GREATER_EQUAL(readOffset, 0);

        SharedCache cache(this);
        if( cache.AcquireNextOverlappedCache(readOffset, readSize) )
        {
            cache.Read(readOffset, reinterpret_cast<char*>(buffer) + bufferOffset + readOffset - offset, readSize);
            *pSize -= readSize;
            isCacheNeeded = false;
        }
        else
        {
            break;
        }
    }

    return isCacheNeeded;
}

/**
* @brief        ストレージの内容をバッファに一括で読み込みます。
*
* @param[in]    offset              読み込み開始位置
* @param[out]   buffer              読み込んだ内容をコピーするバッファ
* @param[in]    size                読み込むデータサイズ
* @param[in]    isHeadCacheNeeded   読み込み領域の先頭をキャッシュする必要があるかどうか
* @param[in]    isTailCacheNeeded   読み込み領域の末尾をキャッシュする必要があるかどうか
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::BulkRead(int64_t offset, void* buffer, size_t size, bool isHeadCacheNeeded, bool isTailCacheNeeded) NN_NOEXCEPT
{
    // ReadCore() のみから呼び出されるため事前条件判定は省略

    const int64_t offsetAligned = util::align_down(offset, m_BlockSize);
    const int64_t offsetAlignedEnd = std::min(
        util::align_up(offset + static_cast<int64_t>(size), m_BlockSize),
        m_BaseStorageSize);
    const int64_t sizeAligned = offsetAlignedEnd - offsetAligned;

    char* readBuffer = nullptr;
    PooledBuffer pooledBuffer;
    if( offset == offsetAligned && size == static_cast<size_t>(sizeAligned) )
    {
        // アライメントが揃っているのでユーザーバッファへ直接読み込み
        readBuffer = reinterpret_cast<char*>(buffer);
    }
    else
    {
        // PooledBuffer で読み込み
        pooledBuffer.AllocateParticularlyLarge(static_cast<size_t>(sizeAligned), 1);
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(pooledBuffer.GetSize()) >= sizeAligned,
            fs::ResultAllocationPooledBufferNotEnoughSize());
        readBuffer = pooledBuffer.GetBuffer();
    }

    // 一括読み込みする範囲をフラッシュ
    {
        SharedCache cache(this);
        while( cache.AcquireNextOverlappedCache(offsetAligned, sizeAligned) )
        {
            NN_RESULT_DO(cache.Flush());
            cache.Invalidate();
        }
    }

    // 元ストレージから一括読み込み
    NN_RESULT_DO(m_BaseStorage.Read(offsetAligned, readBuffer, static_cast<size_t>(sizeAligned)));

    if( readBuffer != reinterpret_cast<char*>(buffer) )
    {
        std::memcpy(buffer, readBuffer + offset - offsetAligned, size);
    }

    bool isCached = false;

    // 必要なら先頭をキャッシュする
    if( isHeadCacheNeeded )
    {
        NN_RESULT_DO(PrepareAllocation());

        SharedCache cache(this);
        for( ; ; )
        {
            NN_RESULT_THROW_UNLESS(cache.AcquireFetchableCache(), nn::fs::ResultOutOfResource());
            UniqueCache fetchCache(this);
            const auto upgradeResult = fetchCache.Upgrade(cache);
            NN_RESULT_DO(upgradeResult.first);
            if( upgradeResult.second )
            {
                NN_RESULT_DO(fetchCache.FetchFromBuffer(
                    offsetAligned,
                    readBuffer,
                    static_cast<size_t>(sizeAligned)));
                break;
            }
        }

        isCached = true;
    }

    // 先頭キャッシュと被っていなければ末尾をキャッシュする
    if( isTailCacheNeeded && (!isHeadCacheNeeded || sizeAligned > static_cast<int64_t>(m_BlockSize)) )
    {
        if( !isCached )
        {
            NN_RESULT_DO(PrepareAllocation());
        }

        SharedCache cache(this);
        for( ; ; )
        {
            NN_RESULT_THROW_UNLESS(cache.AcquireFetchableCache(), nn::fs::ResultOutOfResource());
            UniqueCache fetchCache(this);
            const auto upgradeResult = fetchCache.Upgrade(cache);
            NN_RESULT_DO(upgradeResult.first);
            if( upgradeResult.second )
            {
                const int64_t offsetTailCache = util::align_down(offset + static_cast<int64_t>(size), m_BlockSize);
                const size_t sizeTailCache = static_cast<size_t>(sizeAligned - offsetTailCache + offsetAligned);
                NN_RESULT_DO(fetchCache.FetchFromBuffer(
                    offsetTailCache,
                    readBuffer + offsetTailCache - offsetAligned,
                    sizeTailCache));
                break;
            }
        }
    }

    if( isCached )
    {
        NN_RESULT_DO(ControlDirtiness());
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        バッファの内容をストレージに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result BufferedStorage::WriteCore(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaches);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    const auto baseStorageSize = m_BaseStorageSize;

    // 一部でも書き込めるなら書き込む
    if( offset < 0 || baseStorageSize < offset )
    {
        return nn::fs::ResultInvalidOffset();
    }

    size_t leftSize = static_cast<size_t>(std::min<int64_t>(size, baseStorageSize - offset));
    int64_t writeOffset = offset;
    int64_t bufferOffset = 0;

    while( 0 < leftSize )
    {
        // 書き込む位置とサイズを決める
        const auto writeBuffer = reinterpret_cast<const char*>(buffer) + bufferOffset;
        size_t writeSize = 0;
        if( (writeOffset & (m_BlockSize - 1)) != 0 )
        {
            // 先頭のブロックサイズでアラインされていない領域
            size_t alignedSize = m_BlockSize - (writeOffset & (m_BlockSize - 1));
            writeSize = std::min(alignedSize, leftSize);
        }
        else if( leftSize < m_BlockSize )
        {
            // 末尾のブロックサイズでアラインされていない領域
            writeSize = leftSize;
        }
        else
        {
            // 中央のブロックサイズでアラインされた領域
            writeSize = leftSize & ~(m_BlockSize - 1);
        }

        // 書き込む
        if( writeSize <= m_BlockSize )
        {
            // ブロックサイズ以下の読み込みはキャッシュする
            SharedCache cache(this);
            if( !cache.AcquireNextOverlappedCache(writeOffset, writeSize) )
            {
                NN_RESULT_DO(PrepareAllocation());

                for( ; ; )
                {
                    if( cache.AcquireFetchableCache() )
                    {
                        UniqueCache fetchCache(this);
                        const auto upgradeResult = fetchCache.Upgrade(cache);
                        if( upgradeResult.first.IsSuccess() )
                        {
                            if( upgradeResult.second )
                            {
                                NN_RESULT_DO(fetchCache.Fetch(writeOffset));
                                break;
                            }
                        }
                        else
                        {
                            return upgradeResult.first;
                        }
                    }
                    else
                    {
                        return nn::fs::ResultOutOfResource();
                    }
                }
            }
            cache.Write(writeOffset, writeBuffer, writeSize);

            // キャッシュへの書き込みがあったら、それ以後はバッファ確保でリトライする
            buffers::EnableBlockingBufferManagerAllocation();

            // ダーティキャッシュ数を制限する
            NN_RESULT_DO(ControlDirtiness());
        }
        else
        {
            // ブロックサイズを超える読み込みはフラッシュしてから一括で実行する
            {
                SharedCache cache(this);
                while( cache.AcquireNextOverlappedCache(writeOffset, writeSize) )
                {
                    NN_RESULT_DO(cache.Flush());
                    cache.Invalidate();
                }
            }
            NN_RESULT_DO(m_BaseStorage.Write(writeOffset, writeBuffer, writeSize));

            // 下位ストレージへの書き込みがあったら、それ以後はバッファ確保でリトライする
            buffers::EnableBlockingBufferManagerAllocation();
        }
        leftSize -= writeSize;
        writeOffset += writeSize;
        bufferOffset += writeSize;
    }
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}}}
