﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/os.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/fs_WriteThroughCacheStorage.h>

namespace nn { namespace fssystem {

namespace {

    //! 無効なオフセットです。
    const int64_t InvalidOffset = std::numeric_limits<int64_t>::max();

    //! 一括書き込み時にキャッシュを破棄する閾値のデフォルト値
    const size_t InvalidateThreshold = 64 * 1024;

    const size_t DataCacheLineSize = 64; // TODO: nn::dd などで定義されるべき

    bool IsBufferCapable(const void* buffer) NN_NOEXCEPT
    {
        return nn::fssystem::IsPooledBuffer(buffer)
               && nn::util::is_aligned(
                      reinterpret_cast<const uintptr_t>(buffer),
                      DataCacheLineSize
                  );
    }
}

/**
* @brief    キャッシュ情報の保持、キャッシュアクセス補助を行う構造体です。
*/
struct WriteThroughCacheStorage::Cache
{
    /**
    * @brief        キャッシュを初期化します。
    *
    * @param[in]    pParent         キャッシュ管理クラス
    * @param[in]    pCacheBuffer    キャッシュとして割り当てるバッファ
    *
    * @pre
    *               - pParent != nullptr
    *               - pCacheBuffer != nullptr
    */
    void Initialize(
             WriteThroughCacheStorage* pParent,
             char* pCacheBuffer
         ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pParent);
        NN_SDK_REQUIRES_NOT_NULL(pCacheBuffer);
        NN_SDK_REQUIRES(IsBufferCapable(pCacheBuffer));

        this->pStorage = pParent;
        this->pBuffer = pCacheBuffer;
        this->offsetStorage = InvalidOffset;
        this->pNext = nullptr;
        this->pPrevious = nullptr;
    }

    /**
    * @brief        キャッシュバッファからデータを読み込みます。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    * @param[out]   buffer  読み込んだデータを格納するバッファ
    * @param[in]    size    読み込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが有効化されている
    *               - offset はこのキャッシュと重なる範囲を示している
    *               - size はブロックサイズ以下
    */
    void ReadFromCache(int64_t offset, void* buffer, size_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        NN_SDK_REQUIRES_NOT_NULL(this->pBuffer);
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES(Hits(offset, 1));
        NN_SDK_REQUIRES(size <= this->pStorage->GetBlockSize());

        const int64_t cacheOffset = offset - this->offsetStorage;
        NN_SDK_REQUIRES(static_cast<size_t>(cacheOffset + size) <= this->pStorage->GetBlockSize());

        const char* pSrc = reinterpret_cast<char*>(this->pBuffer) + cacheOffset;
        char* pDst = reinterpret_cast<char*>(buffer);
        std::copy(pSrc, pSrc + size, pDst);
    }

    /**
    * @brief        キャッシュバッファのデータをストレージに書き込みます。
    *
    * @param[in]    size    書き込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが有効化されている
    *               - size はブロックサイズ以下
    */
    Result Write(size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        NN_SDK_REQUIRES_NOT_NULL(this->pBuffer);
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES(size <= this->pStorage->GetBlockSize());

        NN_SDK_REQUIRES(IsBufferCapable(this->pBuffer));

        // 下位ストレージに書き込む
        // AesXtsStorage で PooledBuffer 上のデータが壊れるので
        // 一旦別の PooledBuffer を経由して書き込む
        auto blockSize = this->pStorage->GetBlockSize();
        PooledBuffer pooledBuffer(blockSize, blockSize);
        std::copy(this->pBuffer, this->pBuffer + size, pooledBuffer.GetBuffer());
        NN_RESULT_DO(
            this->pStorage->GetBaseStorage().Write(
                this->offsetStorage,
                pooledBuffer.GetBuffer(),
                size
            )
        );

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        親ストレージからデータを読み込んで新たにキャッシュバッファに格納します。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    * @param[in]    size    読み込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが無効化されている
    *               - offset はブロックサイズでアラインされている
    *               - size はブロックサイズ以下
    */
    Result Fetch(int64_t offset, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        NN_SDK_REQUIRES_NOT_NULL(this->pBuffer);
        NN_SDK_REQUIRES(!IsValid());
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, this->pStorage->GetBlockSize()));
        NN_SDK_REQUIRES(size <= this->pStorage->GetBlockSize());

        NN_SDK_REQUIRES(IsBufferCapable(this->pBuffer));

        // 下位ストレージから読み込む
        NN_RESULT_DO(
            this->pStorage->GetBaseStorage().Read(
                offset,
                this->pBuffer,
                size
            )
        );

        // オフセットを覚えておく
        this->offsetStorage = offset;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        別のバッファの内容をキャッシュバッファに格納します。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer  読み込むバッファ
    * @param[in]    size    読み込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが無効化されている
    *               - offset はブロックサイズでアラインされている
    *               - size はブロックサイズ以下
    */
    Result Fetch(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        NN_SDK_REQUIRES_NOT_NULL(this->pBuffer);
        NN_SDK_REQUIRES(!IsValid());
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, this->pStorage->GetBlockSize()));
        NN_SDK_REQUIRES(size <= this->pStorage->GetBlockSize());

        NN_SDK_REQUIRES(IsBufferCapable(this->pBuffer));

        std::memcpy(this->pBuffer, buffer, size);

        // オフセットを覚えておく
        this->offsetStorage = offset;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        親ストレージから読み込んだデータをキャッシュバッファに格納します。
    *
    * @param[in]    offset  読み込む位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer  読み込んだデータ
    * @param[in]    size    読み込んだサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが無効化されている
    */
    void SaveToCache(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        NN_SDK_REQUIRES_NOT_NULL(this->pBuffer);
        NN_SDK_REQUIRES(!IsValid());
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, this->pStorage->GetBlockSize()));
        NN_SDK_REQUIRES(size <= this->pStorage->GetBlockSize());

        NN_SDK_REQUIRES(IsBufferCapable(this->pBuffer));

        const char* pSrc = reinterpret_cast<const char*>(buffer);
        char* pDst = reinterpret_cast<char*>(this->pBuffer);
        std::copy(pSrc, pSrc + size, pDst);

        // オフセットを覚えておく
        this->offsetStorage = offset;
    }

    /**
    * @brief        キャッシュバッファの内容を更新します。
    *
    * @param[in]    offset  書き込む位置の親ストレージの先頭からのオフセット
    * @param[in]    buffer  書き込むデータを格納したバッファ
    * @param[in]    size    書き込むサイズ
    *
    * @return       処理の結果を返します。
    *
    * @pre
    *               - キャッシュが初期化済み
    *               - キャッシュが有効化されている
    *               - offset はこのキャッシュと重なる範囲を示している
    */
    void UpdateCache(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        NN_SDK_REQUIRES_NOT_NULL(this->pBuffer);
        NN_SDK_REQUIRES(IsValid());
        NN_SDK_REQUIRES(Hits(offset, size));

        NN_SDK_REQUIRES(IsBufferCapable(this->pBuffer));

        // キャッシュバッファの該当範囲のみ書き換える

        const auto updateOffset = std::max(offset, this->offsetStorage);
        const auto updateEnd = std::min(
            offset + static_cast<int64_t>(size),
            this->offsetStorage + static_cast<int64_t>(this->pStorage->GetBlockSize()));

        const char* const pSrc = reinterpret_cast<const char*>(buffer) + (updateOffset - offset);
        char* const pDst = reinterpret_cast<char*>(this->pBuffer)
                           + (updateOffset - this->offsetStorage);
        const auto updateSize = static_cast<size_t>(updateEnd - updateOffset);

        std::copy(pSrc, pSrc + updateSize, pDst);
    }

    /**
    * @brief        キャッシュを無効化します。
    *
    * @pre
    *               - キャッシュが初期化済み
    */
    void Invalidate() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        this->offsetStorage = InvalidOffset;
    }

    /**
    * @brief        キャッシュが有効かどうか判定します。
    *
    * @return       判定の結果を返します。
    * @retval       true    キャッシュは有効です。
    * @retval       false   キャッシュは無効です。
    *
    * @pre
    *               - キャッシュが初期化済み
    *
    * @details      キャッシュが実際に無効になるのはそれが参照されなくなったときであり、
    *               それまでは有効なキャッシュとして扱われます。
    */
    bool IsValid() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        return (this->offsetStorage != InvalidOffset);
    }

    /**
    * @brief        指定の範囲と重なる範囲をキャッシュしているかどうか判定します。
    *
    * @param[in]    offset  判定する範囲の親ストレージの先頭からのオフセット
    * @param[in]    size    判定する範囲のサイズ
    *
    * @return       判定の結果を返します。
    * @retval       true    指定の範囲と重なる範囲をキャッシュしています。
    * @retval       false   指定の範囲と重なる範囲はキャッシュされていません。
    *
    * @pre
    *               - キャッシュが初期化済み
    */
    bool Hits(int64_t offset, int64_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        const auto blockSize = static_cast<int64_t>(this->pStorage->GetBlockSize());
        return (offset < this->offsetStorage + blockSize) && (this->offsetStorage < offset + size);
    }

    /**
    * @brief        指定オフセットから開始する範囲をキャッシュしているかどうか判定します。
    *
    * @param[in]    offset  判定する範囲の親ストレージの先頭オフセット
    *
    * @return       判定の結果を返します。
    * @retval       true    指定オフセットから開始する範囲をキャッシュしています。
    * @retval       false   指定オフセットから開始する範囲はキャッシュされていません。
    * @pre
    *               - キャッシュが初期化済み
    */
    bool Hits(int64_t offset) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(this->pStorage);
        return (this->offsetStorage == offset);
    }

    //! バッファへのポインタを取得します。
    void* GetBuffer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBufferCapable(this->pBuffer));
        return this->pBuffer;
    }

    WriteThroughCacheStorage* pStorage; //!< 親ストレージ
    char* pBuffer;                      //!< バッファ
    int64_t offsetStorage;              //!< ストレージ内オフセット
    Cache* pNext;                       //!< 次のキャッシュエントリ
    Cache* pPrevious;                   //!< 前のキャッシュエントリ
};

/**
* @brief        コンストラクタ
*/
WriteThroughCacheStorage::WriteThroughCacheStorage() NN_NOEXCEPT
    : m_BaseStorage(),
      m_pCacheList(nullptr),
      m_pCacheListHead(nullptr),
      m_pCacheListTail(nullptr),
      m_CacheCount(0)
{
    NN_STATIC_ASSERT(std::is_pod<Cache>::value);
}

/**
* @brief        デストラクタ
*/
WriteThroughCacheStorage::~WriteThroughCacheStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        キャッシュが有効なストレージとして初期化します。
*
* @param[in]    pAllocator       キャッシュに使用するメモリを確保するアロケータ
* @param[in]    baseStorage      キャッシュ対象ストレージ
* @param[in]    blockSize        キャッシュするブロックのサイズ
* @param[in]    bufferCount      キャッシュバッファ数
*
* @return 関数の処理結果を返します。
*
* @pre
*               - pAllocator != nullptr
*               - 0 < blockSize
*               - blockSize が 2 のべき乗
*               - 0 < bufferCount
*/
Result WriteThroughCacheStorage::Initialize(
           nn::MemoryResource* pAllocator,
           fs::SubStorage baseStorage,
           size_t blockSize,
           int bufferCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES_LESS(static_cast<size_t>(0), blockSize);
    NN_SDK_REQUIRES(nn::util::ispow2(blockSize));
    NN_SDK_REQUIRES(blockSize >= DataCacheLineSize);
    NN_SDK_REQUIRES_LESS(0, bufferCount);

    // キャッシュを初期化する
    {
        Cache* pCacheList = reinterpret_cast<Cache*>(pAllocator->allocate(sizeof(Cache) * bufferCount));
        NN_RESULT_THROW_UNLESS(
            pCacheList != nullptr,
            nn::fs::ResultAllocationMemoryFailedInWriteThroughCacheStorageA()
        );

        auto cacheSize = blockSize * bufferCount;
        m_BufferCache.Allocate(cacheSize, cacheSize);

        char* pBuffer = reinterpret_cast<char*>(m_BufferCache.GetBuffer());
        for( int allocatedCount = 0; allocatedCount < bufferCount; ++allocatedCount )
        {
            pCacheList[allocatedCount].Initialize(this, pBuffer + blockSize * allocatedCount);
        }

        m_pCacheList = pCacheList;
    }

    m_pAllocator = pAllocator;
    m_BaseStorage = baseStorage;
    m_BlockSize = blockSize;
    m_CacheCount = bufferCount;

    // キャッシュリストを初期化する
    m_pCacheListHead = &m_pCacheList[0];
    m_pCacheListTail = &m_pCacheList[0];
    for( auto cacheIndex = 1; cacheIndex < m_CacheCount; ++cacheIndex )
    {
        Cache* pCache = &m_pCacheList[cacheIndex];
        pCache->pNext = m_pCacheListHead;
        m_pCacheListHead->pPrevious = pCache;
        m_pCacheListHead = pCache;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief    終了処理を行います。
*/
void WriteThroughCacheStorage::Finalize() NN_NOEXCEPT
{
    if( !IsInitialized() )
    {
        return;
    }

    m_BaseStorage = fs::SubStorage();
    m_BufferCache.Deallocate();
    m_pAllocator->deallocate(m_pCacheList, sizeof(Cache) * m_CacheCount);

    m_pAllocator = nullptr;
    m_pCacheList = nullptr;
    m_pCacheListHead = nullptr;
    m_pCacheListTail = nullptr;
    m_CacheCount = 0;
}

/**
* @brief        ストレージの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @pre
*               - Initialize に成功している
*/
Result WriteThroughCacheStorage::Read(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    NN_SDK_REQUIRES(IsInitialized());

    int64_t baseStorageSize = 0;
    NN_RESULT_DO(m_BaseStorage.GetSize(&baseStorageSize));
    const int64_t maxSize = baseStorageSize - offset;

    // 一部でも読み込めるなら読み込む
    NN_RESULT_THROW_UNLESS(
        (offset >= 0) && (baseStorageSize > offset),
        nn::fs::ResultInvalidOffset()
    );

    auto readBuffer = reinterpret_cast<char*>(buffer);
    int64_t readOffset = offset;
    int64_t alignedOffset = nn::util::align_down(readOffset, m_BlockSize);
    size_t leftSize = static_cast<size_t>(std::min<int64_t>(size, maxSize));

    const int64_t headCacheOffset = alignedOffset;
    const int64_t tailCacheOffset = nn::util::align_down(readOffset + static_cast<int64_t>(leftSize) - 1, m_BlockSize);

    // 読み込む領域が2ブロック以下ならキャッシュする
    if( static_cast<size_t>(tailCacheOffset - headCacheOffset) <= m_BlockSize )
    {

        // 読み込む領域の先頭ブロックがキャッシュされていればそこから読み込む
        {
            Cache* pHeadCache = GetCache(headCacheOffset);
            if( pHeadCache != nullptr )
            {
                const size_t readSize = std::min(
                    leftSize,
                    static_cast<size_t>(headCacheOffset + static_cast<int64_t>(m_BlockSize) - readOffset));

                // キャッシュからデータを読み込む
                pHeadCache->ReadFromCache(readOffset, readBuffer, readSize);

                // アクセスしたキャッシュをリストの先頭に移動
                MoveToHead(pHeadCache);

                leftSize -= readSize;
                readOffset += readSize;
                readBuffer += readSize;
                alignedOffset += static_cast<int64_t>(m_BlockSize);

                if( leftSize == 0 )
                {
                    NN_RESULT_SUCCESS;
                }
            }
        }

        // 先頭と末尾のブロックが異なり、末尾ブロックがキャッシュされていればそこから読み込む
        if( headCacheOffset < tailCacheOffset )
        {
            Cache* pTailCache = GetCache(tailCacheOffset);
            if( pTailCache != nullptr )
            {
                const size_t readSize = static_cast<size_t>(readOffset + static_cast<int64_t>(leftSize) - tailCacheOffset);

                // キャッシュからデータを読み込む
                pTailCache->ReadFromCache(tailCacheOffset, readBuffer + tailCacheOffset - readOffset, readSize);

                // アクセスしたキャッシュをリストの先頭に移動
                MoveToHead(pTailCache);

                leftSize -= readSize;

                if( leftSize == 0 )
                {
                    NN_RESULT_SUCCESS;
                }
            }
        }

        const int64_t maxFetchSize = baseStorageSize - alignedOffset;

        // 2 ブロックにまたがり、PooledBuffer で 2 ブロック確保できるならまとめ読みする
        if( (readOffset + leftSize) > (alignedOffset + m_BlockSize) )
        {
            size_t fetchSize = static_cast<size_t>(std::min<int64_t>(m_BlockSize * 2, maxFetchSize));
            PooledBuffer pooledBuffer(fetchSize, 1);
            if( pooledBuffer.GetSize() >= fetchSize )
            {
                // ストレージから PooledBuffer へまとめて読み込み
                NN_RESULT_DO(m_BaseStorage.Read(alignedOffset, pooledBuffer.GetBuffer(), fetchSize));

                // ユーザーバッファへコピー
                std::memcpy(readBuffer, pooledBuffer.GetBuffer() + readOffset - alignedOffset, leftSize);

                // 1 ブロック目をキャッシュに登録
                Cache* pHeadCache = GetFreeCache();
                NN_RESULT_DO(pHeadCache->Fetch(
                    alignedOffset,
                    pooledBuffer.GetBuffer(),
                    m_BlockSize));
                MoveToHead(pHeadCache);

                // 2 ブロック目をキャッシュに登録
                Cache* pTailCache = GetFreeCache();
                NN_RESULT_DO(pTailCache->Fetch(
                    alignedOffset + m_BlockSize,
                    pooledBuffer.GetBuffer() + m_BlockSize,
                    fetchSize - m_BlockSize));
                MoveToHead(pTailCache);

                NN_RESULT_SUCCESS;
            }
        }

        // キャッシュからフェッチして読み込む
        while( leftSize > 0 )
        {
            // キャッシュの新規確保、フェッチを行う
            const size_t fetchSize =
                static_cast<size_t>(std::min<int64_t>(m_BlockSize, maxFetchSize));
            Cache* pCache = GetFreeCache();
            NN_RESULT_DO(pCache->Fetch(alignedOffset, fetchSize));

            // キャッシュからデータを読み込む
            const size_t readSize = std::min(
                leftSize,
                static_cast<size_t>(alignedOffset + static_cast<int64_t>(fetchSize) - readOffset));
            pCache->ReadFromCache(readOffset, readBuffer, readSize);
            readOffset += readSize;
            readBuffer += readSize;
            leftSize -= readSize;
            alignedOffset += fetchSize;

            // アクセスしたキャッシュをリストの先頭に移動
            MoveToHead(pCache);
        }

        NN_RESULT_SUCCESS;
    }

    // これ以降は、キャッシュしない読み込みの場合の処理

    // キャッシュが存在すればコピーする
    do
    {
        // 先頭ブロックのキャッシュが存在すればコピーする
        Cache* pCache = GetCache(alignedOffset);
        if( pCache != nullptr )
        {
            const size_t readSize = static_cast<size_t>(m_BlockSize - (readOffset - alignedOffset));
            // キャッシュからデータを読み込む
            pCache->ReadFromCache(readOffset, readBuffer, readSize);

            // アクセスしたキャッシュをリストの先頭に移動
            MoveToHead(pCache);

            leftSize -= readSize;
            readBuffer += readSize;
            readOffset += readSize;
        }
        else
        {
            break;
        }

        // 先頭ブロック以降にもキャッシュが存在すればコピーする
        while( leftSize > 0 )
        {
            pCache = GetCache(readOffset);
            if( pCache != nullptr )
            {
                const size_t readSize = std::min(leftSize, m_BlockSize);
                pCache->ReadFromCache(readOffset, readBuffer, readSize);

                // アクセスしたキャッシュをリストの先頭に移動
                MoveToHead(pCache);

                readBuffer += readSize;
                readOffset += readSize;
                leftSize -= readSize;
            }
            else
            {
                break;
            }
        }
    } while( NN_STATIC_CONDITION(false) );

    // まだ読み込む必要のあるデータは下位ストレージから一括読み込みする
    if( leftSize > 0 )
    {
        if( IsBufferCapable(readBuffer) )
        {
            NN_RESULT_DO(m_BaseStorage.Read(readOffset, readBuffer, leftSize));
        }
        else
        {
            PooledBuffer pooledBuffer(leftSize, m_BlockSize);
            while( leftSize > 0 )
            {
                auto currentSize = std::min(leftSize, pooledBuffer.GetSize());
                NN_RESULT_DO(m_BaseStorage.Read(readOffset, pooledBuffer.GetBuffer(), currentSize));
                std::copy(pooledBuffer.GetBuffer(), pooledBuffer.GetBuffer() + currentSize, readBuffer);
                readOffset += currentSize;
                readBuffer += currentSize;
                leftSize -= currentSize;
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        バッファの内容をストレージに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @pre
*               - Initialize に成功している
*/
Result WriteThroughCacheStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    NN_SDK_REQUIRES(IsInitialized());

    int64_t baseStorageSize = 0;
    NN_RESULT_DO(m_BaseStorage.GetSize(&baseStorageSize));
    const int64_t maxSize = baseStorageSize - offset;

    // 一部でも書き込めるなら書き込む
    NN_RESULT_THROW_UNLESS(
        (offset >= 0) && (baseStorageSize > offset),
        nn::fs::ResultInvalidOffset()
    );

    Result result;
    auto writeBuffer = reinterpret_cast<const char*>(buffer);
    int64_t writeOffset = offset;
    size_t leftSize = static_cast<size_t>(std::min<int64_t>(size, maxSize));

    // 先頭アドレスがアラインされていなければ、先頭ブロックを一旦キャッシュしてから書き込む
    if( !nn::util::is_aligned(writeOffset, m_BlockSize) )
    {
        const int64_t alignedOffset = nn::util::align_down(writeOffset, m_BlockSize);
        size_t updateHeadSize = m_BlockSize - static_cast<size_t>(writeOffset - alignedOffset);
        updateHeadSize = std::min(updateHeadSize, leftSize);

        // キャッシュの取得
        Cache* pCache = GetCache(alignedOffset);
        if( pCache == nullptr )
        {
            // キャッシュされていなかった場合は、フェッチを行う
            const int64_t maxFetchSize = baseStorageSize - alignedOffset;
            const size_t fetchSize = static_cast<size_t>(
                std::min<int64_t>(m_BlockSize, maxFetchSize));
            pCache = GetFreeCache();
            NN_RESULT_DO(pCache->Fetch(alignedOffset, fetchSize));
        }

        // 今回アクセスするキャッシュをリストの先頭に移動
        MoveToHead(pCache);

        // 書き込む範囲が大きい場合、先頭以降のキャッシュを破棄する
        if( leftSize > InvalidateThreshold )
        {
            InvalidateCache(
                writeOffset + updateHeadSize,
                static_cast<int64_t>(leftSize - updateHeadSize)
            );
            NN_SDK_ASSERT(pCache->IsValid());
        }

        // 書き込む範囲のキャッシュを更新する
        UpdateCache(writeOffset, writeBuffer, leftSize);

        // キャッシュブロックアライメントでデータを書き込む
        result = pCache->Write(static_cast<size_t>(writeOffset - alignedOffset + updateHeadSize));
        if( result.IsFailure() )
        {
            // 書き込みに失敗した場合はキャッシュを無効化する
            InvalidateCache(offset, static_cast<int64_t>(size));
            NN_RESULT_THROW(result);
        }

        leftSize -= updateHeadSize;
        writeBuffer += updateHeadSize;
        writeOffset += updateHeadSize;
    }
    else
    {
        if( leftSize > InvalidateThreshold )
        {
            // 書き込む範囲が大きい場合、キャッシュを破棄する
            InvalidateCache(writeOffset, static_cast<int64_t>(leftSize));
        }
        else
        {
            // 書き込む範囲のキャッシュを更新する
            UpdateCache(writeOffset, writeBuffer, leftSize);
        }
    }

    // 下位ストレージに書き込む
    if( leftSize > 0 )
    {
        NN_SDK_REQUIRES(nn::util::is_aligned(writeOffset, m_BlockSize));

        if( IsBufferCapable(writeBuffer) )
        {
            NN_RESULT_DO(m_BaseStorage.Write(writeOffset, writeBuffer, leftSize));
        }
        else
        {
            PooledBuffer pooledBuffer(leftSize, m_BlockSize);
            while( leftSize > 0 )
            {
                auto currentSize = std::min(leftSize, pooledBuffer.GetSize());
                std::copy(writeBuffer, writeBuffer + currentSize, pooledBuffer.GetBuffer());
                result = m_BaseStorage.Write(writeOffset, pooledBuffer.GetBuffer(), currentSize);
                if (result.IsFailure())
                {
                    // 書き込みに失敗した場合はキャッシュを無効化する
                    InvalidateCache(offset, static_cast<int64_t>(size));
                    NN_RESULT_THROW(result);
                }
                writeOffset += currentSize;
                writeBuffer += currentSize;
                leftSize -= currentSize;
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        ストレージサイズを取得します。
*
* @param[out]   outValue    ストレージサイズ
*
* @return       関数の処理結果を返します。
*
* @pre
*               - Initialize に成功している
*/
Result WriteThroughCacheStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_RESULT_THROW_UNLESS(outValue != nullptr, nn::fs::ResultNullptrArgument());

    NN_RESULT_DO(m_BaseStorage.GetSize(outValue));
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージサイズを変更します。
*
* @param[in]    size    変更後のストレージサイズ
*
* @return       失敗の結果を返します。
*
* @pre
*               - Initialize に成功している
*/
Result WriteThroughCacheStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultOutOfRange());

    int64_t previousSize = 0;
    NN_RESULT_DO(GetSize(&previousSize));
    if( previousSize < size )
    {
        // 拡大する場合、末尾の半端なキャッシュを無効化する
        if( !nn::util::is_aligned(previousSize, m_BlockSize) )
        {
            InvalidateCache(previousSize, 1);
        }
    }
    else if( size < previousSize )
    {
        // 縮小する場合、新たに末尾となる半端なキャッシュと以降のキャッシュを無効化する
        const auto invalidateOffset = size;
        const auto invalidateSize = previousSize - size;
        InvalidateCache(invalidateOffset, invalidateSize);
    }

    if( size != previousSize )
    {
        NN_RESULT_DO(m_BaseStorage.SetSize(size));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        出力バッファ
* @param[in]   outBufferSize    出力バッファサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         入力バッファ
* @param[in]   inBufferSize     入力バッファサイズ
*
* @return      関数の処理結果を返します。
*
* @pre
*               - Initialize に成功している
*/
Result WriteThroughCacheStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_RESULT_THROW_UNLESS(offset >= 0, nn::fs::ResultInvalidOffset());
    NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultOutOfRange());

    if( operationId == fs::OperationId::Invalidate )
    {
        InvalidateCache(offset, size);
    }
    NN_RESULT_DO(
        m_BaseStorage.OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize
        )
    );
    NN_RESULT_SUCCESS;
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*
* @pre
*               - Initialize に成功している
*/
Result WriteThroughCacheStorage::Flush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // Dirty な Cache は無いため処理は不要
    NN_RESULT_DO(m_BaseStorage.Flush());
    NN_RESULT_SUCCESS;
}

/**
* @brief       キャッシュをリストの先頭に移動します。
*
* @param[in]   pCache       範囲指定処理の種類
*/
void WriteThroughCacheStorage::MoveToHead(Cache* pCache) NN_NOEXCEPT
{
    if( pCache != m_pCacheListHead )
    {
        if( pCache == m_pCacheListTail )
        {
            pCache->pPrevious->pNext = nullptr;
            m_pCacheListTail = pCache->pPrevious;
        }
        else
        {
            pCache->pPrevious->pNext = pCache->pNext;
            pCache->pNext->pPrevious = pCache->pPrevious;
        }

        m_pCacheListHead->pPrevious = pCache;
        pCache->pNext = m_pCacheListHead;
        m_pCacheListHead = pCache;
        pCache->pPrevious = nullptr;
    }
}

/**
* @brief       キャッシュをリストの末尾に移動します。
*
* @param[in]   pCache       範囲指定処理の種類
*/
void WriteThroughCacheStorage::MoveToTail(Cache* pCache) NN_NOEXCEPT
{
    if( pCache != m_pCacheListTail )
    {
        if( pCache == m_pCacheListHead )
        {
            pCache->pNext->pPrevious = nullptr;
            m_pCacheListHead = pCache->pNext;
        }
        else
        {
            pCache->pPrevious->pNext = pCache->pNext;
            pCache->pNext->pPrevious = pCache->pPrevious;
        }

        m_pCacheListTail->pNext = pCache;
        pCache->pPrevious = m_pCacheListTail;
        m_pCacheListTail = pCache;
        pCache->pNext = nullptr;
    }
}

/**
* @brief       指定したオフセットのキャッシュを取得します。
*
* @param[in]   offset       オフセット
*
* @return      キャッシュへのポインタを返します。
*/
WriteThroughCacheStorage::Cache* WriteThroughCacheStorage::GetCache(int64_t offset) NN_NOEXCEPT
{
    // キャッシュを取得する
    auto pCache = m_pCacheListHead;
    while( (pCache != nullptr) && pCache->IsValid() )
    {
        if( pCache->Hits(offset) )
        {
            // キャッシュの取得成功
            return pCache;
        }
        pCache = pCache->pNext;
    }

    // キャッシュは見つからなかった
    return nullptr;
}

/**
* @brief       使用可能なキャッシュを取得します。
*
* @param[in]   offset       オフセット
*
* @return      キャッシュへのポインタを返します。
*/
WriteThroughCacheStorage::Cache* WriteThroughCacheStorage::GetFreeCache() NN_NOEXCEPT
{
    auto pCache = m_pCacheListTail;
    pCache->Invalidate();
    return pCache;
}

/**
* @brief       指定した範囲のキャッシュを無効化します。
*
* @param[in]   offset       無効化開始オフセット
* @param[in]   size         無効化する範囲
*/
void WriteThroughCacheStorage::InvalidateCache(int64_t offset, int64_t size) NN_NOEXCEPT
{
    // 範囲が重なっている有効なキャッシュを無効化
    auto pCache = m_pCacheListHead;
    while( (pCache != nullptr) && (pCache->IsValid()) )
    {
        Cache* pNext = pCache->pNext;
        if( pCache->Hits(offset, size) )
        {
            pCache->Invalidate();
            MoveToTail(pCache);
        }
        pCache = pNext;
    }
}

/**
* @brief       すべてのキャッシュを無効化します。
*/
void WriteThroughCacheStorage::InvalidateCacheAll() NN_NOEXCEPT
{
    auto pCache = m_pCacheListHead;
    while( (pCache != nullptr) && pCache->IsValid() )
    {
        Cache* pNext = pCache->pNext;
        pCache->Invalidate();
        pCache = pNext;
    }
}

/**
* @brief       指定した範囲のキャッシュが保持するデータを更新します。
*
* @param[in]   offset       更新開始オフセット
* @param[in]   buffer       更新に使用するデータ
* @param[in]   size         更新する範囲
*/
void WriteThroughCacheStorage::UpdateCache(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
{
    auto pCache = m_pCacheListHead;
    while( (pCache != nullptr) && pCache->IsValid() )
    {
        Cache* pNext = pCache->pNext;
        if( pCache->Hits(offset, size) )
        {
            pCache->UpdateCache(offset, buffer, size);

            // リストの先頭に移動
            MoveToHead(pCache);
        }
        pCache = pNext;
    }
}

}}
