﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>

#include <nn/fssystem/fs_Utility.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_FormatString.h>

using namespace nn::fs;

namespace nn { namespace fssystem {

    Result ListDirectoryRecursive(nn::fs::fsa::IFileSystem* pFileSystem) NN_NOEXCEPT
    {
        return IterateDirectoryRecursive(pFileSystem,
            [](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(path);
            NN_UNUSED(entry);
            NN_SDK_LOG("DIR  <%s>\n", path);
            NN_RESULT_SUCCESS;
        },
            [](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(path);
            NN_UNUSED(entry);
            NN_RESULT_SUCCESS;
        },
            [](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(path);
            NN_UNUSED(entry);
            NN_SDK_LOG("FILE <%-40s> %10lld\n", path, entry.fileSize);
            NN_RESULT_SUCCESS;
        }
        );
    }

    Result CopyFile(fs::fsa::IFileSystem* pDstFileSystem, fs::fsa::IFileSystem* pSrcFileSystem, const char* destinationParentPath, const char* SourcePath, const DirectoryEntry* pEntry, char* workBuffer, size_t workBufferSize)
    {
        std::unique_ptr<fs::fsa::IFile> pSrcFile;
        NN_RESULT_DO(pSrcFileSystem->OpenFile(&pSrcFile, SourcePath, OpenMode_Read));

        std::unique_ptr<fs::fsa::IFile> pDstFile;
        char path[nn::fs::EntryNameLengthMax + 1];
        const auto untruncatedSize = nn::util::SNPrintf(path, sizeof(path), "%s%s", destinationParentPath, pEntry->name);
        NN_SDK_ASSERT_LESS(static_cast<size_t>(untruncatedSize), sizeof(path));
        NN_RESULT_DO(pDstFileSystem->CreateFile(path, pEntry->fileSize));
        NN_RESULT_DO(pDstFileSystem->OpenFile(&pDstFile, path, OpenMode_Write));
        NN_UNUSED(untruncatedSize);

        int64_t restSize = pEntry->fileSize;
        int64_t offset = 0;
        while (restSize > 0)
        {
            size_t readSize;
            NN_RESULT_DO(pSrcFile->Read(&readSize, offset, workBuffer, workBufferSize, ReadOption()));
            NN_RESULT_DO(pDstFile->Write(offset, workBuffer, readSize, WriteOption()));

            restSize -= readSize;
            offset += readSize;
        }

        NN_RESULT_SUCCESS;
    }

    Result CopyDirectoryRecursive(fs::fsa::IFileSystem* pDstFileSystem, fs::fsa::IFileSystem* pSrcFileSystem, const char* destinationPath, const char* sourcePath, char* workBuffer, size_t workBufferSize)
    {
        char destinationPathBuffer[nn::fs::EntryNameLengthMax + 1];
        const auto untruncatedSize = nn::util::SNPrintf(destinationPathBuffer, sizeof(destinationPathBuffer), "%s", destinationPath);
        NN_SDK_ASSERT_LESS(static_cast<size_t>(untruncatedSize), sizeof(destinationPathBuffer));
        NN_UNUSED(untruncatedSize);

        return IterateDirectoryRecursive(pSrcFileSystem, sourcePath,
            [&](const char* path, const DirectoryEntry& entry) -> Result
            {
                NN_UNUSED(path);
                strncat(destinationPathBuffer, entry.name, sizeof(destinationPathBuffer) - strnlen(destinationPathBuffer, sizeof(destinationPathBuffer) - 1) - 1);
                strncat(destinationPathBuffer, "/", sizeof(destinationPathBuffer) - strnlen(destinationPathBuffer, sizeof(destinationPathBuffer) - 1) - 1);
                return pDstFileSystem->CreateDirectory(destinationPathBuffer);
            },
            [&](const char* path, const DirectoryEntry& entry) -> Result
            {
                NN_UNUSED(path);
                NN_UNUSED(entry);
                const auto length = strnlen(destinationPathBuffer, sizeof(destinationPathBuffer));
                if (length < 2)
                {
                    NN_SDK_ASSERT(false);
                    return nn::fs::ResultInvalidPathFormat();
                }
                char* p = &destinationPathBuffer[length - 2];
                while (*p != '/' && p > destinationPathBuffer)
                {
                    p--;
                }
                *(p + 1) = '\0';
                NN_RESULT_SUCCESS;
            },
            [&](const char* path, const DirectoryEntry& entry) -> Result
            {
                return CopyFile(pDstFileSystem, pSrcFileSystem, destinationPathBuffer, path, &entry, workBuffer, workBufferSize);
            }
        );
    }


    Result VerifyDirectoryRecursive(nn::fs::fsa::IFileSystem* pFileSystem, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        return IterateDirectoryRecursive(pFileSystem,
            [](const char* path, const DirectoryEntry& entry) NN_NOEXCEPT
            {
                NN_UNUSED(path);
                NN_UNUSED(entry);
                NN_RESULT_SUCCESS;
            },
            [](const char* path, const DirectoryEntry& entry) NN_NOEXCEPT
            {
                NN_UNUSED(path);
                NN_UNUSED(entry);
                NN_RESULT_SUCCESS;
            },
            [&](const char* path, const DirectoryEntry& entry) NN_NOEXCEPT -> Result {
                NN_UNUSED(entry);

                std::unique_ptr<nn::fs::fsa::IFile> file;
                NN_RESULT_DO(pFileSystem->OpenFile(&file, path, OpenMode_Read));

                int64_t offset = 0;
                while(NN_STATIC_CONDITION(true))
                {
                    size_t readSize;
                    NN_RESULT_DO(file->Read(&readSize, offset, buffer, bufferSize, ReadOption()));

                    if( readSize < bufferSize )
                    {
                        break;
                    }

                    offset += readSize;
                }

                NN_RESULT_SUCCESS;
            }
            );
    }

    void AddCounter(void* pCounter, size_t counterSize, uint64_t additionalValue) NN_NOEXCEPT
    {
        uint64_t restAdditionalValue = additionalValue;
        auto pCounterU8 = static_cast<uint8_t*>(pCounter);
        uint8_t carry = 0;

        for (int i = 0; i < static_cast<int>(counterSize); ++i)
        {
            auto sum = pCounterU8[counterSize - i - 1] + (restAdditionalValue & 0xFF) + carry;
            carry = static_cast<uint8_t>(sum >> 8);
            auto sumU8 = static_cast<uint8_t>(sum & 0xFF);

            pCounterU8[counterSize - i - 1] = sumU8;

            restAdditionalValue >>= 8;
            if (carry == 0 && restAdditionalValue == 0)
            {
                return;
            }
        }
    }

    bool IsWindowsDrive(const char* path) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(path);
        return (('a' <= path[0] && path[0] <= 'z') || ('A' <= path[0] && path[0] <= 'Z'))
            && path[0] != '\0' && path[1] == ':';
    }

}}
