﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstring>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/os/os_Mutex.h>

#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_TmFileSystem.h>
#include <nn/fssystem/fs_Utility.h>
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
#include <nn/tma/tma_FileIO.h>
#include <nn/fs/fs_QueryRange.h>
#endif

using namespace nn::fs;
using namespace nn::fs::detail;

namespace nn { namespace fssystem {

#if !defined( NN_BUILD_CONFIG_OS_WIN32 )

    namespace {
        // TMA から返ってくるシステム時刻(先発グレゴリオ暦) から UNIX 時間への変換オフセット(秒)
        // 先発グレゴリオ暦 : 1601/1/1 0:00 開始
        // UNIX時間         : 1970/1/1 0:00 開始
        static const uint64_t OffsetToConvertToPosixTime = 11644473600;

        bool ExistsEntry(const char* path) NN_NOEXCEPT
        {
            {
                auto exists = false;
                const auto result = tma::file_io::FileExists(&exists, path);
                if( result.IsSuccess() && exists )
                {
                    return true;
                }
            }
            {
                auto exists = false;
                const auto result = tma::file_io::DirectoryExists(&exists, path);
                if( result.IsSuccess() && exists )
                {
                    return true;
                }
            }
            return false;
        }

    }

    class TmFileSystemFile: public fsa::IFile, public fs::detail::Newable
    {
    public:

        NN_IMPLICIT TmFileSystemFile( tma::file_io::FileHandle Handle, nn::fs::OpenMode mode ) NN_NOEXCEPT
            : m_Handle( Handle ),
              m_Mode( mode )
        {
        }

        virtual ~TmFileSystemFile() NN_NOEXCEPT
        {
            tma::file_io::CloseFile( m_Handle );
        }

        virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const ReadOption& option ) NN_NOEXCEPT NN_OVERRIDE
        {
            size_t readSize = 0;
            NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));
            return tma::file_io::ReadFile( outValue, m_Handle, offset, buffer, readSize, option );
        }

        virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            if( offset < 0 )
            {
                return nn::fs::ResultOutOfRange();
            }

            NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Write) != 0, ResultInvalidOperationForOpenMode() );

            if ((m_Mode & nn::fs::OpenMode_AllowAppend) == 0)
            {
                int64_t currentSize;
                NN_RESULT_DO(GetSize(&currentSize));
                NN_FSP_REQUIRES((offset + static_cast<int64_t>(size)) <= currentSize && (offset + static_cast<int64_t>(size)) >= offset,
                    ResultFileExtensionWithoutOpenModeAllowAppend(), "OpenMode_AllowAppend is required for implicit extension of file size by WriteFile().");
            }
            return tma::file_io::WriteFile( m_Handle, offset, buffer, size, option );
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_DO(tma::file_io::FlushFile( m_Handle ));
            NN_RESULT_SUCCESS;
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(DrySetSize(size, m_Mode));
            return tma::file_io::SetFileSize( m_Handle, size );
        }
        virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return tma::file_io::GetFileSize( outValue, m_Handle );
        }

        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t,
            int64_t,
            const void*,
            size_t) NN_NOEXCEPT NN_OVERRIDE
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
                NN_RESULT_SUCCESS;
            case OperationId::QueryRange:
                NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
                NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

                reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
                NN_RESULT_SUCCESS;
            default:
                return nn::fs::ResultUnsupportedOperation();
            }
        }

    private:
        tma::file_io::FileHandle m_Handle;
        nn::fs::OpenMode m_Mode;
    };

    //===========================================================================================================

    class TmFileSystemDirectory : public fsa::IDirectory, public fs::detail::Newable
    {
    public:

        NN_IMPLICIT TmFileSystemDirectory( tma::file_io::DirectoryHandle Handle ) NN_NOEXCEPT
        : m_Handle( Handle )
        {
        }

        virtual ~TmFileSystemDirectory() NN_NOEXCEPT
        {
            tma::file_io::CloseDirectory( m_Handle );
        }

        virtual Result DoRead(int64_t *outValue, DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(outValue);
            if( 0 < entryBufferCount )
            {
                return tma::file_io::ReadDirectory(outValue, entryBuffer, m_Handle, entryBufferCount);
            }
            else
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }
        }

        virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
        {
            return tma::file_io::GetEntryCount( outValue, m_Handle );
        }

    private:
        tma::file_io::DirectoryHandle m_Handle;
    };

#endif

    //===========================================================================================================
    namespace{
        bool g_Initialized = false;
        nn::os::Mutex g_InitializedMutex(false);
    }

    TmFileSystem::TmFileSystem() NN_NOEXCEPT
    {
    }

    TmFileSystem::~TmFileSystem() NN_NOEXCEPT
    {
    }

    Result GenerateTmPath(const char** pPath, const char* path, size_t pathLengthMax) NN_NOEXCEPT
    {
        *pPath = path;
        if( path[0] == '/' )
        {
            ++*pPath;
        }

        static const auto NameLengthMax = 255;
        const auto driveLetterLength = IsWindowsDrive(*pPath) ? 2 : 0;
        return VerifyPath(
            *pPath + driveLetterLength,
            static_cast<int>(pathLengthMax - driveLetterLength),
            NameLengthMax);
    }

    Result GenerateTmFilePath(const char** pPath, const char* path) NN_NOEXCEPT
    {
        static const auto LengthMax = 260 - 1; // MAX_PATH
        return GenerateTmPath(pPath, path, LengthMax);
    }

    Result GenerateTmDirectoryPath(const char** pPath, const char* path) NN_NOEXCEPT
    {
        static const auto LengthMax = 248 - 1; // ::CreateDirectory() の最大パス長
        return GenerateTmPath(pPath, path, LengthMax);
    }

    Result TmFileSystem::Initialize() NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        std::lock_guard<os::Mutex> scopedLock(g_InitializedMutex);
        if (!g_Initialized)
        {
            nn::tma::file_io::Initialize();
            g_Initialized = true;
        }
#endif
        NN_RESULT_SUCCESS;
    }

    Result TmFileSystem::DoCreateFile( const char *path, int64_t size, int option ) NN_NOEXCEPT
    {
        NN_UNUSED(option);
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath( &pPath, path ));
        return tma::file_io::CreateFile( pPath, size );
#else
        NN_UNUSED(path);
        NN_UNUSED(size);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath( &pPath, path ));
        return tma::file_io::DeleteFile( pPath );
#else
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmDirectoryPath( &pPath, path ));
        return tma::file_io::CreateDirectory( pPath );
#else
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath( &pPath, path )); // ファイルパスの最大長まで許容される
        return tma::file_io::DeleteDirectory( pPath, false );
#else
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath( &pPath, path )); // ファイルパスの最大長まで許容される
        return tma::file_io::DeleteDirectory( pPath, true );
#else
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const auto parentLength = strnlen(path, fs::EntryNameLengthMax + 1);
        const auto separatorLength = path[parentLength - 1] == '/' ? 0 : 1;
        const auto childLengthMax = fs::EntryNameLengthMax - parentLength - separatorLength;
        NN_RESULT_THROW_UNLESS(
            parentLength + separatorLength <= fs::EntryNameLengthMax,
            nn::fs::ResultTooLongPath());

        const auto childEntry = fs::detail::MakeUnique<DirectoryEntry>();
        NN_RESULT_THROW_UNLESS(
            childEntry != nullptr,
            nn::fs::ResultAllocationMemoryFailedMakeUnique());

        const auto childPath = fs::detail::MakeUnique<char[]>(fs::EntryNameLengthMax + 1);
        NN_RESULT_THROW_UNLESS(
            childPath != nullptr,
            nn::fs::ResultAllocationMemoryFailedMakeUnique());

        std::strncpy(childPath.get(), path, fs::EntryNameLengthMax);
        if( 0 < separatorLength )
        {
            childPath[parentLength] = '/';
        }
        childPath[parentLength + separatorLength] = '\0';

        for( ; ; )
        {
            std::unique_ptr<fsa::IDirectory> directory;
            NN_RESULT_DO(OpenDirectory(&directory, path, OpenDirectoryMode_All));

            int64_t readCount = 0;
            NN_RESULT_DO(directory->Read(&readCount, childEntry.get(), 1));

            if( readCount == 0 )
            {
                break;
            }

            NN_RESULT_THROW_UNLESS(
                strnlen(childEntry->name, childLengthMax + 1) <= childLengthMax,
                nn::fs::ResultTooLongPath());
            std::strncpy(
                childPath.get() + parentLength + separatorLength,
                childEntry->name,
                childLengthMax);
            childPath[fs::EntryNameLengthMax] = '\0';

            if( childEntry->directoryEntryType == fs::DirectoryEntryType_File )
            {
                NN_RESULT_DO(DeleteFile(childPath.get()));
            }
            else
            {
                NN_SDK_ASSERT_EQUAL(
                    childEntry->directoryEntryType,
                    fs::DirectoryEntryType_Directory);
                NN_RESULT_DO(DeleteDirectoryRecursively(childPath.get()));
            }
        }

        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pCurrentPath;
        NN_RESULT_DO(GenerateTmFilePath( &pCurrentPath, currentPath ));
        const char* pNewPath;
        NN_RESULT_DO(GenerateTmFilePath( &pNewPath, newPath ));
        NN_RESULT_TRY(tma::file_io::RenameFile( pCurrentPath, pNewPath ))
            NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
            {
                if( strncmp(pCurrentPath, pNewPath, fs::EntryNameLengthMax) == 0 )
                {
                    NN_RESULT_SUCCESS;
                }
                else if( ExistsEntry(pNewPath) )
                {
                    return nn::fs::ResultPathAlreadyExists();
                }
                else
                {
                    NN_RESULT_RETHROW;
                }
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pCurrentPath;
        NN_RESULT_DO(GenerateTmDirectoryPath( &pCurrentPath, currentPath ));
        const char* pNewPath;
        NN_RESULT_DO(GenerateTmDirectoryPath( &pNewPath, newPath ));
        NN_RESULT_TRY(tma::file_io::RenameDirectory( pCurrentPath, pNewPath ))
            NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
            {
                if( strncmp(pCurrentPath, pNewPath, fs::EntryNameLengthMax) == 0 )
                {
                    NN_RESULT_SUCCESS;
                }
                else if( ExistsEntry(pNewPath) )
                {
                    return nn::fs::ResultPathAlreadyExists();
                }
                else
                {
                    NN_RESULT_RETHROW;
                }
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoGetEntryType( DirectoryEntryType* outValue, const char *path ) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath( &pPath, path ));
        return tma::file_io::GetIOType( outValue, pPath );
#else
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoOpenFile( std::unique_ptr<fsa::IFile>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        tma::file_io::FileHandle fileHandle;
        NN_FSP_REQUIRES( (mode & (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)) != 0, ResultInvalidArgument(), "OpenMode_Read or OpenMode_Write is required.");

        tma::file_io::OpenMode tmaOpenMode;
        if (mode & nn::fs::OpenMode_Read)
        {
            tmaOpenMode = tma::file_io::OpenMode_Read;
            if (mode & nn::fs::OpenMode_Write)
            {
                tmaOpenMode = static_cast<tma::file_io::OpenMode>(tma::file_io::OpenMode_Read | tma::file_io::OpenMode_Write);
            }
        }
        else
        {
            tmaOpenMode = tma::file_io::OpenMode_Write;
        }

        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath( &pPath, path ));
        Result Res = tma::file_io::OpenFile( &fileHandle, pPath, tmaOpenMode );
        if( Res.IsSuccess() )
        {
            std::unique_ptr<TmFileSystemFile> file(new TmFileSystemFile(fileHandle, mode));
            if( file == nullptr )
            {
                tma::file_io::CloseFile(fileHandle);
                return ResultAllocationMemoryFailedInTmFileSystemA();
            }
            outValue->reset(file.release());
        }

        return Res;
#else
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        NN_UNUSED(mode);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoOpenDirectory(std::unique_ptr<fsa::IDirectory>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        const char* pPath;
        NN_RESULT_DO(GenerateTmDirectoryPath( &pPath, path ));

        tma::file_io::DirectoryHandle dirHandle;
        nn::Result Res = tma::file_io::OpenDirectory(&dirHandle, pPath, mode );
        if( Res.IsSuccess() )
        {
            std::unique_ptr<TmFileSystemDirectory> directory(new TmFileSystemDirectory(dirHandle) );
            if( directory == nullptr )
            {
                tma::file_io::CloseDirectory(dirHandle);
                return ResultAllocationMemoryFailedInTmFileSystemB();
            }
            outValue->reset(directory.release());
        }

        return Res;
#else
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        NN_UNUSED(mode);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
        uint64_t createTime = 0;
        uint64_t accessTime = 0;
        uint64_t modifyTime = 0;

        const char* pPath;
        NN_RESULT_DO(GenerateTmFilePath(&pPath, path));
        NN_RESULT_DO(tma::file_io::GetFileTimeStamp(&createTime, &accessTime, &modifyTime, pPath));

        // 得られた値は 100ns 単位なので s に変換した後、 UNIX 時間に変換
        outTimeStamp->create = createTime / 10000000 - OffsetToConvertToPosixTime;
        outTimeStamp->access = accessTime / 10000000 - OffsetToConvertToPosixTime;
        outTimeStamp->modify = modifyTime / 10000000 - OffsetToConvertToPosixTime;
        outTimeStamp->isLocalTime = false; // UTC なので false
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(outTimeStamp);
        NN_UNUSED(path);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result TmFileSystem::DoCommit() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
}}
